/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors.inputs;

import java.util.List;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IPersistableElement;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.handlers.IConfigurationUpdater;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.AssemblyConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.AssemblyTypeConversationUtils;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.HwSchema;

/**
 * Input object for xml editing of an assembly.
 * @author sharring
 */
public class AssemblyXmlEditorInput implements IXmlEditorInput 
{
	private static final String OPTICAL_TELESCOPE_ASSEMBLYTYPE_NAME = "OpticalTelescope";
	public Assembly assembly;
	public IConfigurationUpdater saver;
	
	public AssemblyXmlEditorInput(Assembly assembly, IConfigurationUpdater saver)
	{
		this.assembly = assembly;
		this.saver = saver;
	}
	
	@Override
	public boolean editedObjectHasXmlSchema() {
		return true;
	}

	@Override
	public String getXmlForEditedObject() {
		return assembly.getData();
	}

	@Override
	public String getXmlSchemaForEditedObject() {

		String schema = null;
		try {
			AssemblyConversationUtils.getInstance().hydrateAssemblyType(assembly.getAssemblyType());
			List<HwSchema> schemas = AssemblyTypeConversationUtils.getInstance().findHwSchemasForAssemblyType(assembly.getAssemblyType());
			if( schemas != null && schemas.size() > 0 )
				schema = schemas.get(0).getSchema();
		} catch(Exception e) {
			e.printStackTrace();
		}

		return schema;
	}

	@Override
	public void setXmlForEditedObject(String xmldata) {
		assembly.setData(xmldata);
	}

	@Override
	public boolean exists() {
		return false;
	}

	@Override
	public ImageDescriptor getImageDescriptor() {
		return RcpUtils.getImageDescriptor("icons/assembly.png");
	}

	@Override
	public String getName() {
		return LabelHelper.getAssemblyLabel(assembly);
	}

	@Override
	public IPersistableElement getPersistable() {
		return null;
	}

	@Override
	public String getToolTipText() {
		return LabelHelper.getAssemblyLabel(assembly) + " in " + assembly.getConfiguration().getName();
	}

	@SuppressWarnings("unchecked")
	@Override
	public Object getAdapter(Class adapter) {
		return null;
	}
	
	public Assembly getAssembly() {
		return this.assembly;
	}
	
	public boolean equals(Object o) 
	{
		if(super.equals(o))
			return true;
		if( !(o instanceof AssemblyXmlEditorInput) )
			return false;

		AssemblyXmlEditorInput assemblyEditorInput = (AssemblyXmlEditorInput)o;
		Assembly ass = assemblyEditorInput.getAssembly();

		if( ass.getId() != null && assembly.getId() != null ) {
			if( assembly.getId().equals(ass.getId()) ) {
				return true;
			}
			return false;
		}

		return false;
	}

	@Override
	public void updateConfiguration() {
		saver.updateConfiguration();
	}

	@Override
	public boolean isReadOnly() {
		// Non-almamgr (not god) user cannot edit assemblies, with the exception of the optical telescope; 
		// The optical telescope is a special case, per requirements from JAO / OSF team.
		boolean retVal = !GuiUtils.isGodUser()&& !assembly.getAssemblyType().getName().equals(OPTICAL_TELESCOPE_ASSEMBLYTYPE_NAME);
		return retVal;
	}

	@Override
	public int hashCode() {
		int retVal = 17;
		retVal += this.getAssembly() == null ? 0 : this.getAssembly().hashCode();
		return retVal;
	}

	/* (non-Javadoc)
	 * @see alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput#getDomainObject()
	 */
	@Override
	public Object getTopLevelDomainObjectForLocking() {
		return this.getAssembly();
	}
}
