/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.obops.tmcdbgui.editors.inputs.HistoricalXpDelaysEditorInput;
import alma.obops.tmcdbgui.editors.inputs.XpDelaysHistoryEditorInput;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.providers.HistoryRecordViewerSorter;
import alma.obops.tmcdbgui.views.providers.XpDelaysHistoryTableContentsProvider;
import alma.obops.tmcdbgui.views.providers.XpDelaysHistoryTableLabelProvider;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.XPDelay;
import alma.tmcdb.history.HistoryRecord;

/**
 * "Editor" (read only) for xp delays history.
 * @author sharring
 *
 */
public class XpDelaysHistoryEditor extends EditorPart
{
	public static final String ID = "xpdelays-history.editor";
	private TableViewer historyViewer;
	private HwConfiguration owningConfig;

	@Override
	public void createPartControl( Composite parent ) 
	{
		historyViewer = new TableViewer(parent, SWT.BORDER | SWT.FULL_SELECTION | SWT.MULTI);

		// Setup the columns
		String [] titles = { "Version", "Description", "Modifier", "Date" };
		for(int i = 0; i != titles.length; i++) {
			TableViewerColumn col = new TableViewerColumn(historyViewer, SWT.NONE);
			col.getColumn().setText(titles[i]);
			col.getColumn().setMoveable(false);
			col.getColumn().setResizable(true);
			col.getColumn().setWidth(250);
		}
		
		historyViewer.setSorter(new HistoryRecordViewerSorter());
		historyViewer.setContentProvider( new XpDelaysHistoryTableContentsProvider() );
		historyViewer.setLabelProvider( new XpDelaysHistoryTableLabelProvider() );
		historyViewer.setInput(this.owningConfig);
		
		Table table = historyViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		
		MenuManager popupMenu = new MenuManager();
		final CompareXpDelaysAction compareXpDelaysAction = new CompareXpDelaysAction();
		popupMenu.add(compareXpDelaysAction);
		Menu menu = popupMenu.createContextMenu(table);
		table.setMenu(menu);

		historyViewer.addSelectionChangedListener(new ISelectionChangedListener() 
		{
			@Override
			public void selectionChanged(SelectionChangedEvent evt) 
			{
				ISelection selection = evt.getSelection();
				if(selection instanceof IStructuredSelection) {
					IStructuredSelection structuredSelection = (IStructuredSelection) selection;
					Object[] recordsSelected = structuredSelection.toArray();
					if(recordsSelected.length == 2) {
						compareXpDelaysAction.setEnabled(true);
						compareXpDelaysAction.setPreviousRecord((HistoryRecord)recordsSelected[0]);
						compareXpDelaysAction.setReferenceRecord((HistoryRecord)recordsSelected[1]);
					}
					else {
						compareXpDelaysAction.setEnabled(false);
					}
				}
			}
		});

		IDoubleClickListener listener = new GetHistoricalXpDelaysDoubleClickListener();
		historyViewer.addDoubleClickListener(listener);
	}

	@Override
	public void setFocus() {
		historyViewer.getControl().setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor arg0) {
		// NOOP
	}

	@Override
	public void doSaveAs() {
		// NOOP
	}

	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException 
	{
		XpDelaysHistoryEditorInput editorInput = (XpDelaysHistoryEditorInput)input;
		setInput(input);
		if(null != historyViewer) {
			historyViewer.setInput(editorInput.getHwConfiguration());
		}
		setSite(site);
		setPartName(editorInput.getName());
	}

	@Override 
	public void setInput(IEditorInput input)
	{
		super.setInput(input);
		this.owningConfig = ((XpDelaysHistoryEditorInput) input).getHwConfiguration();
	}
	
	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}
	
	private class GetHistoricalXpDelaysDoubleClickListener implements IDoubleClickListener
	{
		@Override
		public void doubleClick(DoubleClickEvent evt) 
		{
			ISelection selection = evt.getSelection();
			if(selection instanceof IStructuredSelection) {
				IStructuredSelection structuredSelection = (IStructuredSelection) selection;
				if(structuredSelection.getFirstElement() != null) {
					HistoryRecord clickedRecord = (HistoryRecord) structuredSelection.getFirstElement();
					Set<XPDelay> historicalXpDelays = null;
					try {
						historicalXpDelays = HwConfigurationConversationUtils.getInstance().getHistoricalXpDelays(owningConfig, clickedRecord);
					} catch (Exception e) {
						e.printStackTrace();
						throw new RuntimeException("Unable to get historical xp delays" + e);
					}
					
					HistoricalXpDelaysEditorInput editorInput =  
						new HistoricalXpDelaysEditorInput(historicalXpDelays, historicalXpDelays, clickedRecord, owningConfig);
					
					IWorkbenchWindow win = getSite().getWorkbenchWindow();
					try {
						win.getActivePage().openEditor(editorInput, HistoricalXpDelaysEditor.ID);
					} 
					catch (PartInitException e1) {
						e1.printStackTrace();
						throw new RuntimeException("Could not open historical xp delays editor", e1);
					}
				}
			}
		}
	}
	
	private class CompareXpDelaysAction extends Action
	{
		private HistoryRecord referenceRecord;
		private HistoryRecord previousRecord;
		
		public CompareXpDelaysAction() 
		{
			super("Show differences");
		}
		
		public void setReferenceRecord(HistoryRecord rec)
		{
			this.referenceRecord = rec;
		}
		
		public void setPreviousRecord(HistoryRecord rec)
		{
			this.previousRecord = rec;
		}

		public void run() 
		{
			Set<XPDelay> historicalXpDelays = null;
			Set<XPDelay> historicalXpDelaysPreviousVersion = null;
			try {
				historicalXpDelays = HwConfigurationConversationUtils.getInstance().getHistoricalXpDelays(owningConfig, referenceRecord);
				historicalXpDelaysPreviousVersion = HwConfigurationConversationUtils.getInstance().getHistoricalXpDelays(owningConfig, previousRecord);
			} catch (Exception e) {
				throw new RuntimeException("Unable to get historical xp delays" + e);
			}
			
			HistoryRecord junkRecord = new HistoryRecord();
			junkRecord.setVersion(0L - (referenceRecord.getVersion() - previousRecord.getVersion()));
			HistoricalXpDelaysEditorInput editorInput =  
				new HistoricalXpDelaysEditorInput(historicalXpDelays, historicalXpDelaysPreviousVersion, 
						junkRecord, "Diff xp delays v." + referenceRecord.getVersion() 
						+ " to v." + previousRecord.getVersion() + " for config " + owningConfig.getName(), owningConfig);
			
			IWorkbenchWindow win = getSite().getWorkbenchWindow();
			try {
				win.getActivePage().openEditor(editorInput, HistoricalXpDelaysEditor.ID);
			} 
			catch (PartInitException e1) {
				throw new RuntimeException("Could not open historical xp delays editor", e1);
			}
		}
	}
}
