/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;

import alma.BasebandNameMod.BasebandName;
import alma.NetSidebandMod.NetSideband;
import alma.ReceiverBandMod.ReceiverBand;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.editors.inputs.XpDelaysEditorInput;
import alma.obops.tmcdbgui.editors.inputs.XpDelaysHistoryEditorInput;
import alma.obops.tmcdbgui.editors.sorters.XpDelaysViewerSorter;
import alma.obops.tmcdbgui.utils.DelayEditingUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.providers.XpDelayModelRow;
import alma.obops.tmcdbgui.views.providers.XpDelaysContentsProvider;
import alma.obops.tmcdbgui.views.providers.XpDelaysEditingSupport;
import alma.obops.tmcdbgui.views.providers.XpDelaysLabelProvider;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.XPDelay;

/**
 * XP (cross polarization) delays editor.
 * @author sharring
 */
public class XpDelaysEditor extends TmcdbObjectEditorPart implements DirtyListener 
{
	public static final String ID = "xpdelays.editor";

	private HwConfiguration owningConfig;
	private boolean dirty = false;
	private Set<XPDelay> xpDelays;
	private Set<XPDelay> xpDelaysCopy;
	private TableViewer xpDelaysTableViewer;
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		InputDialog descriptionInputDialog = new InputDialog(this.getSite().getShell(), "Description", "Please add any comments about your change", "", null);
		if(descriptionInputDialog.open() != Window.OK) 
		{
			return;
		}
		
		try 
		{
			// try to create a new version
			String description = descriptionInputDialog.getValue();
			String userId = System.getProperty("user.name");
			boolean canSave = HwConfigurationConversationUtils.getInstance().prepareHwConfigurationSave(owningConfig, userId, description);

			// if the new version preparation was successful, we can then perform the save
			if(canSave) 
			{			
				for(XpDelayModelRow row : (XpDelayModelRow[]) this.xpDelaysTableViewer.getInput())
				{
					// for any 'new' entries in the XPDelay table, add them to our XP delays
					if(row.getUsbBasebandZeroDelay().getId() == null)
					{
						// for any 'new' entries in the XPDelay table, add them to our XP delays
						xpDelays.add(row.getUsbBasebandZeroDelay());
					}
					else
					{
						// for existing entries, update the delay in our original, so that we can then save it
						XPDelay matchingDelay = findMatchingXPDelay(row.getUsbBasebandZeroDelay());
						matchingDelay.setDelay(row.getUsbBasebandZeroDelay().getDelay());
						matchingDelay.setReceiverBand(row.getUsbBasebandZeroDelay().getReceiverBand());
				        matchingDelay.setSideband(row.getUsbBasebandZeroDelay().getSideband());
				        matchingDelay.setBaseband(row.getUsbBasebandZeroDelay().getBaseband());
					}
					
					if(row.getLsbBasebandZeroDelay().getId() == null)
					{
						// for any 'new' entries in the XPDelay table, add them to our XP delays
						xpDelays.add(row.getLsbBasebandZeroDelay());	
					}
					else
					{
						// for existing entries, update the delay in our original, so that we can then save it
						XPDelay matchingDelay = findMatchingXPDelay(row.getLsbBasebandZeroDelay());
						matchingDelay.setDelay(row.getLsbBasebandZeroDelay().getDelay());
						matchingDelay.setReceiverBand(row.getLsbBasebandZeroDelay().getReceiverBand());
				        matchingDelay.setSideband(row.getLsbBasebandZeroDelay().getSideband());
				        matchingDelay.setBaseband(row.getLsbBasebandZeroDelay().getBaseband());
					}
					
					if(row.getUsbBasebandOneDelay().getId() == null)
					{
						// for any 'new' entries in the XPDelay table, add them to our XP delays
						xpDelays.add(row.getUsbBasebandOneDelay());	
					}
					else
					{
						// for existing entries, update the delay in our original, so that we can then save it
						XPDelay matchingDelay = findMatchingXPDelay(row.getUsbBasebandOneDelay());
						matchingDelay.setDelay(row.getUsbBasebandOneDelay().getDelay());
						matchingDelay.setReceiverBand(row.getUsbBasebandOneDelay().getReceiverBand());
				        matchingDelay.setSideband(row.getUsbBasebandOneDelay().getSideband());
				        matchingDelay.setBaseband(row.getUsbBasebandOneDelay().getBaseband());
					}
					
					if(row.getLsbBasebandOneDelay().getId() == null)
					{
						// for any 'new' entries in the XPDelay table, add them to our XP delays
						xpDelays.add(row.getLsbBasebandOneDelay());
					}
					else
					{
						// for existing entries, update the delay in our original, so that we can then save it
						XPDelay matchingDelay = findMatchingXPDelay(row.getLsbBasebandOneDelay());
						matchingDelay.setDelay(row.getLsbBasebandOneDelay().getDelay());
						matchingDelay.setReceiverBand(row.getLsbBasebandOneDelay().getReceiverBand());
				        matchingDelay.setSideband(row.getLsbBasebandOneDelay().getSideband());
				        matchingDelay.setBaseband(row.getLsbBasebandOneDelay().getBaseband());
					}
					
					if(row.getUsbBasebandTwoDelay().getId() == null)
					{
						// for any 'new' entries in the XPDelay table, add them to our XP delays
						xpDelays.add(row.getUsbBasebandTwoDelay());
					}
					else
					{
						// for existing entries, update the delay in our original, so that we can then save it
						XPDelay matchingDelay = findMatchingXPDelay(row.getUsbBasebandTwoDelay());
						matchingDelay.setDelay(row.getUsbBasebandTwoDelay().getDelay());
						matchingDelay.setReceiverBand(row.getUsbBasebandTwoDelay().getReceiverBand());
				        matchingDelay.setSideband(row.getUsbBasebandTwoDelay().getSideband());
				        matchingDelay.setBaseband(row.getUsbBasebandTwoDelay().getBaseband());
					}
					
					if(row.getLsbBasebandTwoDelay().getId() == null)
					{
						// for any 'new' entries in the XPDelay table, add them to our XP delays
						xpDelays.add(row.getLsbBasebandTwoDelay());
					}
					else
					{
						// for existing entries, update the delay in our original, so that we can then save it
						XPDelay matchingDelay = findMatchingXPDelay(row.getLsbBasebandTwoDelay());
						matchingDelay.setDelay(row.getLsbBasebandTwoDelay().getDelay());
						matchingDelay.setReceiverBand(row.getLsbBasebandTwoDelay().getReceiverBand());
				        matchingDelay.setSideband(row.getLsbBasebandTwoDelay().getSideband());
				        matchingDelay.setBaseband(row.getLsbBasebandTwoDelay().getBaseband());
					}
					
					if(row.getUsbBasebandThreeDelay().getId() == null)
					{
						// for any 'new' entries in the XPDelay table, add them to our XP delays
						xpDelays.add(row.getUsbBasebandThreeDelay());
					}
					else
					{
						// for existing entries, update the delay in our original, so that we can then save it
						XPDelay matchingDelay = findMatchingXPDelay(row.getUsbBasebandThreeDelay());
						matchingDelay.setDelay(row.getUsbBasebandThreeDelay().getDelay());
						matchingDelay.setReceiverBand(row.getUsbBasebandThreeDelay().getReceiverBand());
				        matchingDelay.setSideband(row.getUsbBasebandThreeDelay().getSideband());
				        matchingDelay.setBaseband(row.getUsbBasebandThreeDelay().getBaseband());
					}
					
					if(row.getLsbBasebandThreeDelay().getId() == null)
					{
						// for any 'new' entries in the XPDelay table, add them to our XP delays
						xpDelays.add(row.getLsbBasebandThreeDelay());
					}
					else
					{
						// for existing entries, update the delay in our original, so that we can then save it
						XPDelay matchingDelay = findMatchingXPDelay(row.getLsbBasebandThreeDelay());
						matchingDelay.setDelay(row.getLsbBasebandThreeDelay().getDelay());
						matchingDelay.setReceiverBand(row.getLsbBasebandThreeDelay().getReceiverBand());
				        matchingDelay.setSideband(row.getLsbBasebandThreeDelay().getSideband());
				        matchingDelay.setBaseband(row.getLsbBasebandThreeDelay().getBaseband());
					}
				}
				
				HwConfigurationConversationUtils.getInstance().updateConfiguration(owningConfig);
				this.setDirty(false);	
			}
		}
		catch(Exception ex) 
		{
			ex.printStackTrace();
			throw new RuntimeException("Could not save configuration", ex);
		} 
		finally 
		{
			try {
				HwConfigurationConversationUtils.getInstance().endHwConfigurationSave(owningConfig);
			} catch (Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Could not end save of XP delays", e);
			}
		}
	}

	private XPDelay findMatchingXPDelay(XPDelay xpDelay) 
	{
		XPDelay retVal = null;

		for(XPDelay delay : xpDelays)
		{
			if(delay.getId() != null && delay.getId().equals(xpDelay.getId()))
			{
				retVal = delay;
				break;
			}
		}
		return retVal;
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		XpDelaysEditorInput xpDelayEdInput = (XpDelaysEditorInput)input;
		this.owningConfig = xpDelayEdInput.getConfiguration();
		setInput(input);
		setSite(site);
		setPartName(xpDelayEdInput.getName());
		xpDelays = xpDelayEdInput.getXpDelays();
		makeXpDelaysCopy();
	}
	
	private void makeXpDelaysCopy() 
	{
		xpDelaysCopy = new HashSet<XPDelay>();
		for(XPDelay delay : this.xpDelays)
		{
			XPDelay delayCopy = new XPDelay(delay.getReceiverBand(), 
					delay.getBaseband(), delay.getSideband(), 
					delay.getDelay(), delay.getConfiguration());
			
			delayCopy.setId(delay.getId());
			xpDelaysCopy.add(delayCopy);
		}
	}
	
	private XpDelayModelRow[] populateRows() 
	{
		XpDelayModelRow[] retVal = new XpDelayModelRow[10];
		HashMap<Integer, ArrayList<XPDelay> > xpdelaysMap = new HashMap<Integer, ArrayList<XPDelay> >();
		
		ArrayList<XPDelay> band1Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band2Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band3Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band4Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band5Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band6Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band7Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band8Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band9Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band10Delays = new ArrayList<XPDelay>();
		
		for(XPDelay delay : xpDelaysCopy) 
		{
			if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_01))
			{
				band1Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_02))
			{
				band2Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_03))
			{
				band3Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_04))
			{
				band4Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_05))
			{
				band5Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_06))
			{
				band6Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_07))
			{
				band7Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_08))
			{
				band8Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_09))
			{
				band9Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_10))
			{
				band10Delays.add(delay);
			}
		}
		
		xpdelaysMap.put(0, band1Delays);
		xpdelaysMap.put(1, band2Delays);
		xpdelaysMap.put(2, band3Delays);
		xpdelaysMap.put(3, band4Delays);
		xpdelaysMap.put(4, band5Delays);
		xpdelaysMap.put(5, band6Delays);
		xpdelaysMap.put(6, band7Delays);
		xpdelaysMap.put(7, band8Delays);
		xpdelaysMap.put(8, band9Delays);
		xpdelaysMap.put(9, band10Delays);

		for(int count = 0; count < 10; count++)
		{
			XPDelay usbBB0Delay = findCorrespondingDelay(BasebandName.BB_1, NetSideband.USB, count, xpdelaysMap);
			XPDelay usbBB1Delay = findCorrespondingDelay(BasebandName.BB_2, NetSideband.USB, count, xpdelaysMap);
			XPDelay usbBB2Delay = findCorrespondingDelay(BasebandName.BB_3, NetSideband.USB, count, xpdelaysMap);
			XPDelay usbBB3Delay = findCorrespondingDelay(BasebandName.BB_4, NetSideband.USB, count, xpdelaysMap);
			
			XPDelay lsbBB0Delay = findCorrespondingDelay(BasebandName.BB_1, NetSideband.LSB, count, xpdelaysMap);
			XPDelay lsbBB1Delay = findCorrespondingDelay(BasebandName.BB_2, NetSideband.LSB, count, xpdelaysMap);
			XPDelay lsbBB2Delay = findCorrespondingDelay(BasebandName.BB_3, NetSideband.LSB, count, xpdelaysMap);
			XPDelay lsbBB3Delay = findCorrespondingDelay(BasebandName.BB_4, NetSideband.LSB, count, xpdelaysMap);
			retVal[count] = new XpDelayModelRow(usbBB0Delay, lsbBB0Delay, usbBB1Delay, lsbBB1Delay, usbBB2Delay, 
					                            lsbBB2Delay, usbBB3Delay, lsbBB3Delay, 
					                            DelayEditingUtils.getReceiverBandForValue(count), owningConfig);
		}      
		
		return retVal;
	}

	private XPDelay findCorrespondingDelay(BasebandName bb1, NetSideband nsb,
			int count, HashMap<Integer, ArrayList<XPDelay> > xpdelaysMap) 
	{
		XPDelay retVal = null;
		
		ArrayList<XPDelay> delayList = xpdelaysMap.get(count);
		if(null != delayList)
		{
			for(XPDelay delay: delayList)
			{
				if(delay.getBaseband().equals(bb1) && delay.getSideband().equals(nsb))
				{
					retVal = delay;
					break;
				}
			}
		}
		
		return retVal;
	}

	@Override
	public void createPartControl(Composite parent) {
		Composite editorComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		editorComposite.setLayout(gridLayout);
		gridLayout.numColumns = 1;
		
		Composite tableComposite = new Composite(editorComposite, SWT.NONE);
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = true;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.FILL;
		tableComposite.setLayoutData(gdata);
		tableComposite.setLayout(new FillLayout());
		
		xpDelaysTableViewer = new TableViewer(tableComposite, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Band", "BB0 USB (s)", "BB0 LSB (s)", "BB1 USB (s)", "BB1 LSB (s)", "BB2 USB (s)", "BB2 LSB (s)", "BB3 USB (s)", "BB3 LSB (s)" };
    	for(int i = 0; i < titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(xpDelaysTableViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.setEditingSupport(new XpDelaysEditingSupport(xpDelaysTableViewer, i, this));
    		col.getColumn().pack();
    	}
    	Table table = xpDelaysTableViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

    	xpDelaysTableViewer.setSorter(new XpDelaysViewerSorter());
    	xpDelaysTableViewer.setContentProvider( new XpDelaysContentsProvider() );
    	xpDelaysTableViewer.setLabelProvider( new XpDelaysLabelProvider() );
    	xpDelaysTableViewer.setInput(populateRows()); // trigger a content reload
    	
    	// create a button for retrieving the history
    	Composite buttonComposite = new Composite(editorComposite, SWT.NONE);
    	GridData gridData = new GridData();
    	gridData.grabExcessHorizontalSpace = true;
    	gridData.grabExcessVerticalSpace = false;
    	gridData.horizontalAlignment = SWT.FILL;
    	buttonComposite.setLayoutData(gridData);
    	
    	GridLayout glayout = new GridLayout();
    	glayout.numColumns = 1;
    	glayout.makeColumnsEqualWidth = false;
    	buttonComposite.setLayout(glayout);

    	Button historyButton = new Button(buttonComposite, SWT.PUSH | SWT.CENTER);
        historyButton.setText("History");
        
        historyButton.addSelectionListener(new SelectionAdapter() 
        {
        	public void widgetSelected(SelectionEvent e) 
            {
         	   XpDelaysHistoryEditorInput editorInput =  new XpDelaysHistoryEditorInput(owningConfig);
         	   IWorkbenchWindow win = getSite().getWorkbenchWindow();
         	   try {
         		   win.getActivePage().openEditor(editorInput, XpDelaysHistoryEditor.ID);
         	   } 
         	   catch (PartInitException e1) {
         		   throw new RuntimeException("Could not open XP delays history editor", e1);
         	   }
            }
        });
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}
	
	@Override
	public void doSaveAs() {
		// noop - save as is not allowed
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean d) {
		this.dirty = d;
		firePropertyChange(PROP_DIRTY);
	}
}
