/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.hibernate.exception.ConstraintViolationException;

import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.domain.IModelChangePublisher;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.editors.inputs.WeatherStationEditorInput;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.widgets.WeatherStationAttributesComposite;
import alma.tmcdb.domain.WeatherStationController;

/**
 * Used to edit a weather station.
 * @author sharring
 */
public class WeatherStationEditor extends TmcdbObjectEditor implements
		IModelChangePublisher 
{
	public static final String ID = "weatherstation.editor";
    private static final String CHANGES_NOT_SAVED = "Changes not saved";
	private WeatherStationController weatherStation;
	private WeatherStationController origWeatherStation;
	private WeatherStationAttributesComposite downcastControl;
    private boolean shouldNotifyListeners;
    private List<IModelChangeListener> modelChangeListeners = new ArrayList<IModelChangeListener>();
    
    @Override
    public void setFocus() {
    	downcastControl.setFocus();
    }
    
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		if((!downcastControl.getWeatherStationName().equals(origWeatherStation.getName()) &&
		   (downcastControl.getStatus() != null && downcastControl.getStatus().trim().length() > 0)))
		{
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, downcastControl.getStatus());
			setPartName(origWeatherStation.getName());
		} 
		else 
		{ 
			try {
				this.getSite().getShell().setCursor(this.getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
				applyChangesAndSave();
				this.setEditedObjectAsOriginalContent();
			} 
			catch (ConstraintViolationException e) {
				GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, "WeatherStation already exists: WeatherStation name must be unique within configuration");
				this.resetToOriginalContent();
			} 
			catch (Exception e) {
				GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
				e.printStackTrace();
				this.resetToOriginalContent();
			} 
			finally {
				this.getSite().getShell().setCursor(null);
			}
		}
		
		if(shouldNotifyListeners) {
			this.modelChanged();
			this.shouldNotifyListeners = false;
		}
		setPartName(weatherStation.getName());
		this.downcastControl.setWeatherStation(this.weatherStation);
		this.downcastControl.setDirty(false);
		setDirty(false);
	}
	
	private void applyChangesAndSave() throws Exception 
	{
		String newWeatherStationName = downcastControl.getWeatherStationName();
		if(!this.weatherStation.getName().equals(newWeatherStationName)) {
			shouldNotifyListeners = true;
			this.setPartName(newWeatherStationName);
		} else {
			shouldNotifyListeners = false;
		}
		this.weatherStation.setName(newWeatherStationName);
		this.weatherStation.setCommissionDate(downcastControl.getCommissionDate().getTime());

		BaseElementConversationUtils.getInstance().saveOrUpdateWeatherStation(weatherStation);
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input) throws PartInitException 
	{
		WeatherStationEditorInput wsei = (WeatherStationEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(wsei.getName());

		weatherStation = wsei.getWeatherStation();
		setEditedObjectAsOriginalContent();
	}
	
	@Override
	public void setInput( IEditorInput input ) 
	{
		super.setInput(input);
		WeatherStationEditorInput wsEdInput = ((WeatherStationEditorInput)input);
		WeatherStationController ws = (wsEdInput).getWeatherStation();
		this.modelChangeListeners.clear();
		this.addModelChangeListener(wsEdInput.getModelChangeListener());
		this.weatherStation = ws;
		if(null != downcastControl) 
		{
			configure();
		}
	}

    @Override
	public void createPartControl(Composite parent) 
    {
		parent.setLayout(new FillLayout());
		ScrolledComposite sc1 = new ScrolledComposite(parent,SWT.H_SCROLL |
				SWT.V_SCROLL | SWT.BORDER);
		FillLayout sc1Layout = new FillLayout(org.eclipse.swt.SWT.HORIZONTAL);
		sc1.setLayout(sc1Layout);
		sc1.setExpandHorizontal(true);
		sc1.setExpandVertical(true);
        
		Composite comp = new Composite(sc1, SWT.NONE);
		comp.setLayout(new FillLayout());
		downcastControl = new WeatherStationAttributesComposite(comp, SWT.NONE, this);
		sc1.setContent(comp);
		
		configure();
	}
    
    private void configure() 
	{
		this.downcastControl.setWeatherStation(weatherStation);
		setEditedObjectAsOriginalContent();
	}

	@Override
	public void addModelChangeListener(IModelChangeListener listener) {
		if(null != listener)
		{
			this.modelChangeListeners.add(listener);
		}
	}

	@Override
	public void modelChanged() 
	{
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.internalModelChange();
		}
	}

	@Override
	public void modelShouldBeReloaded() {
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.externalModelChange();
		}
	}

	@Override
	public void removeModelChangeListener(IModelChangeListener listener) {
		this.modelChangeListeners.remove(listener);
	}

	public void resetToOriginalContent() {
		this.weatherStation.setName(this.origWeatherStation.getName());
		this.weatherStation.setCommissionDate(this.origWeatherStation.getCommissionDate());
	}

	@Override
	protected Object getEditedObject() {
		return weatherStation;
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {
		this.origWeatherStation = new WeatherStationController();
		origWeatherStation.setName(weatherStation.getName());
		origWeatherStation.setCommissionDate(weatherStation.getCommissionDate());
	}

}
