/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * TmcdbObjectEditor.java
 */
package alma.obops.tmcdbgui.editors;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.beans.PojoObservables;
import org.eclipse.jface.databinding.swt.SWTObservables;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DateTime;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;

import alma.acs.tmcdb.ComputerProcessorType;
import alma.acs.tmcdb.ContainerImplLang;
import alma.acs.tmcdb.ContainerRealTimeType;
import alma.acs.tmcdb.translator.TmcdbObject;
import alma.obops.tmcdbgui.observablevalues.BACIPropArchMechComboObservableValue;
import alma.obops.tmcdbgui.observablevalues.ComponentImplLangComboObservableValue;
import alma.obops.tmcdbgui.observablevalues.ContStartOptTypeComboObservableValue;
import alma.obops.tmcdbgui.observablevalues.LogLevelComboObservableValue;
import alma.obops.tmcdbgui.observablevalues.RadioButtonsToComputerProcessorTypeObservableValue;
import alma.obops.tmcdbgui.observablevalues.RadioButtonsToContainerImplLangObservableValue;
import alma.obops.tmcdbgui.observablevalues.RadioButtonsToContainerRealTimeTypeObservableValue;
import alma.obops.tmcdbgui.observablevalues.RadioButtonsToStringObservableValue;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;

/**
 * Base class for TMCDB Explorer editors. It contains useful method implementations
 * that can be used by subclasses, as dirty marking and widget subscriptions
 * for change recognition.
 *
 * @author rtobar, Mar 2, 2010
 */


public abstract class TmcdbObjectEditor extends TmcdbObjectEditorPart implements DirtyListener, PropertyChangeListener
{
	private Map<String, PropertyChangeListener> propertyChangeListenerMap = new HashMap<String, PropertyChangeListener>();
	private boolean dirty = false;
	private DataBindingContext _ctx;

	public void doSaveAs() {
		return;
	}
	
	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	/**
	 * Marks the editor as dirty depending on the value of <code>d</code>.
	 *
	 * @param d If the editor should be set as dirty or not
	 */
	public void setDirty(boolean d) {
		dirty = d;
		firePropertyChange(PROP_DIRTY);
	}

	/**
	 * This implementation of the dispose() method performs common things to all the <code>TmcdbObjectEditor</code>s.
	 * First, if the edited object is not null, and is a <code>TmcdbObject</code> object, then it removes
	 * itself from the list of the object's listeners. Then, if the editor is dirty,
	 * it resets the content of the edited object to those previously set as "original".
	 * Finally, it executes the {@link #onDispose()} method, which each editor can override
	 * for its own purposes
	 *
	 * @see #setEditedObjectAsOriginalContent()
	 * @see org.eclipse.ui.part.WorkbenchPart#dispose()
	 */
	public void dispose() {
		if( getEditedObject() != null ) {
			if( getEditedObject() instanceof TmcdbObject ) {
				for(Entry<String, PropertyChangeListener> entry : propertyChangeListenerMap.entrySet()) {
					((TmcdbObject)getEditedObject()).removePropertyChangeListener(entry.getKey(), entry.getValue());					
				}
				propertyChangeListenerMap.clear();
			}
			if( isDirty() ) {
				resetToOriginalContent();
			}
		}
		onDispose();
		super.dispose();
	}

	/**
	 * Returns the instance of the object being edited in the editor.
	 *
	 * @return The object being edited
	 */
	protected abstract Object getEditedObject();

	/**
	 * Set the contents of the edited object to those originally saved through
	 * the {@link #setEditedObjectAsOriginalContent()} method. This method
	 * is called when the editor is closed, and is in a dirty state, meaning
	 * that the changes should not be saved to the DB, so the application object
	 * should reflect the fact that no changes have beend done
	 */
	protected abstract void resetToOriginalContent();

	/**
	 * Editors should implement this method in order to save the initial
	 * contents of the editor. The responsibility of this method is to save
	 * a new instance of a "backup" object, with the contents of the editor input,
	 * which will be used in case of canceling the edition of the object.
	 *
	 * Both objects (editor input and backup copy) should be stored as attributes
	 * of the Editor class. This is why this method doesn't receive any parameter
	 * or returns any object.
	 */
	protected abstract void setEditedObjectAsOriginalContent();

	/**
	 * Extra actions to be taken when disposing the editor. Default implementation does nothing
	 */
	protected void onDispose() {}


	protected void subscribeToChanges(Text ... widgets) {
		KeyListener listener = new KeyListener() {
			public void keyReleased(KeyEvent e) { }
			public void keyPressed(KeyEvent e) {
				switch(e.keyCode) {
					case SWT.CR:
					case SWT.LF:
					case SWT.COMMAND:
					case SWT.KEYPAD_CR:
					case SWT.ESC:
					case SWT.NUM_LOCK:
					case SWT.CAPS_LOCK:
					case SWT.SCROLL_LOCK:
					case SWT.CTRL:
					case SWT.SHIFT:
					case SWT.ALT:
					case SWT.INSERT:
					case SWT.PAGE_DOWN:
					case SWT.PAGE_UP:
					case SWT.HOME:
					case SWT.END:
						break;
					default:
						setDirty(true);
				}

			}
		};
		for(Text t: widgets)
			t.addKeyListener(listener);
	}

	protected void subscribeToChanges(Spinner ... widgets) {
		ModifyListener listener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				setDirty(true);
			}
		};
		for(Spinner s: widgets)
			s.addModifyListener(listener);
	}

	protected void subscribeToChanges(Button ... widgets) {
		SelectionListener listener = new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
			public void widgetDefaultSelected(SelectionEvent e) {
				setDirty(true);
			}
		};
		for(Button b: widgets)
			b.addSelectionListener(listener);
	}

	protected void subscribeToChanges(Combo ... widgets) {
		SelectionListener listener = new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
			public void widgetDefaultSelected(SelectionEvent e) {
				setDirty(true);
			}
		};
		for(Combo b: widgets)
			b.addSelectionListener(listener);
	}

	protected void subscribeToChanges(DateTime ... widgets) {
		SelectionListener listener = new SelectionListener() {
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
			public void widgetDefaultSelected(SelectionEvent e) {
				setDirty(true);
			}
		};
		for(DateTime b: widgets)
			b.addSelectionListener(listener);
	}

	public static Boolean nullSafeBoolean(Boolean b, Boolean v) {
		if( b == null )
			return v;
		return b;
	}

	public static String nullSafeString(Object s, String s2) {
		if( s == null )
			return s2;
		return s.toString();
	}

	public static Integer nullSafeInteger(Integer i1, Integer i2) {
		if( i1 == null )
			return i2;
		return i1;
	}

	public static Double nullSafeDouble(Double d1, Double d2) {
		if( d1 == null )
			return d2;
		return d1;
	}

	public static Byte nullSafeByte(Byte i1, Byte i2) {
		if( i1 == null )
			return i2;
		return i1;
	}

	@Override
	public void propertyChange(PropertyChangeEvent evt) {
		setDirty(true);
	}

	/*********************** DATA BINDING methods **************************/
	public DataBindingContext getDataBindingContext() {
		if( _ctx == null )
			_ctx = new DataBindingContext();
		return _ctx;
	}

	public void bind( String prop, Control widget ) {
		if( widget instanceof Text)
			bind(prop, (Text)widget);
		else if( widget instanceof Button )
			bind(prop, (Button)widget);
		else if( widget instanceof Combo )
			bind(prop, (Combo)widget);
		else if( widget instanceof Spinner )
			bind(prop, (Spinner)widget);
	}

	public void bind( String prop, Text text ) {

		DataBindingContext ctx = getDataBindingContext();
		Object o = getEditedObject();

		ctx.bindValue( SWTObservables.observeText( text, SWT.Modify ),
				PojoObservables.observeValue( o, prop));

		// Not clear why this is necessary with bindValue using SWT.Modify
		// but it seems that it is (for paste to properly set things dirty)
		// e.g. see COMP-4879
		text.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent evt) {
				setDirty(true);
			}
		});


		if( o instanceof TmcdbObject ) {
			((TmcdbObject)o).addPropertyChangeListener(prop, this);
			propertyChangeListenerMap.put(prop, this);
		}
	}

	public void bind( String prop, Button check ) {

		if( (check.getStyle() & SWT.CHECK) != SWT.CHECK )
			return;

		DataBindingContext ctx = getDataBindingContext();
		Object o = getEditedObject();

		ctx.bindValue( SWTObservables.observeSelection( check ),
                PojoObservables.observeValue( o, prop));
		
		if( o instanceof TmcdbObject ) {
			((TmcdbObject)o).addPropertyChangeListener(prop, this);
			propertyChangeListenerMap.put(prop, this);
		}
	}

	public void bind( String prop, ComputerProcessorType [] values, Button ... radios ) {

		for (Button radio: radios)
			if( (radio.getStyle() & SWT.RADIO) != SWT.RADIO )
				return;

		DataBindingContext ctx = getDataBindingContext();
		Object o = getEditedObject();

		ctx.bindValue( new RadioButtonsToComputerProcessorTypeObservableValue(values, radios),
                PojoObservables.observeValue( o, prop));
		
		if( o instanceof TmcdbObject ) {
			((TmcdbObject)o).addPropertyChangeListener(prop, this);
			propertyChangeListenerMap.put(prop, this);
		}
	}

	public void bind( String prop, ContainerImplLang [] values, Button ... radios ) {

		for (Button radio: radios)
			if( (radio.getStyle() & SWT.RADIO) != SWT.RADIO )
				return;

		DataBindingContext ctx = getDataBindingContext();
		Object o = getEditedObject();

		ctx.bindValue( new RadioButtonsToContainerImplLangObservableValue(values, radios),
                PojoObservables.observeValue( o, prop));
		
		if( o instanceof TmcdbObject ) {
			((TmcdbObject)o).addPropertyChangeListener(prop, this);
			propertyChangeListenerMap.put(prop, this);
		}
	}

	public void bind( String prop, ContainerRealTimeType [] values, Button ... radios ) {

		for (Button radio: radios)
			if( (radio.getStyle() & SWT.RADIO) != SWT.RADIO )
				return;

		DataBindingContext ctx = getDataBindingContext();
		Object o = getEditedObject();

		ctx.bindValue( new RadioButtonsToContainerRealTimeTypeObservableValue(values, radios),
                PojoObservables.observeValue( o, prop));
		
		if( o instanceof TmcdbObject ) {
			((TmcdbObject)o).addPropertyChangeListener(prop, this);
			propertyChangeListenerMap.put(prop, this);
		}
	}

	public void bind( String prop, String [] values, Button ... radios ) {

		for (Button radio: radios)
			if( (radio.getStyle() & SWT.RADIO) != SWT.RADIO )
				return;

		DataBindingContext ctx = getDataBindingContext();
		Object o = getEditedObject();

		ctx.bindValue( new RadioButtonsToStringObservableValue(values, radios),
                PojoObservables.observeValue( o, prop));
		
		if( o instanceof TmcdbObject ) {
			((TmcdbObject)o).addPropertyChangeListener(prop, this);
			propertyChangeListenerMap.put(prop, this);
		}
	}

	/**
	 * Special case method for binding the log level combo.
	 * @param prop the property of interest.
	 * @param combo the combo that will be bound.
	 */
	public void bindLogLevelCombo( String prop, Combo combo ) {
		DataBindingContext ctx = getDataBindingContext();
		if( (combo.getStyle() & SWT.READ_ONLY) != SWT.READ_ONLY )
			return;

		if( combo.getData("type") == null || !combo.getData("type").equals("logLevel") )
			return;

		ctx.bindValue( new LogLevelComboObservableValue(combo),
                PojoObservables.observeValue( getEditedObject(), prop));
		
		if(getEditedObject() instanceof TmcdbObject) {
			((TmcdbObject)getEditedObject()).addPropertyChangeListener(prop, this);
			propertyChangeListenerMap.put(prop, this);
		}
	}

	/**
	 * General method for binding a combo to a string field.
	 * @param prop the property name
	 * @param combo the combo
	 */
	public void bind( String prop, Combo combo ) {
		DataBindingContext ctx = getDataBindingContext();
		if( (combo.getStyle() & SWT.READ_ONLY) != SWT.READ_ONLY )
			return;
		
		ctx.bindValue( SWTObservables.observeSelection( combo ),
                PojoObservables.observeValue( getEditedObject(), prop));
		
		if(getEditedObject() instanceof TmcdbObject) {
			((TmcdbObject)getEditedObject()).addPropertyChangeListener(prop, this);
			propertyChangeListenerMap.put(prop, this);
		}
	}

	/**
	 * General method for binding a combo to a BACIPropArchMech enum field.
	 * @param prop the property name
	 * @param combo the combo
	 */
	public void bindBACIPropArchMechCombo( String prop, Combo combo ) {
		DataBindingContext ctx = getDataBindingContext();
		if( (combo.getStyle() & SWT.READ_ONLY) != SWT.READ_ONLY )
			return;

		if( combo.getData("type") == null || !combo.getData("type").equals("archive_mechanism") )
			return;

		ctx.bindValue( new BACIPropArchMechComboObservableValue(combo),
                PojoObservables.observeValue( getEditedObject(), prop));
		
		if(getEditedObject() instanceof TmcdbObject) {
			((TmcdbObject)getEditedObject()).addPropertyChangeListener(prop, this);
			propertyChangeListenerMap.put(prop, this);
		}
	}

	/**
	 * General method for binding a combo to a ComponentImplLang enum field.
	 * @param prop the property name
	 * @param combo the combo
	 */
	public void bindComponentImplLangCombo( String prop, Combo combo ) {
		DataBindingContext ctx = getDataBindingContext();
		if( (combo.getStyle() & SWT.READ_ONLY) != SWT.READ_ONLY )
			return;

		if( combo.getData("type") == null || !combo.getData("type").equals("implLang") )
			return;

		ctx.bindValue( new ComponentImplLangComboObservableValue(combo),
                PojoObservables.observeValue( getEditedObject(), prop));
		
		if(getEditedObject() instanceof TmcdbObject) {
			((TmcdbObject)getEditedObject()).addPropertyChangeListener(prop, this);
			propertyChangeListenerMap.put(prop, this);
		}
	}

	/**
	 * General method for binding a combo to a ContStartOptType enum field.
	 * @param prop the property name
	 * @param combo the combo
	 */
	public void bindContStartOptTypeCombo( String prop, Combo combo ) {
		DataBindingContext ctx = getDataBindingContext();
		if( (combo.getStyle() & SWT.READ_ONLY) != SWT.READ_ONLY )
			return;

		if( combo.getData("type") == null || !combo.getData("type").equals("optionType") )
			return;

		ctx.bindValue( new ContStartOptTypeComboObservableValue(combo),
                PojoObservables.observeValue( getEditedObject(), prop));
		
		if(getEditedObject() instanceof TmcdbObject) {
			((TmcdbObject)getEditedObject()).addPropertyChangeListener(prop, this);
			propertyChangeListenerMap.put(prop, this);
		}
	}

	public void bind( String prop, Spinner spinner ) {

		DataBindingContext ctx = getDataBindingContext();
		Object o = getEditedObject();

		ctx.bindValue( SWTObservables.observeSelection( spinner ),
                PojoObservables.observeValue( o, prop));
		if( o instanceof TmcdbObject ) {
			((TmcdbObject)o).addPropertyChangeListener(prop, this);
			propertyChangeListenerMap.put(prop, this);
		}
	}

	protected boolean invalidInput(Object input, String field) {

		boolean invalid = false;

		if( input == null )
			invalid = true;
		else {
			if( input instanceof String && ((String)input).trim().length() == 0 )
				invalid = true;
		}

		if( invalid ) {
			setDirty(true);
			MessageDialog.openInformation(getSite().getShell(), "Invalid " + field,
			    "The value for field '" + field + "' is invalid");
		}

		return invalid;
	}
}
