/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * RawDataView.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.tmcdbgui.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Layout;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.domain.IModelChangePublisher;
import alma.obops.tmcdbgui.editors.inputs.StartupScenarioEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.StartupScenarioConversationUtils;
import alma.tmcdb.domain.StartupScenario;

/**
 * Edit the contents of a StartupScenario
 * 
 * @author amchavan, Sep 3, 2008
 * 
 */



public class StartupScenarioEditor extends TmcdbObjectEditor implements IModelChangePublisher
{
    public static final String ID = "startup-scenario.editor";

    private List<IModelChangeListener> modelChangeListeners = new ArrayList<IModelChangeListener>();
    private StartupScenario _se;
    private StartupScenario _originalSe;
    private Text name;

    @Override
    public void doSave(IProgressMonitor monitor) 
    {
    	if( _se!= null && !(_se.getName().equals(_originalSe.getName())) )
    	{
        	try {
        		StartupScenarioConversationUtils.getInstance().saveOrUpdateStartupScenario(_se);
    			this.setPartName(_se.getName());
       			this.modelChanged();
    		} catch (Exception e) {
    			GuiUtils.showErrorDialog(getSite().getShell(), "Could not save changes", e.getMessage());
    			e.printStackTrace();
    		}
    	}

    	setEditedObjectAsOriginalContent();
    	setDirty(false);
    }

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		StartupScenarioEditorInput feei = (StartupScenarioEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(feei.getName());

		_se = feei.getStartupScenario();
		setEditedObjectAsOriginalContent();
	}
	
	@Override
	public void setInput( IEditorInput input ) 
    {
		super.setInput(input);
		StartupScenarioEditorInput ssEdInput = ((StartupScenarioEditorInput)input);
		this.modelChangeListeners.clear();
		this.addModelChangeListener(ssEdInput.getModelChangeListener());
    }

    /**
     * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
     */
    @Override
    public void createPartControl( Composite parent ) {

        Layout layout = new GridLayout( 2, false );
        parent.setLayout( layout );

        Label nameLabel = new Label(parent, SWT.NONE);
        nameLabel.setText("Name");
        name = new Text(parent, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = SWT.FILL;
        gridData.grabExcessHorizontalSpace = true;
        name.setLayoutData(gridData);

        bind( "name", name );
        subscribeToChanges(name);
    }
    
	@Override
	public void addModelChangeListener(IModelChangeListener listener) {
		if(null != listener)
		{
			this.modelChangeListeners.add(listener);
		}
	}

	@Override
	public void modelChanged() 
	{
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.internalModelChange();
		}
	}

	@Override
	public void modelShouldBeReloaded() {
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.externalModelChange();
		}
	}

	@Override
	public void removeModelChangeListener(IModelChangeListener listener) {
		this.modelChangeListeners.remove(listener);
	}

	@Override
	public void setFocus() {
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {
		_originalSe = new StartupScenario(_se.getName());
	}

	public void resetToOriginalContent() {
		
	}

	@Override
	protected Object getEditedObject() {
		return _se;
	}
}
