/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.obops.tmcdbgui.editors.inputs.HistoricalPointingModelEditorInput;
import alma.obops.tmcdbgui.editors.inputs.PointingModelHistoryEditorInput;
import alma.obops.tmcdbgui.utils.conversation.PointingModelConversationUtils;
import alma.obops.tmcdbgui.views.providers.HistoryRecordViewerSorter;
import alma.obops.tmcdbgui.views.providers.PointingModelHistoryTableContentsProvider;
import alma.obops.tmcdbgui.views.providers.PointingModelHistoryTableLabelProvider;
import alma.tmcdb.domain.PointingModel;
import alma.tmcdb.history.HistoryRecord;

/**
 * "Editor" (read only) for pointing model history.
 * @author sharring
 *
 */
public class PointingModelHistoryEditor extends EditorPart
{
	public static final String ID = "pointingmodel-history.editor";
	private TableViewer historyViewer;
	private PointingModel pointingModel;

	@Override
	public void createPartControl( Composite parent ) 
	{
		historyViewer = new TableViewer(parent, SWT.BORDER | SWT.FULL_SELECTION | SWT.MULTI);

		// Setup the columns
		String [] titles = { "Version", "Description", "Modifier", "Date" };
		for(int i = 0; i != titles.length; i++) {
			TableViewerColumn col = new TableViewerColumn(historyViewer, SWT.NONE);
			col.getColumn().setText(titles[i]);
			col.getColumn().setMoveable(false);
			col.getColumn().setResizable(true);
			col.getColumn().setWidth(250);
		}
		
		historyViewer.setSorter(new HistoryRecordViewerSorter());
		historyViewer.setContentProvider( new PointingModelHistoryTableContentsProvider() );
		historyViewer.setLabelProvider( new PointingModelHistoryTableLabelProvider() );
		historyViewer.setInput(this.pointingModel);
		
		Table table = historyViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		
		MenuManager popupMenu = new MenuManager();
		final ComparePointingModelsAction comparePointingModelsAction = new ComparePointingModelsAction();
		popupMenu.add(comparePointingModelsAction);
		Menu menu = popupMenu.createContextMenu(table);
		table.setMenu(menu);

		historyViewer.addSelectionChangedListener(new ISelectionChangedListener() 
		{
			@Override
			public void selectionChanged(SelectionChangedEvent evt) 
			{
				ISelection selection = evt.getSelection();
				if(selection instanceof IStructuredSelection) {
					IStructuredSelection structuredSelection = (IStructuredSelection) selection;
					Object[] recordsSelected = structuredSelection.toArray();
					if(recordsSelected.length == 2) {
						comparePointingModelsAction.setEnabled(true);
						comparePointingModelsAction.setPreviousRecord((HistoryRecord)recordsSelected[0]);
						comparePointingModelsAction.setReferenceRecord((HistoryRecord)recordsSelected[1]);
					}
					else {
						comparePointingModelsAction.setEnabled(false);
					}
				}

			}
		});

		IDoubleClickListener listener = new GetHistoricalPointingModelDoubleClickListener();
		historyViewer.addDoubleClickListener(listener);
	}

	@Override
	public void setFocus() {
		historyViewer.getControl().setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor arg0) {
		// NOOP
	}

	@Override
	public void doSaveAs() {
		// NOOP
	}

	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException 
	{
		PointingModelHistoryEditorInput editorInput = (PointingModelHistoryEditorInput)input;
		setInput(input);
		if(null != historyViewer) {
			historyViewer.setInput(editorInput.getPointingModel());
		}
		setSite(site);
		setPartName(editorInput.getName());
	}

	@Override 
	public void setInput(IEditorInput input)
	{
		super.setInput(input);
		this.pointingModel = ((PointingModelHistoryEditorInput) input).getPointingModel();
	}
	
	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}
	
	private class GetHistoricalPointingModelDoubleClickListener implements IDoubleClickListener
	{
		@Override
		public void doubleClick(DoubleClickEvent evt) 
		{
			ISelection selection = evt.getSelection();
			if(selection instanceof IStructuredSelection) {
				IStructuredSelection structuredSelection = (IStructuredSelection) selection;
				if(structuredSelection.getFirstElement() != null) {
					HistoryRecord clickedRecord = (HistoryRecord) structuredSelection.getFirstElement();
					PointingModel historicalPointingModel = null;
					try {
						historicalPointingModel = PointingModelConversationUtils.getInstance().getHistoricalPointingModel(pointingModel, clickedRecord);
					} catch (Exception e) {
						throw new RuntimeException("Unable to get historical pointing model" + e);
					}
					
					HistoricalPointingModelEditorInput editorInput =  
						new HistoricalPointingModelEditorInput(historicalPointingModel, historicalPointingModel, clickedRecord);
					
					IWorkbenchWindow win = getSite().getWorkbenchWindow();
					try {
						win.getActivePage().openEditor(editorInput, HistoricalPointingModelEditor.ID);
					} 
					catch (PartInitException e1) {
						throw new RuntimeException("Could not open historical pointing model editor", e1);
					}
				}
			}
		}
	}
	
	private class ComparePointingModelsAction extends Action
	{
		private HistoryRecord referenceRecord;
		private HistoryRecord previousRecord;
		
		public ComparePointingModelsAction() 
		{
			super("Show differences");
		}
		
		public void setReferenceRecord(HistoryRecord rec)
		{
			this.referenceRecord = rec;
		}
		
		public void setPreviousRecord(HistoryRecord rec)
		{
			this.previousRecord = rec;
		}

		public void run() 
		{
			PointingModel historicalPointingModel = null;
			PointingModel historicalPointingModelPreviousVersion = null;
			try {
				historicalPointingModel = PointingModelConversationUtils.getInstance().getHistoricalPointingModel(pointingModel, referenceRecord);
				historicalPointingModelPreviousVersion = PointingModelConversationUtils.getInstance().getHistoricalPointingModel(pointingModel, previousRecord);
			} catch (Exception e) {
				throw new RuntimeException("Unable to get historical pointing models" + e);
			}
			
			HistoryRecord junkRecord = new HistoryRecord();
			junkRecord.setVersion(0L - (referenceRecord.getVersion() - previousRecord.getVersion()));
			HistoricalPointingModelEditorInput editorInput =  
				new HistoricalPointingModelEditorInput(historicalPointingModel, historicalPointingModelPreviousVersion, 
						junkRecord, "Diff pointing model v." + referenceRecord.getVersion() 
						+ " to v." + previousRecord.getVersion() + " for " + pointingModel.getAntenna().getName());
			
			IWorkbenchWindow win = getSite().getWorkbenchWindow();
			try {
				win.getActivePage().openEditor(editorInput, HistoricalPointingModelEditor.ID);
			} 
			catch (PartInitException e1) {
				throw new RuntimeException("Could not open historical pointing model editor", e1);
			}
		}
	}
}
