/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.obops.tmcdbgui.editors.inputs.HolographyTowerToPadEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.conversation.HolographyTowerToPadConversationUtils;
import alma.obops.tmcdbgui.views.providers.HolographyTowerToPadContentsProvider;
import alma.obops.tmcdbgui.views.providers.HolographyTowerToPadEditingSupport;
import alma.obops.tmcdbgui.views.providers.HolographyTowerToPadLabelProvider;
import alma.obops.tmcdbgui.views.providers.HolographyTowerToPadRow;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.tmcdb.domain.HolographyTowerToPad;

public class PadToHolographyTowerEditor extends TmcdbObjectEditorPart implements DirtyListener 
{
	private static final String AZIMUTH = "Azimuth";
	private static final String ELEVATION = "Elevation";
	public static final String ID = "padtoholographytower.editor";
	private TableViewer tableViewer;
	private HolographyTowerToPad holographyTowerToPad;
	private boolean dirty;
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		HolographyTowerToPadRow[] rows = (HolographyTowerToPadRow[]) tableViewer.getInput();
		for(HolographyTowerToPadRow row: rows)
		{
			if(row.getCoeffName().equals(ELEVATION))
			{
				holographyTowerToPad.setElevation(row.getCoeffValue());
			}
			else if(row.getCoeffName().equals(AZIMUTH))
			{
				holographyTowerToPad.setAzimuth(row.getCoeffValue());
			}
		}
		
		try {
			HolographyTowerToPadConversationUtils.getInstance().saveOrUpdateHolographyTowerToPad(holographyTowerToPad);
			this.setDirty(false);
		} catch (Exception e) {
			throw new RuntimeException("Could not save HolographyTowerToPad object", e);
		}
	}
	
	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		HolographyTowerToPadEditorInput editorInput = (HolographyTowerToPadEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(editorInput.getName());
		holographyTowerToPad = editorInput.getHolographyTowerToPad();
		if(null != tableViewer) {
			tableViewer.setInput(populateRows()); // trigger a content reload
		}
	}

	private HolographyTowerToPadRow[] populateRows() 
	{
		HolographyTowerToPadRow[] retVal = new HolographyTowerToPadRow[2];
		
		retVal[0] = new HolographyTowerToPadRow();
		retVal[0].setCoeffName(ELEVATION);
		retVal[0].setCoeffValue(holographyTowerToPad.getElevation());
		
		retVal[1] = new HolographyTowerToPadRow();
		retVal[1].setCoeffName(AZIMUTH);
		retVal[1].setCoeffValue(holographyTowerToPad.getAzimuth());
		
		return retVal;
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		// not allowed
		return false;
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		tableViewer = new TableViewer(parent, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Name", "Value (degrees)"};
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(tableViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.getColumn().setWidth(100);
    		col.setEditingSupport(new HolographyTowerToPadEditingSupport(tableViewer, i, this));
    	}
    	Table table = tableViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

    	tableViewer.setSorter(new ViewerSorter());
    	tableViewer.setContentProvider( new HolographyTowerToPadContentsProvider() );
    	tableViewer.setLabelProvider( new HolographyTowerToPadLabelProvider() );
    	tableViewer.setInput(populateRows()); // trigger a content reload
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean dirty) {
		this.dirty = dirty;firePropertyChange(PROP_DIRTY);
		
	}
}
