/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.hibernate.exception.ConstraintViolationException;

import alma.obops.dam.ServiceException;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.domain.IModelChangePublisher;
import alma.obops.tmcdbgui.editors.inputs.PadEditorInput;
import alma.obops.tmcdbgui.editors.inputs.PadHistoryEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.widgets.PadAttributesComposite;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.Pad;

/**
 * Editor for pads.
 * @author sharring
 */
public class PadEditor extends TmcdbObjectEditor implements IModelChangePublisher
{
	public static final String ID = "pad.editor";
	private static final String CHANGES_NOT_SAVED = "Changes not saved";
	
    private Pad pad;
	private String originalName;
    private List<IModelChangeListener> modelChangeListeners = new ArrayList<IModelChangeListener>();
    private PadAttributesComposite downcastControl; 
    
	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
	   throws PartInitException 
	{
		PadEditorInput padEditorInput = (PadEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(padEditorInput.getName());

		pad = padEditorInput.getPad();
	}

	@Override
	public void setInput( IEditorInput input ) 
    {
		super.setInput(input);
		PadEditorInput padEdInput = ((PadEditorInput)input);
		Pad padin = (padEdInput).getPad();
		this.modelChangeListeners.clear();
		this.addModelChangeListener(padEdInput.getModelChangeListener());
		this.pad = padin;
		this.originalName = pad.getName();
		if(null != downcastControl) 
		{
			configure();
		}
    }

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		
		ScrolledComposite sc1 = new ScrolledComposite(parent, SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		sc1.setExpandHorizontal(true);
		sc1.setExpandVertical(true);

		Composite comp = new Composite(sc1, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 1;
		comp.setLayout(gridLayout);
		comp.setLayoutData(new GridData(GridData.FILL_BOTH));
		downcastControl = new PadAttributesComposite(comp, SWT.NONE, this);
		downcastControl.setPad(pad);
		
		Composite buttonComposite = new Composite(comp, SWT.NONE);
    	buttonComposite.setLayout(new FillLayout());

		 //   Create and configure the "history" button
        Button historyButton = new Button(buttonComposite, SWT.PUSH | SWT.CENTER);
        historyButton.setText("History");
        
        historyButton.addSelectionListener(new SelectionAdapter() 
        {
        	public void widgetSelected(SelectionEvent e) 
            {
         	   PadHistoryEditorInput editorInput =  new PadHistoryEditorInput(pad);
         	   IWorkbenchWindow win = getSite().getWorkbenchWindow();
         	   try {
         		   win.getActivePage().openEditor(editorInput, PadHistoryEditor.ID);
         	   } 
         	   catch (PartInitException e1) {
         		   throw new RuntimeException("Could not open pad history editor", e1);
         	   }
            }
        });
        
        sc1.setMinSize(comp.computeSize(SWT.DEFAULT,SWT.DEFAULT));
        sc1.setContent(comp);
	}

	@Override
	public void setFocus() {
		downcastControl.setFocus();
	}

	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		if((!downcastControl.getPadName().equals(originalName) &&
    			(downcastControl.getStatus() != null && downcastControl.getStatus().trim().length() > 0))) 
    	{
    		GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, downcastControl.getStatus());
    	} 
    	else 
    	{   
    		InputDialog descriptionInputDialog = new InputDialog(this.getSite().getShell(), "Description", "Please add any comments about your change", "", null);
    		if(descriptionInputDialog.open() != Window.OK) 
    		{
    			return;
    		}
    		try 
    		{
    			// try to create a new version
    			String description = descriptionInputDialog.getValue();
    			String userId = System.getProperty("user.name");
    			boolean canSave = BaseElementConversationUtils.getInstance().preparePadSave(pad, userId, description);

    			// if the new version preparation was successful, we can then perform the save
    			if(canSave) 
    			{			
    				try {
    					this.getSite().getShell().setCursor(new Cursor(this.getSite().getShell().getDisplay(), SWT.CURSOR_WAIT));
    					applyChangesAndSave();
    					setDirty(false);
    					this.downcastControl.setDirty(false);
    				} catch (Exception e) {
    					GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
    					setDirty(true);
    					this.downcastControl.setDirty(true);
    					e.printStackTrace();
    				} finally {
    					this.getSite().getShell().setCursor(null);
    				}
    			} 
    			else 
    			{
    				MessageDialog.openWarning(this.getSite().getShell(), "Unable to save", "Could not save; perhaps someone else is saving now. Try again later.");
    			}
    		} 
    		catch(Exception ex) 
    		{
    			ex.printStackTrace();
    			throw new RuntimeException("Could not save pad", ex);
    		} 
    		finally 
    		{
    			 try {
    				BaseElementConversationUtils.getInstance().endPadSave(pad);
    			} catch (Exception e) {
    				e.printStackTrace();
    			}
    		}
    	}
	}


	private void applyChangesAndSave() throws ServiceException, SecurityException, IllegalArgumentException 
	{
		String newPadName = downcastControl.getPadName();
		Long commissionDate = downcastControl.getCommissionDate().getTime();
		Coordinate position = downcastControl.getPosition();
		Double padDelay = downcastControl.getCableDelay();		

		// update the pad's fields with (potentially) new information
		pad.setCommissionDate(commissionDate);
		pad.setName(newPadName);
		pad.setPosition(position);
		pad.setAvgDelay(padDelay);
		
		try {
			BaseElementConversationUtils.getInstance().saveOrUpdatePad(pad);
			this.downcastControl.setPad(this.pad);
			this.downcastControl.setDirty(false);
			if(!originalName.equals(newPadName)) {
				modelChanged();
				this.setPartName(newPadName);
				originalName = newPadName;
			}
		} catch (ConstraintViolationException e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, "Pad already exists: pad name (prefix + number) must be unique within configuration");
			pad.setName(originalName);
		} 
		catch (InvocationTargetException e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getTargetException().getMessage());
			e.printStackTrace();
		}
		catch (Exception e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
		}
	}
		
	private void configure()
	{
		this.downcastControl.setPad(pad);
		this.originalName = pad.getName();
	}
	
	@Override
	protected Object getEditedObject() {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	protected void resetToOriginalContent() {
		// TODO Auto-generated method stub
		
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {
	}

	@Override
	public void addModelChangeListener(IModelChangeListener listener) {
		if(null != listener)
		{
			this.modelChangeListeners.add(listener);
		}
	}

	@Override
	public void modelChanged() 
	{
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.internalModelChange();
		}
	}

	@Override
	public void modelShouldBeReloaded() {
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.externalModelChange();
		}
	}

	@Override
	public void removeModelChangeListener(IModelChangeListener listener) {
		this.modelChangeListeners.remove(listener);
	}
}
