/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) NRAO - National Radio Astronomy Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/

package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.acs.tmcdb.NotificationServiceMapping;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.editors.inputs.NotificationServiceMappingEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.NotificationServiceMappingConversationUtils;
import alma.obops.tmcdbgui.views.SoftwareDeploymentView;

public class NotificationServiceMappingEditor extends TmcdbObjectEditor 
{
	public static final String ID = "notificationservicemapping.editor";
	private NotificationServiceMapping notificationServiceMapping;
	private NotificationServiceMapping origNotificationServiceMapping;
	private Text defaultNotificationServiceText;

	@Override
	public void setFocus() {
		defaultNotificationServiceText.setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		boolean isNewOption = false;

		if( notificationServiceMapping.getNotificationServiceMappingId() == null ) {
			isNewOption = true;
		}

		// Check for invalid inputs
		if((notificationServiceMapping.getDefaultNotificationService() == null || notificationServiceMapping.getDefaultNotificationService().toString().trim().equals("")))
		{
			MessageDialog.openInformation(getSite().getShell(),
					"Please specify all fields",
					"Notification Service Mapping cannot be saved without all fields defined");
			return;
		}

		// Persist the object
		try {
			NotificationServiceMappingConversationUtils.getInstance().saveOrUpdateNotificationServiceMapping(notificationServiceMapping);
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(getSite().getShell(), 
					"Cannot save NC NotificationService Mapping", "Error while saving NotificationServiceMapping: " + 
					notificationServiceMapping.getDefaultNotificationService());
			return;
		}

		setEditedObjectAsOriginalContent();
		setDirty(false);

		// If we're adding a new containerstartupoption, let's refresh the SDV if available
		if( isNewOption ) {
			SoftwareDeploymentView sdv = (SoftwareDeploymentView)RcpUtils.findView( SoftwareDeploymentView.ID );
			sdv.internalModelChange();
		}
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		ScrolledComposite sc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.BORDER);
		sc.setExpandHorizontal(true);
		sc.setExpandVertical(true);

		Composite composite = new Composite(sc, SWT.NONE);
		composite.setLayout(new GridLayout(2, false));

		/* Name */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label defaultNotificationServiceLabel = new Label(composite, SWT.NONE);
		defaultNotificationServiceLabel.setText("Default Notification Service");
		defaultNotificationServiceLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		defaultNotificationServiceText = new Text(composite, SWT.BORDER);
		defaultNotificationServiceText.setLayoutData(gd);
		
		bind( "defaultNotificationService", defaultNotificationServiceText );
		defaultNotificationServiceText.setEnabled(false);
		
		sc.setContent(composite);
		sc.setMinSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));
	}

	@Override
	protected Object getEditedObject() {
		return notificationServiceMapping;
	}

	@Override
	protected void resetToOriginalContent() {
		notificationServiceMapping.setDefaultNotificationService(origNotificationServiceMapping.getDefaultNotificationService());
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {
		origNotificationServiceMapping = new NotificationServiceMapping();
		origNotificationServiceMapping.setDefaultNotificationService(notificationServiceMapping.getDefaultNotificationService());

		setTitleImage(ImageHelper.getImage(origNotificationServiceMapping));
		String partName = LabelHelper.getNotificationServiceMappingLabel(origNotificationServiceMapping);
		setPartName(partName);
		setTitleToolTip(partName);
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		NotificationServiceMappingEditorInput cei = (NotificationServiceMappingEditorInput)input;
		setInput(input);
		setSite(site);

		notificationServiceMapping = cei.getNotificationServiceMapping();
		if( notificationServiceMapping.getNotificationServiceMappingId() == null) {
			setDirty(true);
		}

		setEditedObjectAsOriginalContent();
	}
}
