/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) NRAO - National Radio Astronomy Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.util.ArrayList;
import java.util.Arrays;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.LoggingConfig;
import alma.acs.tmcdb.Manager;
import alma.obops.tmcdbgui.editors.inputs.ManagerEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ManagerConversationUtils;
import alma.obops.tmcdbgui.widgets.LoggingConfigComposite;

public class ManagerEditor extends TmcdbObjectEditor 
{
	private static final byte DEFAULT_SERVER_THREADS = 10;
	private static final int DEFAULT_TIMEOUT = 50;
	private static final int DEFAULT_CLIENT_PING_INTERVAL = 60;
	private static final int DEFAULT_ADMINISTRATOR_PING_INTERVAL = 45;
	private static final int DEFAULT_CONTAINER_PING_INTERVAL = 30;

	public static final String ID = "manager.editor";
	
	private Manager manager;
	private Manager originalManager;
	private LoggingConfigComposite loggingConfigComposite;
	private List startupList;

	private Spinner timeoutSpinner;
	private Spinner clientPingIntervalSpinner;
	private Spinner administratorPingIntervalSpinner;
	private Spinner containerPingIntervalSpinner;
	private Spinner serverThreadsSpinner;
	
	@Override
	public void setFocus() {
		loggingConfigComposite.setFocus();
	}
	
	@Override
	protected Object getEditedObject() 
	{
		return manager;
	}

	@Override
	protected void resetToOriginalContent() 
	{
		manager.setAdministratorPingInterval(originalManager.getAdministratorPingInterval());
		manager.setClientPingInterval(originalManager.getClientPingInterval());
		manager.setContainerPingInterval(originalManager.getContainerPingInterval());
		manager.setConfiguration(originalManager.getConfiguration());
		manager.setServerThreads(originalManager.getServerThreads());
		manager.setServiceComponents(originalManager.getServiceComponents());
		manager.setServiceDaemons(originalManager.getServiceDaemons());
		manager.setStartup(originalManager.getStartup());
		manager.setTimeout(originalManager.getTimeout());
		
		// and also the logging config
		if( manager.getLoggingConfig() != null && originalManager.getLoggingConfig() != null ) {
			LoggingConfig lc = manager.getLoggingConfig();
			lc.setMinLogLevelDefault(originalManager.getLoggingConfig().getMinLogLevelDefault());
			lc.setMinLogLevelLocalDefault(originalManager.getLoggingConfig().getMinLogLevelLocalDefault());
			lc.setCentralizedLogger(originalManager.getLoggingConfig().getCentralizedLogger());
			lc.setDispatchPacketSize(originalManager.getLoggingConfig().getDispatchPacketSize());
			lc.setImmediateDispatchLevel(originalManager.getLoggingConfig().getImmediateDispatchLevel());
			lc.setFlushPeriodSeconds(originalManager.getLoggingConfig().getFlushPeriodSeconds());
			lc.setMaxLogQueueSize(originalManager.getLoggingConfig().getMaxLogQueueSize());
			lc.setMaxLogsPerSecond(originalManager.getLoggingConfig().getMaxLogsPerSecond());
		}
	}

	@Override
	protected void setEditedObjectAsOriginalContent() 
	{
		originalManager = new Manager();
		originalManager.setAdministratorPingInterval(manager.getAdministratorPingInterval());
		originalManager.setClientPingInterval(manager.getClientPingInterval());
		originalManager.setContainerPingInterval(manager.getContainerPingInterval());
		originalManager.setConfiguration(manager.getConfiguration());
		originalManager.setServerThreads(manager.getServerThreads());
		originalManager.setServiceComponents(manager.getServiceComponents());
		originalManager.setServiceDaemons(manager.getServiceDaemons());
		originalManager.setStartup(manager.getStartup());
		originalManager.setTimeout(manager.getTimeout());
		
		// and also the logging config
		if( manager.getLoggingConfig() != null ) {
			LoggingConfig lc = new LoggingConfig();
			lc.setMinLogLevelDefault(manager.getLoggingConfig().getMinLogLevelDefault());
			lc.setMinLogLevelLocalDefault(manager.getLoggingConfig().getMinLogLevelLocalDefault());
			lc.setCentralizedLogger(manager.getLoggingConfig().getCentralizedLogger());
			lc.setDispatchPacketSize(manager.getLoggingConfig().getDispatchPacketSize());
			lc.setImmediateDispatchLevel(manager.getLoggingConfig().getImmediateDispatchLevel());
			lc.setFlushPeriodSeconds(manager.getLoggingConfig().getFlushPeriodSeconds());
			lc.setMaxLogQueueSize(manager.getLoggingConfig().getMaxLogQueueSize());
			lc.setMaxLogsPerSecond(manager.getLoggingConfig().getMaxLogsPerSecond());
			originalManager.setLoggingConfig(lc);
		}
		if(null != loggingConfigComposite) {
			loggingConfigComposite.setLoggingConfig(manager.getLoggingConfig());
		}
		
		setTitleImage(ImageHelper.getImage(originalManager));
		String partName = "Manager of id: " + manager.getManagerId();
		setPartName(partName);
		setTitleToolTip(partName);
	}

	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		manager.setTimeout(timeoutSpinner.getSelection());
		manager.setServerThreads( (byte) serverThreadsSpinner.getSelection() );
		manager.setAdministratorPingInterval( administratorPingIntervalSpinner.getSelection() );
		manager.setClientPingInterval( clientPingIntervalSpinner.getSelection() );
		manager.setContainerPingInterval(containerPingIntervalSpinner.getSelection() );
		manager.setStartup(getStartupString());
		
		// and also the logging config
		LoggingConfig config = manager.getLoggingConfig();
		if( manager.getLoggingConfig() == null ) {
			config = new LoggingConfig();
		}
		
		config.setMinLogLevelDefault( loggingConfigComposite.getMinLogLevelDefault() );
		config.setMinLogLevelLocalDefault( loggingConfigComposite.getMinLogLevelLocalDefault() );
		config.setCentralizedLogger( loggingConfigComposite.getCentralizedLogger() );
		config.setDispatchPacketSize( loggingConfigComposite.getDispatchPacketSize() );
		config.setImmediateDispatchLevel( loggingConfigComposite.getImmediateDispatchLevel() );
		config.setFlushPeriodSeconds( loggingConfigComposite.getFlushPeriodSeconds() );
		config.setMaxLogQueueSize( loggingConfigComposite.getMaxLogQueueSize() );
		config.setMaxLogsPerSecond( loggingConfigComposite.getMaxLogsPerSecond() );
		manager.setLoggingConfig(config);
		
		// Persist the object
		try {
			ManagerConversationUtils.getInstance().saveOrUpdateManager(manager);
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(getSite().getShell(), "Cannot save Manager", "Error while saving Manager");
			setDirty(true);
			return;
		}

		setEditedObjectAsOriginalContent();
		setDirty(false);
	}
	
	@Override
	public void setDirty(boolean dirty)
	{
		super.setDirty(dirty);
		loggingConfigComposite.setDirty(dirty);
	}

	private String getStartupString() {
		int[] startupSelections = startupList.getSelectionIndices();
		StringBuffer startupBuf = new StringBuffer();
		int count = 0;
		for(int i : startupSelections) {
			startupBuf.append(startupList.getItem(i));
			if(count++ < startupSelections.length - 1) {
				startupBuf.append(",");
			}
		}
		return startupBuf.toString();
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		if (!(input instanceof ManagerEditorInput)) {
			throw new PartInitException("Invalid Input: Must be ManagerEditorInput");
		}
		setSite(site);
		setInput(input);
		ManagerEditorInput mgrInput = (ManagerEditorInput) input;
		this.manager = mgrInput.getManager();
		
		if( manager.getManagerId() == null) {
			setDirty(true);
		}

		setEditedObjectAsOriginalContent();
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		ScrolledComposite sc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.BORDER);
		sc.setExpandHorizontal(true);
		sc.setExpandVertical(true);

		Composite composite = new Composite(sc, SWT.NONE);
		composite.setLayout(new GridLayout(1, false));
		
		// Logging configuration
		loggingConfigComposite = new LoggingConfigComposite(composite, SWT.NONE, manager.getLoggingConfig() == null ? new LoggingConfig() : manager.getLoggingConfig());
		loggingConfigComposite.addDirtyListener(this);
		
		// General settings 
		createGeneralSettingsGroup(composite);
		
		// Startup components
		createStartupComponentsGroup(composite);
		
		serverThreadsSpinner.setSelection( nullSafeByte(manager.getServerThreads(), DEFAULT_SERVER_THREADS) );
		subscribeToChanges(serverThreadsSpinner);
		
		timeoutSpinner.setSelection( nullSafeInteger(manager.getTimeout(), DEFAULT_TIMEOUT) );
		subscribeToChanges(timeoutSpinner);
		
		clientPingIntervalSpinner.setSelection( nullSafeInteger(manager.getClientPingInterval(), DEFAULT_CLIENT_PING_INTERVAL) );
		subscribeToChanges(clientPingIntervalSpinner);
		
		administratorPingIntervalSpinner.setSelection( nullSafeInteger(manager.getAdministratorPingInterval(), DEFAULT_ADMINISTRATOR_PING_INTERVAL) );
		subscribeToChanges(administratorPingIntervalSpinner);

		containerPingIntervalSpinner.setSelection( nullSafeInteger(manager.getContainerPingInterval(), DEFAULT_CONTAINER_PING_INTERVAL) );
		subscribeToChanges(containerPingIntervalSpinner);
		
		// Finally, calculate the minimum size so the scroll composite knows
		// when to start its role
		sc.setContent(composite);
		sc.setMinSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));
	}

	private void createStartupComponentsGroup(Composite composite) 
	{
		Group startupGroup = new Group(composite, SWT.BORDER);
		startupGroup.setText("Auto-start components");
		startupGroup.setLayout(new GridLayout(1, false));
		startupList = new List (startupGroup, SWT.BORDER | SWT.MULTI | SWT.V_SCROLL);
		try {
			ComponentConversationUtils.getInstance().hydrateComponents(manager.getConfiguration());
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException("Could not hydrate components!");
		}
		String [] componentStrings = new String[manager.getConfiguration().getComponents().size()];
		int count = 0;
		for(Component comp: manager.getConfiguration().getComponents()) {
			componentStrings[count++] = LabelHelper.getFullPath(comp, false);
		}
		Arrays.sort(componentStrings);
		
		// filter out the default components (names being equal to asterisk "*")
		java.util.List<String> filteredComponentStrings = new ArrayList<String>();
		for(int i = 0; i < componentStrings.length; i++) {
			if(!componentStrings[i].equals("*")) {
				filteredComponentStrings.add(componentStrings[i]);
			}
		}
		
		startupList.setItems(filteredComponentStrings.toArray(new String[0]));
		startupList.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
		});
		GridData gd = new GridData();
		gd.heightHint = 5 * startupList.getItemHeight() + 2 * startupList.getBorderWidth(); // height for 5 rows
		startupList.setLayoutData(gd);
		
		String [] selectedItems = this.manager.getStartup() == null ? null : this.manager.getStartup().split(",");
		if(selectedItems != null)
		{
			int[] selectedIndices = new int[selectedItems.length];
			count = 0;
			for(String selItem : selectedItems) {
				int selIndex = startupList.indexOf(selItem);
				selectedIndices[count++] = selIndex;
			}
			startupList.select(selectedIndices);
		}
	}

	private void createGeneralSettingsGroup(Composite composite) 
	{
		Group generalSettingsGroup = new Group(composite, SWT.BORDER);
		generalSettingsGroup.setText("General settings");
		generalSettingsGroup.setLayout(new GridLayout(2, false));

		Label timeoutLabel = new Label(generalSettingsGroup, SWT.NONE);
		timeoutLabel.setText("Timeout (s)");
		timeoutSpinner = new Spinner(generalSettingsGroup, SWT.NONE);
		timeoutSpinner.setMinimum(1);
		timeoutSpinner.setMaximum(999);
		
		Label clientPingIntervalLabel = new Label(generalSettingsGroup, SWT.NONE);
		clientPingIntervalLabel.setText("Client ping interval (s)");
		clientPingIntervalSpinner = new Spinner(generalSettingsGroup, SWT.NONE);
		clientPingIntervalSpinner.setMinimum(1);
		clientPingIntervalSpinner.setMaximum(999);
		
		Label administratorPingIntervalLabel = new Label(generalSettingsGroup, SWT.NONE);
		administratorPingIntervalLabel.setText("Administrator ping interval (s)");
		administratorPingIntervalSpinner = new Spinner(generalSettingsGroup, SWT.NONE);
		administratorPingIntervalSpinner.setMinimum(1);
		administratorPingIntervalSpinner.setMaximum(999);
		
		Label containerPingIntervalLabel = new Label(generalSettingsGroup, SWT.NONE);
		containerPingIntervalLabel.setText("Container ping interval (s)");
		containerPingIntervalSpinner = new Spinner(generalSettingsGroup, SWT.NONE);
		containerPingIntervalSpinner.setMinimum(1);
		containerPingIntervalSpinner.setMaximum(999);
		
		Label serverThreadsLabel = new Label(generalSettingsGroup, SWT.NONE);
		serverThreadsLabel.setText("Server threads");
		serverThreadsSpinner = new Spinner(generalSettingsGroup, SWT.NONE);
		serverThreadsSpinner.setMinimum(1);
		serverThreadsSpinner.setMaximum(999);
	}
}
