/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.BasebandNameMod.BasebandName;
import alma.NetSidebandMod.NetSideband;
import alma.ReceiverBandMod.ReceiverBand;
import alma.obops.tmcdbgui.editors.inputs.HistoricalXpDelaysEditorInput;
import alma.obops.tmcdbgui.editors.sorters.XpDelaysViewerSorter;
import alma.obops.tmcdbgui.utils.DelayEditingUtils;
import alma.obops.tmcdbgui.views.providers.XpDelayModelRow;
import alma.obops.tmcdbgui.views.providers.XpDelaysContentsProvider;
import alma.obops.tmcdbgui.views.providers.XpDelaysLabelProvider;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.tmcdb.domain.XPDelay;

/**
 * "Editor" (read-only) for historical xp delays.
 * @author sharring
 *
 */
public class HistoricalXpDelaysEditor extends EditorPart implements DirtyListener 
{
	private Set<XPDelay> historicalXpDelays;
	private Set<XPDelay> referenceXpDelays;
	private TableViewer xpDelaysTableViewer;
	public static final String ID = "historical-xpdelays.editor";

	@Override
	public void doSave(IProgressMonitor monitor) {
		// noop
	}

	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		HistoricalXpDelaysEditorInput xpDelaysEdInput = (HistoricalXpDelaysEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(xpDelaysEdInput.getName());
		historicalXpDelays = xpDelaysEdInput.getReferenceXpDelay();
		referenceXpDelays = xpDelaysEdInput.getPreviousXpDelay();
		if(null != xpDelaysTableViewer) {
			xpDelaysTableViewer.setInput(populateRows()); // trigger a content reload
		}
	}

	@Override
	public void createPartControl(Composite parent) {
		Composite editorComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		editorComposite.setLayout(gridLayout);
		gridLayout.numColumns = 1;
		
		Composite tableComposite = new Composite(editorComposite, SWT.NONE);
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = true;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.FILL;
		tableComposite.setLayoutData(gdata);
		tableComposite.setLayout(new FillLayout());
		
		xpDelaysTableViewer = new TableViewer(tableComposite, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Band", "BB0 USB (s)", "BB0 LSB (s)", "BB1 USB (s)", "BB1 LSB (s)", "BB2 USB (s)", "BB2 LSB (s)", "BB3 USB (s)", "BB3 LSB (s)" };
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(xpDelaysTableViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.getColumn().pack();
    	}
    	Table table = xpDelaysTableViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

    	xpDelaysTableViewer.setSorter(new XpDelaysViewerSorter());
    	xpDelaysTableViewer.setContentProvider( new XpDelaysContentsProvider() );
    	xpDelaysTableViewer.setLabelProvider( new XpDelaysLabelProvider() );
    	xpDelaysTableViewer.setInput(populateRows()); // trigger a content reload
	}
	
	private XpDelayModelRow[] populateRows() 
	{
		XpDelayModelRow[] retVal = makeRowsForXpDelays(historicalXpDelays);

		// NOTE: There are essentially two modes for this editor, one for displaying a 
		// historical focus model as it existed at some time in the past, and another
		// for showing the differences between 2 versions of a focus model. 
		// When we are in 'diff' mode the historical and reference focus model 
		// variables will not be identical; else they will be the same. 
		// If they are the same, we merely show the focus model w/o any highlighting; 
		// whereas if they are different, we highlight the differences between them.
		if(referenceXpDelays != historicalXpDelays)
		{
			retVal = diffXpDelays(retVal);
		}
		
		return retVal;
	}

	private XpDelayModelRow[] diffXpDelays(XpDelayModelRow[] rows) 
	{
		XpDelayModelRow[] previousrows = makeRowsForXpDelays(referenceXpDelays);
		hiliteChanges(rows, previousrows);
		return rows;
	}

	private XpDelayModelRow[] makeRowsForXpDelays(Set<XPDelay> dm) 
	{
		XpDelayModelRow[] retVal = new XpDelayModelRow[10];
		HashMap<Integer, ArrayList<XPDelay> > xpdelaysMap = new HashMap<Integer, ArrayList<XPDelay> >();

		ArrayList<XPDelay> band1Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band2Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band3Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band4Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band5Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band6Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band7Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band8Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band9Delays = new ArrayList<XPDelay>();
		ArrayList<XPDelay> band10Delays = new ArrayList<XPDelay>();

		for(XPDelay delay : dm) 
		{
			if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_01))
			{
				band1Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_02))
			{
				band2Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_03))
			{
				band3Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_04))
			{
				band4Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_05))
			{
				band5Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_06))
			{
				band6Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_07))
			{
				band7Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_08))
			{
				band8Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_09))
			{
				band9Delays.add(delay);
			}
			else if(delay.getReceiverBand().equals(ReceiverBand.ALMA_RB_10))
			{
				band10Delays.add(delay);
			}
		}

		xpdelaysMap.put(0, band1Delays);
		xpdelaysMap.put(1, band2Delays);
		xpdelaysMap.put(2, band3Delays);
		xpdelaysMap.put(3, band4Delays);
		xpdelaysMap.put(4, band5Delays);
		xpdelaysMap.put(5, band6Delays);
		xpdelaysMap.put(6, band7Delays);
		xpdelaysMap.put(7, band8Delays);
		xpdelaysMap.put(8, band9Delays);
		xpdelaysMap.put(9, band10Delays);

		for(int count = 0; count < 10; count++)
		{
			XPDelay usbBB0Delay = findCorrespondingDelay(BasebandName.BB_1, NetSideband.USB, count, xpdelaysMap);
			XPDelay usbBB1Delay = findCorrespondingDelay(BasebandName.BB_2, NetSideband.USB, count, xpdelaysMap);
			XPDelay usbBB2Delay = findCorrespondingDelay(BasebandName.BB_3, NetSideband.USB, count, xpdelaysMap);
			XPDelay usbBB3Delay = findCorrespondingDelay(BasebandName.BB_4, NetSideband.USB, count, xpdelaysMap);

			XPDelay lsbBB0Delay = findCorrespondingDelay(BasebandName.BB_1, NetSideband.LSB, count, xpdelaysMap);
			XPDelay lsbBB1Delay = findCorrespondingDelay(BasebandName.BB_2, NetSideband.LSB, count, xpdelaysMap);
			XPDelay lsbBB2Delay = findCorrespondingDelay(BasebandName.BB_3, NetSideband.LSB, count, xpdelaysMap);
			XPDelay lsbBB3Delay = findCorrespondingDelay(BasebandName.BB_4, NetSideband.LSB, count, xpdelaysMap);
			retVal[count] = new XpDelayModelRow(usbBB0Delay, lsbBB0Delay, usbBB1Delay, lsbBB1Delay, usbBB2Delay, 
					lsbBB2Delay, usbBB3Delay, lsbBB3Delay, 
					DelayEditingUtils.getReceiverBandForValue(count), null );
		}      

		return retVal;
	}
	
	private XPDelay findCorrespondingDelay(BasebandName bb1, NetSideband nsb,
			int count, HashMap<Integer, ArrayList<XPDelay> > xpdelaysMap) 
	{
		ReceiverBand rb = DelayEditingUtils.getReceiverBandForValue(count);
		XPDelay retVal = new XPDelay(rb, bb1, nsb, 0.0d, null);
		
		ArrayList<XPDelay> delayList = xpdelaysMap.get(count);
		if(null != delayList)
		{
			for(XPDelay delay: delayList)
			{
				if(delay.getBaseband().equals(bb1) && delay.getSideband().equals(nsb))
				{
					retVal = delay;
					break;
				}
			}
		}
		
		return retVal;
	}

	private void hiliteChanges(XpDelayModelRow[] rows, XpDelayModelRow[] comparisonXpDelays) 
	{
		// highlight any value changes
		for(XpDelayModelRow row : rows)
		{
			XpDelayModelRow matchingDelayModelRow = findMatchingDelayModelRow(row, comparisonXpDelays);
			
			if(null == matchingDelayModelRow) 
			{
				// SLH TODO!!!
				continue;
			}
			
			Font boldFont = JFaceResources.getFontRegistry().getBold(JFaceResources.DEFAULT_FONT);
			
			// LSB
			if(!row.getLsbBasebandZeroDelay().getDelay().equals(matchingDelayModelRow.getLsbBasebandZeroDelay().getDelay())) {
				row.setLsbBasebandZeroDelayImage(XpDelayModelRow.CHANGED_IMAGE);
				row.setLsbBasebandZeroDelayFont(boldFont);	
			} else {
				row.setLsbBasebandZeroDelayImage(null);
				row.setLsbBasebandZeroDelayFont(null);
			}
			
			if(!row.getLsbBasebandOneDelay().getDelay().equals(matchingDelayModelRow.getLsbBasebandOneDelay().getDelay())) {
				row.setLsbBasebandOneDelayImage(XpDelayModelRow.CHANGED_IMAGE);
				row.setLsbBasebandOneDelayFont(boldFont);	
			} else {
				row.setLsbBasebandOneDelayImage(null);
				row.setLsbBasebandOneDelayFont(null);
			}
			
			if(!row.getLsbBasebandTwoDelay().getDelay().equals(matchingDelayModelRow.getLsbBasebandTwoDelay().getDelay())) {
				row.setLsbBasebandTwoDelayImage(XpDelayModelRow.CHANGED_IMAGE);
				row.setLsbBasebandTwoDelayFont(boldFont);	
			} else {
				row.setLsbBasebandTwoDelayImage(null);
				row.setLsbBasebandTwoDelayFont(null);
			}
			
			if(!row.getLsbBasebandThreeDelay().getDelay().equals(matchingDelayModelRow.getLsbBasebandThreeDelay().getDelay())) {
				row.setLsbBasebandThreeDelayImage(XpDelayModelRow.CHANGED_IMAGE);
				row.setLsbBasebandThreeDelayFont(boldFont);	
			} else {
				row.setLsbBasebandThreeDelayImage(null);
				row.setLsbBasebandThreeDelayFont(null);
			}
			
			// USB
			if(!row.getUsbBasebandZeroDelay().getDelay().equals(matchingDelayModelRow.getUsbBasebandZeroDelay().getDelay())) {
				row.setUsbBasebandZeroDelayImage(XpDelayModelRow.CHANGED_IMAGE);
				row.setUsbBasebandZeroDelayFont(boldFont);	
			} else {
				row.setUsbBasebandZeroDelayImage(null);
				row.setUsbBasebandZeroDelayFont(null);
			}
			
			if(!row.getUsbBasebandOneDelay().getDelay().equals(matchingDelayModelRow.getUsbBasebandOneDelay().getDelay())) {
				row.setUsbBasebandOneDelayImage(XpDelayModelRow.CHANGED_IMAGE);
				row.setUsbBasebandOneDelayFont(boldFont);	
			} else {
				row.setUsbBasebandOneDelayImage(null);
				row.setUsbBasebandOneDelayFont(null);
			}
			
			if(!row.getUsbBasebandTwoDelay().getDelay().equals(matchingDelayModelRow.getUsbBasebandTwoDelay().getDelay())) {
				row.setUsbBasebandTwoDelayImage(XpDelayModelRow.CHANGED_IMAGE);
				row.setUsbBasebandTwoDelayFont(boldFont);	
			} else {
				row.setUsbBasebandTwoDelayImage(null);
				row.setUsbBasebandTwoDelayFont(null);
			}
			
			if(!row.getUsbBasebandThreeDelay().getDelay().equals(matchingDelayModelRow.getUsbBasebandThreeDelay().getDelay())) {
				row.setUsbBasebandThreeDelayImage(XpDelayModelRow.CHANGED_IMAGE);
				row.setUsbBasebandThreeDelayFont(boldFont);	
			} else {
				row.setUsbBasebandThreeDelayImage(null);
				row.setUsbBasebandThreeDelayFont(null);
			}
		}
	}
	
	private XpDelayModelRow findMatchingDelayModelRow(XpDelayModelRow row, XpDelayModelRow[] comparisonXpDelays) 
	{
		XpDelayModelRow retVal = null;
		
		for(XpDelayModelRow iterRow : comparisonXpDelays)
		{
			if(iterRow.getBand().equals(row.getBand())) {
				retVal = iterRow;
				break;
			}
		}
		
		return retVal;
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean dirty) {
		// TODO Auto-generated method stub
		
	}
}
