/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.util.ArrayList;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.ReceiverBandMod.ReceiverBand;
import alma.obops.tmcdbgui.editors.inputs.HistoricalPointingModelEditorInput;
import alma.obops.tmcdbgui.views.providers.PointingModelContentsProvider;
import alma.obops.tmcdbgui.views.providers.PointingModelLabelProvider;
import alma.obops.tmcdbgui.views.providers.PointingModelRow;
import alma.tmcdb.domain.PointingModel;
import alma.tmcdb.domain.PointingModelCoeff;

/**
 * "Editor" (read-only) for historical pointing model.
 * @author sharring
 *
 */
public class HistoricalPointingModelEditor extends EditorPart 
{
	private PointingModel historicalPointingModel;
	private PointingModel referencePointingModel;
	private TableViewer pointingModelViewer;
	public static final String ID = "historical-pointingmodel.editor";


	@Override
	public void doSave(IProgressMonitor monitor) {
		// noop
	}

	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		HistoricalPointingModelEditorInput pointingEdInput = (HistoricalPointingModelEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(pointingEdInput.getName());
		historicalPointingModel = pointingEdInput.getReferencePointingModel();
		referencePointingModel = pointingEdInput.getPreviousPointingModel();
		if(null != pointingModelViewer) {
			pointingModelViewer.setInput(populateRows()); // trigger a content reload
		}
	}

	@Override
	public void createPartControl(final Composite parent) 
	{
		Composite editorComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		editorComposite.setLayout(gridLayout);
		gridLayout.numColumns = 1;
		
		Composite tableComposite = new Composite(editorComposite, SWT.NONE);
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = true;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.FILL;
		tableComposite.setLayoutData(gdata);
		tableComposite.setLayout(new FillLayout());
		
		pointingModelViewer = new TableViewer(tableComposite, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Coefficient", "Value", "b1 offset", "b2 offset", "b3 offset", "b4 offset", "b5 offset", "b6 offset", "b7 offset", "b8 offset", "b9 offset", "b10 offset" };
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(pointingModelViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.getColumn().pack();
    	}
    	Table table = pointingModelViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

    	pointingModelViewer.setSorter(new ViewerSorter());
    	pointingModelViewer.setContentProvider( new PointingModelContentsProvider() );
    	pointingModelViewer.setLabelProvider( new PointingModelLabelProvider() );

    	pointingModelViewer.setInput(populateRows()); // trigger a content reload
    	
    	Composite buttonComposite = new Composite(editorComposite, SWT.NONE);
    	GridData gridData = new GridData();
    	gridData.grabExcessHorizontalSpace = true;
    	gridData.grabExcessVerticalSpace = false;
    	gridData.horizontalAlignment = SWT.FILL;
    	buttonComposite.setLayoutData(gridData);
    	
    	GridLayout glayout = new GridLayout();
    	glayout.numColumns = 4;
    	glayout.makeColumnsEqualWidth = false;
    	buttonComposite.setLayout(glayout);
	}

	private PointingModelRow[] populateRows() 
	{
		PointingModelRow[] retVal = makeRowsForPointingModel(historicalPointingModel);

		// NOTE: There are essentially two modes for this editor, one for displaying a 
		// historical pointing model as it existed at some time in the past, and another
		// for showing the differences between 2 versions of a pointing model. 
		// When we are in 'diff' mode the historical and reference pointing model 
		// variables will not be identical; else they will be the same. 
		// If they are the same, we merely show the pointing model w/o any highlighting; 
		// whereas if they are different, we highlight the differences between them.
		if(referencePointingModel != historicalPointingModel)
		{
			retVal = diffPointingModels(retVal);
		}
		
		return retVal;
	}

	private PointingModelRow[] diffPointingModels(PointingModelRow[] rows) 
	{
		hiliteChangesAndAdditions(rows, referencePointingModel);
		PointingModelRow[] previousrows = makeRowsForPointingModel(referencePointingModel);
		PointingModelRow[] totalRows = hiliteDeletions(rows, previousrows);
		return totalRows;
	}

	private PointingModelRow[] makeRowsForPointingModel(PointingModel pm) 
	{
		PointingModelRow[] previousrows = new PointingModelRow[pm.getTerms().size()];
		int count = 0;
		for(Entry<String, PointingModelCoeff> entry : pm.getTerms().entrySet()) 
		{
			previousrows[count] = new PointingModelRow(pm.getAntenna(), entry.getKey(), entry.getValue());
			count++;
		}
		return previousrows;
	}

	private PointingModelRow[] hiliteDeletions(PointingModelRow[] rows,	PointingModelRow[] previousrows) 
	{
		// now, for completeness we also need to iterate over the current pm and check to see if there are
		// rows in the current pm that are not present in the historical pm; these we will display as italic rows
		// to indicate that they are in the new pm but not in the old pm
		ArrayList<PointingModelRow> missingRows = new ArrayList<PointingModelRow>(); 
		for(PointingModelRow row: previousrows) 
		{
			PointingModelCoeff matchingCoeff = findMatchingEntry(row.getCoeff(), historicalPointingModel);
			if(null == matchingCoeff) 
			{
				missingRows.add(row);
			}
		}
		PointingModelRow[] totalRows = new PointingModelRow[rows.length + missingRows.size()];
		for(int i = 0; i < rows.length; i++) 
		{
			totalRows[i] = rows[i];
		}
		
		int i = rows.length;
		for(PointingModelRow row: missingRows)
		{
			totalRows[i] = row;
			Font italicFont = (JFaceResources.getFontRegistry().getItalic(JFaceResources.DEFAULT_FONT));
			row.setCoeffNameFont(italicFont);
			row.setCoeffNameImage(PointingModelRow.DELETED_IMAGE);
			row.setCoeffValueFont(italicFont);
			row.setOffset1Font(italicFont);
			row.setOffset2Font(italicFont);
			row.setOffset3Font(italicFont);
			row.setOffset4Font(italicFont);
			row.setOffset5Font(italicFont);
			row.setOffset6Font(italicFont);
			row.setOffset7Font(italicFont);
			row.setOffset8Font(italicFont);
			row.setOffset9Font(italicFont);
			row.setOffset10Font(italicFont);
		}
		return totalRows;
	}

	private void hiliteChangesAndAdditions(PointingModelRow[] rows,	PointingModel comparisonPointingModel) 
	{
		// highlight any value changes and/or additions of new coeff's
		for(PointingModelRow row : rows)
		{
			PointingModelCoeff matchingCoeff = findMatchingEntry(row.getCoeff(), comparisonPointingModel);
			
			// 1) deal with any changes to coeffs (e.g. adding of new coeffs)
			if(null == matchingCoeff) 
			{
				// if there was _no_ matchingCoeff in the comparison (previous) pointing model
				// (i.e. the coeff exists in the reference historical pm but not in the reference pm's predecessor)
				// then this means the row was added; highlight the coeffname and return
				markAsAddition(row);
				continue;
			}
			row.setCoeffNameImage(null);
			row.setCoeffNameFont(null);
			
			// 2) deal with any changes to the coeff value
			if(!row.getCoeffValue().equals(matchingCoeff.getValue())) {
				row.setCoeffValueImage(PointingModelRow.CHANGED_IMAGE);
				row.setCoeffValueFont(JFaceResources.getFontRegistry().getBold(
                        JFaceResources.DEFAULT_FONT));
			} else {
				row.setCoeffValueImage(null);
				row.setCoeffValueFont(null);
			}
			
			// 3) deal with any changes to the coeff offsets
			boolean[] foundRb = new boolean[10]; 
			Map<ReceiverBand, Double> matchingCoeffOffsets = matchingCoeff.getOffsets();
			for(Entry<ReceiverBand, Double> entry : matchingCoeffOffsets.entrySet())
			{
				markFoundReceiverBand(entry.getKey(), foundRb);
				markChanges(row, entry.getKey(), entry.getValue());
			}
			markMissingChanges(row, foundRb);
		}
	}

	private void markFoundReceiverBand(ReceiverBand rband, boolean[] foundRb) 
	{
		   if(rband.equals(ReceiverBand.ALMA_RB_01)) {
			   foundRb[0] = true;
		   }
		   else if(rband.equals(ReceiverBand.ALMA_RB_02)) {
			   foundRb[1] = true;
		   }
		   else if(rband.equals(ReceiverBand.ALMA_RB_03)) {
			   foundRb[2] = true;
		   }
		   else if(rband.equals(ReceiverBand.ALMA_RB_04)) {
			   foundRb[3] = true;
		   }
		   else if(rband.equals(ReceiverBand.ALMA_RB_05)) {
			   foundRb[4] = true;
		   }
		   else if(rband.equals(ReceiverBand.ALMA_RB_06)) {
			   foundRb[5] = true;
		   }
		   else if(rband.equals(ReceiverBand.ALMA_RB_07)) {
			   foundRb[6] = true;
		   }
		   else if(rband.equals(ReceiverBand.ALMA_RB_08)) {
			   foundRb[7] = true;
		   }
		   else if(rband.equals(ReceiverBand.ALMA_RB_09)) {
			   foundRb[8] = true;
		   }
		   else if(rband.equals(ReceiverBand.ALMA_RB_10)) {
			   foundRb[9] = true;
		   }
	}
	
	private void markMissingChanges(PointingModelRow row, boolean[] foundRb)
	{
		for(int i = 0; i < foundRb.length; i++) 
		{
			if(foundRb[i] == true) {
				continue;
			}
			
			switch(i) {
			case 0:
				markChanges(row, ReceiverBand.ALMA_RB_01, 0.0);
				break;
			case 1:
				markChanges(row, ReceiverBand.ALMA_RB_02, 0.0);
				break;
			case 2:
				markChanges(row, ReceiverBand.ALMA_RB_03, 0.0);
				break;
			case 3:
				markChanges(row, ReceiverBand.ALMA_RB_04, 0.0);
				break;
			case 4:
				markChanges(row, ReceiverBand.ALMA_RB_05, 0.0);
				break;
			case 5:
				markChanges(row, ReceiverBand.ALMA_RB_06, 0.0);
				break;
			case 6:
				markChanges(row, ReceiverBand.ALMA_RB_07, 0.0);
				break;
			case 7:
				markChanges(row, ReceiverBand.ALMA_RB_08, 0.0);
				break;
			case 8:
				markChanges(row, ReceiverBand.ALMA_RB_09, 0.0);
				break;
			case 9:
				markChanges(row, ReceiverBand.ALMA_RB_10, 0.0);
				break;
			}
		}
	}
	
	private void markChanges(PointingModelRow row, ReceiverBand band, Double value)
	{
		if(band.equals(ReceiverBand.ALMA_RB_01)) {
			if(!value.equals(row.getOffset1())) {
				row.setOffset1Image(PointingModelRow.CHANGED_IMAGE);
				row.setOffset1Font(JFaceResources.getFontRegistry().getBold(
						JFaceResources.DEFAULT_FONT));
			}
			else {
				row.setOffset1Image(null);
				row.setOffset1Font(null);
			}
		}
		else if(band.equals(ReceiverBand.ALMA_RB_02)) {
			if(!value.equals(row.getOffset2())) {
				row.setOffset2Image(PointingModelRow.CHANGED_IMAGE);
				row.setOffset2Font(JFaceResources.getFontRegistry().getBold(
						JFaceResources.DEFAULT_FONT));
			}
			else {
				row.setOffset2Image(null);
				row.setOffset2Font(null);
			}
		}
		else if(band.equals(ReceiverBand.ALMA_RB_03)) {
			if(!value.equals(row.getOffset3())) {
				row.setOffset3Image(PointingModelRow.CHANGED_IMAGE);
				row.setOffset3Font(JFaceResources.getFontRegistry().getBold(
						JFaceResources.DEFAULT_FONT));
			}
			else {
				row.setOffset3Image(null);
				row.setOffset3Font(null);
			}
		}
		else if(band.equals(ReceiverBand.ALMA_RB_04)) {
			if(!value.equals(row.getOffset4())) {
				row.setOffset4Image(PointingModelRow.CHANGED_IMAGE);
				row.setOffset4Font(JFaceResources.getFontRegistry().getBold(
						JFaceResources.DEFAULT_FONT));
			}
			else {
				row.setOffset4Image(null);
				row.setOffset4Font(null);
			}
		}
		else if(band.equals(ReceiverBand.ALMA_RB_05)) {
			if(!value.equals(row.getOffset5())) {
				row.setOffset5Image(PointingModelRow.CHANGED_IMAGE);
				row.setOffset5Font(JFaceResources.getFontRegistry().getBold(
						JFaceResources.DEFAULT_FONT));
			}
			else {
				row.setOffset5Image(null);
				row.setOffset5Font(null);
			}
		}
		else if(band.equals(ReceiverBand.ALMA_RB_06)) {
			if(!value.equals(row.getOffset6())) {
				row.setOffset6Image(PointingModelRow.CHANGED_IMAGE);
				row.setOffset6Font(JFaceResources.getFontRegistry().getBold(
						JFaceResources.DEFAULT_FONT));
			}
			else {
				row.setOffset6Image(null);
				row.setOffset6Font(null);
			}
		}
		else if(band.equals(ReceiverBand.ALMA_RB_07)) {
			if(!value.equals(row.getOffset7())) {
				row.setOffset7Image(PointingModelRow.CHANGED_IMAGE);
				row.setOffset7Font(JFaceResources.getFontRegistry().getBold(
						JFaceResources.DEFAULT_FONT));
			}
			else {
				row.setOffset7Image(null);
				row.setOffset7Font(null);
			}
		}
		else if(band.equals(ReceiverBand.ALMA_RB_08)) {
			if(!value.equals(row.getOffset8())) {
				row.setOffset8Image(PointingModelRow.CHANGED_IMAGE);
				row.setOffset8Font(JFaceResources.getFontRegistry().getBold(
						JFaceResources.DEFAULT_FONT));
			}
			else {
				row.setOffset8Image(null);
				row.setOffset8Font(null);
			}
		}
		else if(band.equals(ReceiverBand.ALMA_RB_09)) {
			if(!value.equals(row.getOffset9())) {
				row.setOffset9Image(PointingModelRow.CHANGED_IMAGE);
				row.setOffset9Font(JFaceResources.getFontRegistry().getBold(
						JFaceResources.DEFAULT_FONT));
			}
			else {
				row.setOffset9Image(null);
				row.setOffset9Font(null);
			}
		}
		else if(band.equals(ReceiverBand.ALMA_RB_10)) {
			if(!value.equals(row.getOffset10())) {
				row.setOffset10Image(PointingModelRow.CHANGED_IMAGE);
				row.setOffset10Font(JFaceResources.getFontRegistry().getBold(
						JFaceResources.DEFAULT_FONT));
			}
			else {
				row.setOffset10Image(null);
				row.setOffset10Font(null);
			}
		}
	}

	private void markAsAddition(PointingModelRow row) 
	{
		Font italicFont = (JFaceResources.getFontRegistry().getItalic(JFaceResources.DEFAULT_FONT));
		row.setCoeffNameFont(italicFont);
		row.setCoeffNameImage(PointingModelRow.ADDED_IMAGE);
		row.setCoeffValueFont(italicFont);
		row.setOffset1Font(italicFont);
		row.setOffset2Font(italicFont);
		row.setOffset3Font(italicFont);
		row.setOffset4Font(italicFont);
		row.setOffset5Font(italicFont);
		row.setOffset6Font(italicFont);
		row.setOffset7Font(italicFont);
		row.setOffset8Font(italicFont);
		row.setOffset9Font(italicFont);
		row.setOffset10Font(italicFont);
	}

	private PointingModelCoeff findMatchingEntry(PointingModelCoeff coeff, PointingModel pmToSearch) 
	{
		PointingModelCoeff retVal = null;
		
		retVal = pmToSearch.getTerm(coeff.getName());
		
		return retVal;
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void setFocus() {
	}
}
