/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.text.DecimalFormat;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableFontProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.obops.tmcdbgui.editors.inputs.HistoricalPadEditorInput;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.views.providers.CoordinateRow;
import alma.obops.tmcdbgui.widgets.AntennaAttributesComposite;
import alma.obops.tmcdbgui.widgets.PadAttributesComposite;
import alma.tmcdb.domain.Pad;

public class HistoricalPadEditor extends EditorPart 
{
	public static final String ID = "historical-pad.editor";
	
	private Pad referencePad;
	private Pad historicalPad;
	private TableViewer padViewer;

	@Override
	public void doSave(IProgressMonitor monitor) {
		// noop
	}
	
	@Override
	public void doSaveAs() {
		// noop - not allowed
	}
	
	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		HistoricalPadEditorInput padEdInput = (HistoricalPadEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(padEdInput.getName());
		historicalPad = padEdInput.getReferencePad();
		referencePad = padEdInput.getPreviousPad();
		if(null != padViewer) {
			padViewer.setInput(populateRows()); // trigger a content reload
		}
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		Composite editorComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		editorComposite.setLayout(gridLayout);
		gridLayout.numColumns = 1;
		
		createPadDelayComposite(editorComposite);
		createPadPositionsGroup(editorComposite);
	}
	
	private void createPadPositionsGroup(Composite editorComposite) 
	{
		Group padPositionTableGroup = new Group(editorComposite, SWT.BORDER);
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = false;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.BEGINNING;
		padPositionTableGroup.setLayoutData(gdata);
		padPositionTableGroup.setLayout(new FillLayout());
		
		padViewer = new TableViewer(padPositionTableGroup, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "X position (m)", "Y position (m)", "Z position (m)"};
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(padViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.getColumn().pack();
    	}
    	Table table = padViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

    	padViewer.setContentProvider( new PadPositionContentsProvider() );
    	padViewer.setLabelProvider( new PadPositionLabelProvider() );
    	padViewer.setInput(populateRows()); // trigger a content reload
	}

	private CoordinateRow[] populateRows() 
	{
		CoordinateRow[] retVal = new CoordinateRow[1];
		retVal[0] = new CoordinateRow();
		retVal[0].setPosition(historicalPad.getPosition());
		
		// NOTE: There are essentially two modes for this editor, one for displaying a 
		// historical pad as it existed at some time in the past, and another
		// for showing the differences between 2 versions of a pad. 
		// When we are in 'diff' mode the historical and reference pad 
		// variables will not be identical; else they will be the same. 
		// If they are the same, we merely show the pad w/o any highlighting; 
		// whereas if they are different, we highlight the differences between them.
		if(referencePad != historicalPad)
		{
			retVal = diffPads(retVal);
		}
		
		return retVal;
	}

	private CoordinateRow[] diffPads(CoordinateRow[] rows) 
	{
		// X position
		CoordinateRow onlyRow = rows[0];
	    Double xPos = onlyRow.getPosition() == null ? 0.0 : onlyRow.getPosition().getX();
	    Double referenceXpos = referencePad.getPosition() == null ? 0.0 : referencePad.getPosition().getX();
	    
		if(!xPos.equals(referenceXpos)) {
			onlyRow.setPositionXImage(CoordinateRow.CHANGED_IMAGE);
			onlyRow.setPositionXFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			onlyRow.setPositionXImage(null);
			onlyRow.setPositionXFont(null);
		}
		
		// Y position
	    Double yPos = onlyRow.getPosition() == null ? 0.0 : onlyRow.getPosition().getY();
	    Double referenceYpos = referencePad.getPosition() == null ? 0.0 : referencePad.getPosition().getY();
	    
		if(!yPos.equals(referenceYpos)) {
			onlyRow.setPositionYImage(CoordinateRow.CHANGED_IMAGE);
			onlyRow.setPositionYFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			onlyRow.setPositionYImage(null);
			onlyRow.setPositionYFont(null);
		}
		
		// Z position
	    Double zPos = onlyRow.getPosition() == null ? 0.0 : onlyRow.getPosition().getZ();
	    Double referenceZpos = referencePad.getPosition() == null ? 0.0 : referencePad.getPosition().getZ();
	    
		if(!zPos.equals(referenceZpos)) {
			onlyRow.setPositionZImage(CoordinateRow.CHANGED_IMAGE);
			onlyRow.setPositionZFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			onlyRow.setPositionZImage(null);
			onlyRow.setPositionZFont(null);
		}
		
		return rows;
	}

	private void createPadDelayComposite(Composite editorComposite)
	{
		Composite composite = new Composite(editorComposite, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 6;
		gridLayout.makeColumnsEqualWidth = false;
		composite.setLayout(gridLayout);
		
		Label tPadLabel = new Label(composite, SWT.None);
		tPadLabel.setText(PadAttributesComposite.PAD_DELAY + " for pad " 
				+ this.historicalPad.getName() + " " + PadAttributesComposite.PAD_DELAY_UNITS);
		Text tPadDelayText = new Text(composite, SWT.BORDER);
		GridData gd = GuiUtils.getGridDataForCharWidth(PadAttributesComposite.NUM_CHARS_FOR_DELAY, tPadDelayText);
		tPadDelayText.setLayoutData(gd);
		tPadDelayText.setEditable(false);
		
		DecimalFormat formatter = new DecimalFormat(AntennaAttributesComposite.OFFSET_FORMAT);
		if(null != historicalPad.getAvgDelay()) {
			String formattedDelay = formatter.format(historicalPad.getAvgDelay());
			tPadDelayText.setText(formattedDelay);
			if(!historicalPad.getAvgDelay().equals(referencePad.getAvgDelay())) {
				Font font = JFaceResources.getFontRegistry().getBold(JFaceResources.DEFAULT_FONT);
				tPadDelayText.setFont(font);
			} else {
				tPadDelayText.setFont(null);
			}
		} else {
			tPadDelayText.setText("N/A");
		}
	}

	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void setFocus() {
	}
	
	private static class PadPositionContentsProvider implements IStructuredContentProvider 
	{
		private CoordinateRow[] rows = null;

		@Override
		public void dispose() {
		}

		@Override
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) 
		{
			rows = (CoordinateRow[]) newInput;
		}

		@Override
		public Object[] getElements(Object inputElement) {
			return rows;
		}
	}
	
	private static class PadPositionLabelProvider extends LabelProvider implements ITableLabelProvider,  ITableFontProvider
	{
		@Override
		public Image getColumnImage(Object element, int columnIndex) {
			Image retVal = null;

			if( !(element instanceof CoordinateRow) ) 
			{
				retVal = null;
			}
			else 
			{
				CoordinateRow row = (CoordinateRow)element;
				switch(columnIndex) 
				{
				case 0:
					retVal = row.getPositionXImage();
					break;
				case 1:
					retVal = row.getPositionYImage();
					break;
				case 2:
					retVal = row.getPositionZImage();
					break;
				default:
					break;
				}
			}

			return retVal;
		}

		@Override
		public String getColumnText(Object element, int columnIndex) 
		{
			String retVal = null;
			
			if( !(element instanceof CoordinateRow) ) 
			{
				retVal = null;
			}
			else 
			{
				CoordinateRow row = (CoordinateRow)element;
				switch(columnIndex) 
				{
				case 0:
					retVal = String.valueOf(row.getPosition().getX());
					break;
				case 1:
					retVal =  String.valueOf(row.getPosition().getY());
					break;
				case 2:
					retVal =  String.valueOf(row.getPosition().getZ());
					break;
				default:
					break;
				}
			}
			return retVal;
		}

		@Override
		public Font getFont(Object element, int columnIndex) {
			Font retVal = null;

			if( !(element instanceof CoordinateRow) ) 
			{
				retVal = null;
			}
			else 
			{
				CoordinateRow row = (CoordinateRow)element;
				switch(columnIndex) 
				{
				case 0:
					retVal = row.getPositionXFont();
					break;
				case 1:
					retVal = row.getPositionYFont();
					break;
				case 2:
					retVal = row.getPositionZFont();
					break;
				default:
					break;
				}
			}
			return retVal;
		}
	}
}
