/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.text.DecimalFormat;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.BasebandNameMod.BasebandName;
import alma.NetSidebandMod.NetSideband;
import alma.PolarizationTypeMod.PolarizationType;
import alma.ReceiverBandMod.ReceiverBand;
import alma.obops.tmcdbgui.editors.inputs.HistoricalDelayModelEditorInput;
import alma.obops.tmcdbgui.editors.sorters.FeDelayViewerSorter;
import alma.obops.tmcdbgui.editors.sorters.IfDelayViewerSorter;
import alma.obops.tmcdbgui.editors.sorters.LoDelayViewerSorter;
import alma.obops.tmcdbgui.utils.DelayEditingUtils;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.views.providers.FeDelayModelContentsProvider;
import alma.obops.tmcdbgui.views.providers.FeDelayModelLabelProvider;
import alma.obops.tmcdbgui.views.providers.FeDelayModelRow;
import alma.obops.tmcdbgui.views.providers.IfDelayModelContentsProvider;
import alma.obops.tmcdbgui.views.providers.IfDelayModelLabelProvider;
import alma.obops.tmcdbgui.views.providers.IfDelayModelRow;
import alma.obops.tmcdbgui.views.providers.LoDelayModelContentsProvider;
import alma.obops.tmcdbgui.views.providers.LoDelayModelLabelProvider;
import alma.obops.tmcdbgui.views.providers.LoDelayModelRow;
import alma.obops.tmcdbgui.views.providers.helpers.config.DelayModel;
import alma.obops.tmcdbgui.widgets.AntennaAttributesComposite;
import alma.tmcdb.domain.FEDelay;
import alma.tmcdb.domain.IFDelay;
import alma.tmcdb.domain.IFProcConnectionState;
import alma.tmcdb.domain.LODelay;

public class HistoricalDelayModelEditor extends EditorPart 
{
	private DelayModel historicalDelayModel;
	private DelayModel referenceDelayModel;
	private TableViewer ifDelayModelViewer, loDelayModelViewer, feDelayModelViewer;
	public static final String ID = "historical-delaymodel.editor";


	@Override
	public void doSave(IProgressMonitor monitor) {
		// noop
	}

	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		HistoricalDelayModelEditorInput delayEdInput = (HistoricalDelayModelEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(delayEdInput.getName());
		historicalDelayModel = delayEdInput.getReferenceDelayModel();
		referenceDelayModel = delayEdInput.getPreviousDelayModel();
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		GridLayout gridLayout = new GridLayout();

		ScrolledComposite sc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER);
		sc.setExpandHorizontal(true);
		sc.setExpandVertical(true);

		Composite editorComposite = new Composite(sc, SWT.NONE);
		editorComposite.setLayout(gridLayout);
		gridLayout.numColumns = 1;
		
		createAntDelayComposite(editorComposite);
		createIfDelaysGroup(editorComposite);
		createLoDelaysGroup(editorComposite);
		createFeDelaysGroup(editorComposite);

		// Finally, calculate the minimum size so the scroll composite knows
		// when to start its role
		sc.setContent(editorComposite);
		sc.setMinSize(editorComposite.computeSize(SWT.DEFAULT, SWT.DEFAULT));
	}
	
	private void createAntDelayComposite(Composite editorComposite)
	{
		Composite composite = new Composite(editorComposite, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 6;
		gridLayout.makeColumnsEqualWidth = false;
		composite.setLayout(gridLayout);
		
		Label tAntLabel = new Label(composite, SWT.None);
		tAntLabel.setText(DelayModelEditor.ANTENNA_DELAY + " for antenna " 
				+ this.historicalDelayModel.getAntenna().getName() + " " + DelayModelEditor.ANTENNA_DELAY_UNITS);
		Text tAntDelayText = new Text(composite, SWT.BORDER);
		GridData gd = GuiUtils.getGridDataForCharWidth(DelayModelEditor.NUM_CHARS_FOR_DELAY, tAntDelayText);
		tAntDelayText.setLayoutData(gd);
		tAntDelayText.setEditable(false);
		
		DecimalFormat formatter = new DecimalFormat(AntennaAttributesComposite.OFFSET_FORMAT);
		if(null != historicalDelayModel.getAntenna().getAvgDelay()) {
			String formattedDelay = formatter.format(historicalDelayModel.getAntenna().getAvgDelay());
			tAntDelayText.setText(formattedDelay);
			if(referenceDelayModel != historicalDelayModel) { 
				if(!historicalDelayModel.getAntenna().getAvgDelay().equals(referenceDelayModel.getAntenna().getAvgDelay())) {
					Font font = JFaceResources.getFontRegistry().getBold(JFaceResources.DEFAULT_FONT);
					tAntDelayText.setFont(font);
				} else {
					tAntDelayText.setFont(null);
				}
			}
		} else {
			tAntDelayText.setText("N/A");
		}
	}

	private void createLoDelaysGroup(Composite editorComposite) {
		Group loDelayTableGroup = new Group(editorComposite, SWT.BORDER);
		loDelayTableGroup.setText("LO Delays");
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = false;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.BEGINNING;
		loDelayTableGroup.setLayoutData(gdata);
		loDelayTableGroup.setLayout(new FillLayout());
		
		loDelayModelViewer = new TableViewer(loDelayTableGroup, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Baseband", "Delay (s)"};
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(loDelayModelViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.getColumn().pack();
    	}
    	Table table = loDelayModelViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

    	loDelayModelViewer.setSorter(new LoDelayViewerSorter());
    	loDelayModelViewer.setContentProvider( new LoDelayModelContentsProvider() );
    	loDelayModelViewer.setLabelProvider( new LoDelayModelLabelProvider() );
    	loDelayModelViewer.setInput(populateLoRows()); // trigger a content reload
	}
	
	private LoDelayModelRow[] populateLoRows() 
	{
		LoDelayModelRow[] retVal = makeRowsForLoDelayModel();
		
		// NOTE: There are essentially two modes for this editor, one for displaying a 
		// historical delay model as it existed at some time in the past, and another
		// for showing the differences between 2 versions of a delay model. 
		// When we are in 'diff' mode the historical and reference delay model 
		// variables will not be identical; else they will be the same. 
		// If they are the same, we merely show the pointing model w/o any highlighting; 
		// whereas if they are different, we highlight the differences between them.
		if(referenceDelayModel != historicalDelayModel)
		{
			retVal = diffLoDelayModels(retVal);
		}
		
		return retVal;
	}

	private LoDelayModelRow[] makeRowsForLoDelayModel() 
	{
		LoDelayModelRow[] retVal = new LoDelayModelRow[4];
		for(LODelay loDelay : this.historicalDelayModel.getLoDelays())
		{
			// for each LODelay in the 
			int bb = this.getIntFromBasebandEnum(loDelay.getBaseband());
			retVal[bb] = new LoDelayModelRow(loDelay);
		}
		for(short i = 0; i < 4; i++) 
		{
			BasebandName bbname = getBasebandNameForValue(i);
			if(retVal[i] == null)
			{
				LODelay newDelay = new LODelay();
				newDelay.setBaseband(bbname);
				newDelay.setDelay(new Double(0));
				newDelay.setId(null);
				retVal[i] = new LoDelayModelRow(newDelay);
			}
		}
		return retVal;
	}
	
	private void createIfDelaysGroup(Composite editorComposite) 
	{
		Group ifDelayTableGroup = new Group(editorComposite, SWT.BORDER);
		ifDelayTableGroup.setText("IF Delays");
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = false;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.BEGINNING;
		ifDelayTableGroup.setLayoutData(gdata);
		ifDelayTableGroup.setLayout(new FillLayout());
		
		ifDelayModelViewer = new TableViewer(ifDelayTableGroup, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Baseband", "USB Low Pol X (s)", "USB Low Pol Y (s)", "USB High Pol X (s)", "USB High Pol Y (s)", 
    			                         "LSB Low Pol X (s)", "LSB Low Pol Y (s)", "LSB High Pol X (s)", "LSB High Pol Y (s)"};
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(ifDelayModelViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.getColumn().pack();
    	}
    	Table table = ifDelayModelViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

    	ifDelayModelViewer.setSorter(new IfDelayViewerSorter());
    	ifDelayModelViewer.setContentProvider( new IfDelayModelContentsProvider() );
    	ifDelayModelViewer.setLabelProvider( new IfDelayModelLabelProvider() );
    	ifDelayModelViewer.setInput(populateIfRows()); // trigger a content reload
	}
	
	private IfDelayModelRow[] populateIfRows() 
	{
		IfDelayModelRow[] retVal = makeRowsForIfDelayModel();
		
		// NOTE: There are essentially two modes for this editor, one for displaying a 
		// historical delay model as it existed at some time in the past, and another
		// for showing the differences between 2 versions of a delay model. 
		// When we are in 'diff' mode the historical and reference delay model 
		// variables will not be identical; else they will be the same. 
		// If they are the same, we merely show the pointing model w/o any highlighting; 
		// whereas if they are different, we highlight the differences between them.
		if(referenceDelayModel != historicalDelayModel)
		{
			retVal = diffIfDelayModels(retVal);
		}
		
		return retVal;
	}
	
	private IfDelayModelRow[] makeRowsForIfDelayModel() 
	{
		IfDelayModelRow[] retVal = new IfDelayModelRow[4];

		for(short i = 0; i < 4; i++)
		{
			retVal[i] = new IfDelayModelRow(this.getBasebandNameForValue(i));	
		}

		for(IFDelay ifDelay : this.historicalDelayModel.getIfDelays())
		{
			int basebandInt = getIntFromBasebandEnum(ifDelay.getBaseband());
			switch(ifDelay.getIfSwitch())
			{
			case USB_LOW: 
				if(ifDelay.getPolarization().equals(PolarizationType.X)) 
				{
					retVal[basebandInt].setUsbLowPolXDelay(ifDelay);
				}
				else if(ifDelay.getPolarization().equals(PolarizationType.Y)) 
				{
					retVal[basebandInt].setUsbLowPolYDelay(ifDelay);
				}
				break;
			case USB_HIGH:
				if(ifDelay.getPolarization().equals(PolarizationType.X)) 
				{
					retVal[basebandInt].setUsbHighPolXDelay(ifDelay);
				}
				else if(ifDelay.getPolarization().equals(PolarizationType.Y)) 
				{
					retVal[basebandInt].setUsbHighPolYDelay(ifDelay);
				}
				break;
			case LSB_LOW:
				if(ifDelay.getPolarization().equals(PolarizationType.X)) 
				{
					retVal[basebandInt].setLsbLowPolXDelay(ifDelay);
				}
				else if(ifDelay.getPolarization().equals(PolarizationType.Y)) 
				{
					retVal[basebandInt].setLsbLowPolYDelay(ifDelay);
				}
				break;
			case LSB_HIGH:
				if(ifDelay.getPolarization().equals(PolarizationType.X)) 
				{
					retVal[basebandInt].setLsbHighPolXDelay(ifDelay);
				}
				else if(ifDelay.getPolarization().equals(PolarizationType.Y)) 
				{
					retVal[basebandInt].setLsbHighPolYDelay(ifDelay);
				}
				break;
			}
		}
		
		for(short i = 0; i < 4; i++)
		{
			BasebandName bbname = getBasebandNameForValue(i);
			if(retVal[i].getUsbHighPolXDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.USB_HIGH);
				ifdelay.setPolarization(PolarizationType.X);
				ifdelay.setId(null);
				retVal[i].setUsbHighPolXDelay(ifdelay);
			}
			if(retVal[i].getUsbHighPolYDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.USB_HIGH);
				ifdelay.setPolarization(PolarizationType.Y);
				ifdelay.setId(null);
				retVal[i].setUsbHighPolYDelay(ifdelay);
			}
			if(retVal[i].getUsbLowPolXDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.USB_LOW);
				ifdelay.setPolarization(PolarizationType.X);
				ifdelay.setId(null);
				retVal[i].setUsbLowPolXDelay(ifdelay);
			}
			if(retVal[i].getUsbLowPolYDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.USB_LOW);
				ifdelay.setPolarization(PolarizationType.Y);
				ifdelay.setId(null);
				retVal[i].setUsbLowPolYDelay(ifdelay);
			}
			if(retVal[i].getLsbHighPolXDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.LSB_HIGH);
				ifdelay.setPolarization(PolarizationType.X);
				ifdelay.setId(null);
				retVal[i].setLsbHighPolXDelay(ifdelay);
			}
			if(retVal[i].getLsbHighPolYDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.LSB_HIGH);
				ifdelay.setPolarization(PolarizationType.Y);
				ifdelay.setId(null);
				retVal[i].setLsbHighPolYDelay(ifdelay);
			}
			if(retVal[i].getLsbLowPolXDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.LSB_LOW);
				ifdelay.setPolarization(PolarizationType.X);
				ifdelay.setId(null);
				retVal[i].setLsbLowPolXDelay(ifdelay);
			}
			if(retVal[i].getLsbLowPolYDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.LSB_LOW);
				ifdelay.setPolarization(PolarizationType.Y);
				ifdelay.setId(null);
				retVal[i].setLsbLowPolYDelay(ifdelay);
			}
		}
		
		return retVal;
	}
	
	private BasebandName getBasebandNameForValue(short i) 
	{
		BasebandName retVal = null;
		switch(i)
		{
		case 0:
			retVal = BasebandName.BB_1;
			break;
		case 1:
			retVal = BasebandName.BB_2;
			break;
		case 2:
			retVal = BasebandName.BB_3;
			break;
		case 3:
			retVal = BasebandName.BB_4;
			break;
		default:
			throw new IllegalStateException("ALMA only supports 4 basebands; enum attempts to use more than 4");
		}
		return retVal;
	}

	private int getIntFromBasebandEnum(BasebandName baseband) 
	{
		int retVal = -1;
		if(baseband.equals(BasebandName.BB_1))
		{
			retVal = 0;
		}
		else if(baseband.equals(BasebandName.BB_2))
		{
			retVal = 1;
		}
		else if(baseband.equals(BasebandName.BB_3))
		{
			retVal = 2;
		}
		else if(baseband.equals(BasebandName.BB_4))
		{
			retVal = 3;
		}	
		else 
		{
			throw new IllegalStateException("ALMA only supports 4 basebands, but enumeration attempts to use more than 4");
		}
		return retVal;
	}
	
	private void createFeDelaysGroup(Composite editorComposite) 
	{
		Group feDelayTableGroup = new Group(editorComposite, SWT.BORDER);
		feDelayTableGroup.setText("FE Delays");
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = true;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.FILL;
		feDelayTableGroup.setLayoutData(gdata);
		feDelayTableGroup.setLayout(new FillLayout());
		
		feDelayModelViewer = new TableViewer(feDelayTableGroup, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Receiver band", "USB Pol X (s)", "USB Pol Y (s)", "LSB Pol X (s)", "LSB Pol Y (s)"};
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(feDelayModelViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.getColumn().pack();
    	}
    	Table table = feDelayModelViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

    	feDelayModelViewer.setSorter(new FeDelayViewerSorter());
    	feDelayModelViewer.setContentProvider( new FeDelayModelContentsProvider() );
    	feDelayModelViewer.setLabelProvider( new FeDelayModelLabelProvider() );
    	feDelayModelViewer.setInput(populateFeRows()); // trigger a content reload
	}
	
	private FeDelayModelRow[] populateFeRows() 
	{
		FeDelayModelRow[] retVal = makeRowsForFeDelayModel();
		
		// NOTE: There are essentially two modes for this editor, one for displaying a 
		// historical delay model as it existed at some time in the past, and another
		// for showing the differences between 2 versions of a delay model. 
		// When we are in 'diff' mode the historical and reference delay model 
		// variables will not be identical; else they will be the same. 
		// If they are the same, we merely show the pointing model w/o any highlighting; 
		// whereas if they are different, we highlight the differences between them.
		if(referenceDelayModel != historicalDelayModel)
		{
			retVal = diffFeDelayModels(retVal);
		}
		
		return retVal;
	}
	
	
	private FeDelayModelRow[] diffFeDelayModels(FeDelayModelRow[] rows) 
	{
		hiliteFeChanges(rows, referenceDelayModel);
		return rows;
	}

	private void hiliteFeChanges(FeDelayModelRow[] rows, DelayModel comparisonDelayModel) 
	{
		// highlight any value changes 
		for(FeDelayModelRow row : rows)
		{
			markFeChanges(row, comparisonDelayModel);
		}
	}

	private void markFeChanges(FeDelayModelRow row, DelayModel comparisonDelayModel) 
	{ 
		FEDelay matchingLsbPolXDelay = findMatchingFeDelay(row.getBand(), PolarizationType.X, NetSideband.LSB, comparisonDelayModel);
	    FEDelay matchingLsbPolYDelay = findMatchingFeDelay(row.getBand(), PolarizationType.Y, NetSideband.LSB, comparisonDelayModel);
	    
	    FEDelay matchingUsbPolXDelay = findMatchingFeDelay(row.getBand(), PolarizationType.X, NetSideband.USB, comparisonDelayModel);
	    FEDelay matchingUsbPolYDelay = findMatchingFeDelay(row.getBand(), PolarizationType.Y, NetSideband.USB, comparisonDelayModel);
	    
	    // LSB polarization X
	    Double rowDelay = row.getLsbPolXDelay() == null ? 0.0 : row.getLsbPolXDelay().getDelay();
	    Double matchingDelay = matchingLsbPolXDelay == null ? 0.0 : matchingLsbPolXDelay.getDelay();
	    
		if(!rowDelay.equals(matchingDelay)) {
			row.setLsbPolXDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setLsbPolXDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setLsbPolXDelayImage(null);
			row.setLsbPolXDelayFont(null);
		}
		
		// USB polarization X
		rowDelay = row.getUsbPolXDelay() == null ? 0.0 : row.getUsbPolXDelay().getDelay();
	    matchingDelay = matchingUsbPolXDelay == null ? 0.0 : matchingUsbPolXDelay.getDelay();
	    
		if(!rowDelay.equals(matchingDelay)) {
			row.setUsbPolXDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setUsbPolXDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setUsbPolXDelayImage(null);
			row.setUsbPolXDelayFont(null);
		}
		
		// LSB polarization Y
		rowDelay = row.getLsbPolYDelay() == null ? 0.0 : row.getLsbPolYDelay().getDelay();
	    matchingDelay = matchingLsbPolYDelay == null ? 0.0 : matchingLsbPolYDelay.getDelay();
	    
		if(!rowDelay.equals(matchingDelay)) {
			row.setLsbPolYDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setLsbPolYDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setLsbPolYDelayImage(null);
			row.setLsbPolYDelayFont(null);
		}
		
		// USB polarization Y
		rowDelay = row.getUsbPolYDelay() == null ? 0.0 : row.getUsbPolYDelay().getDelay();
	    matchingDelay = matchingUsbPolYDelay == null ? 0.0 : matchingUsbPolYDelay.getDelay();
	    
		if(!rowDelay.equals(matchingDelay)) {
			row.setUsbPolYDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setUsbPolYDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setUsbPolYDelayImage(null);
			row.setUsbPolYDelayFont(null);
		}
	}

	private FEDelay findMatchingFeDelay(short band, PolarizationType polarization, NetSideband sideband,
			DelayModel comparisonDelayModel) 
	{
		FEDelay retVal = null;
		
		for(FEDelay delay: comparisonDelayModel.getFeDelays()) 
		{
			if(delay.getReceiverBand().value() == band && 
			   delay.getPolarization().equals(polarization) &&
			   delay.getSideband().equals(sideband)) 
			{
				retVal = delay;
				break;
			}
		}
		
		return retVal;
	}

	private FeDelayModelRow[] makeRowsForFeDelayModel() 
	{
		FeDelayModelRow[] retVal = new FeDelayModelRow[10];

		for(short i = 0; i < 10; i++)
		{
			retVal[i] = new FeDelayModelRow(i);	
		}

		for(FEDelay feDelay : this.historicalDelayModel.getFeDelays())
		{
			int basebandVal = DelayEditingUtils.getIntFromReceiverBandEnum(feDelay.getReceiverBand());
			if(feDelay.getSideband().equals(NetSideband.USB)) 
			{
				if(feDelay.getPolarization().equals(PolarizationType.X)) 
				{
					retVal[basebandVal].setUsbPolXDelay(feDelay);
				}
				else if(feDelay.getPolarization().equals(PolarizationType.Y)) 
				{
					retVal[basebandVal].setUsbPolYDelay(feDelay);
				}
			} 
			else if(feDelay.getSideband().equals(NetSideband.LSB))
			{
				if(feDelay.getPolarization().equals(PolarizationType.X)) 
				{
					retVal[basebandVal].setLsbPolXDelay(feDelay);
				}
				else if(feDelay.getPolarization().equals(PolarizationType.Y)) 
				{
					retVal[basebandVal].setLsbPolYDelay(feDelay);
				}
			}
		}
		
		for(short i = 0; i < 10; i++)
		{
			if(retVal[i].getUsbPolXDelay() == null) 
			{
				FEDelay fedelay = new FEDelay();
				assignBaseband(fedelay, i);
				fedelay.setDelay(new Double(0));
				fedelay.setSideband(NetSideband.USB);
				fedelay.setPolarization(PolarizationType.X);
				fedelay.setId(null);
				retVal[i].setUsbPolXDelay(fedelay);
			}
			if(retVal[i].getUsbPolYDelay() == null) 
			{
				FEDelay fedelay = new FEDelay();
				fedelay.setReceiverBand(DelayEditingUtils.getReceiverBandForValue(i)); 
				fedelay.setDelay(new Double(0));
				fedelay.setSideband(NetSideband.USB);
				fedelay.setPolarization(PolarizationType.Y);
				fedelay.setId(null);
				retVal[i].setUsbPolYDelay(fedelay);
			}
			if(retVal[i].getLsbPolYDelay() == null) 
			{
				FEDelay fedelay = new FEDelay();
				fedelay.setReceiverBand(DelayEditingUtils.getReceiverBandForValue(i)); 
				fedelay.setDelay(new Double(0));
				fedelay.setSideband(NetSideband.LSB);
				fedelay.setPolarization(PolarizationType.Y);
				fedelay.setId(null);
				retVal[i].setLsbPolYDelay(fedelay);
			}
			if(retVal[i].getLsbPolXDelay() == null) 
			{
				FEDelay fedelay = new FEDelay();
				fedelay.setReceiverBand(DelayEditingUtils.getReceiverBandForValue(i)); 
				fedelay.setDelay(new Double(0));
				fedelay.setSideband(NetSideband.LSB);
				fedelay.setPolarization(PolarizationType.X);
				fedelay.setId(null);
				retVal[i].setLsbPolXDelay(fedelay);
			}
		}
		
		return retVal;
	}
	
	private void assignBaseband(FEDelay fedelay, short i) 
	{
		switch(i) 
		{
		case 0:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_01);
			break;
		case 1:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_02);
			break;
		case 2:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_03);
			break;
		case 3:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_04);
			break;
		case 4:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_05);
			break;
		case 5:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_06);
			break;
		case 6:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_07);
			break;
		case 7:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_08);
			break;
		case 8:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_09);
			break;
		case 9:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_10);
			break;
		default:
			throw new IllegalStateException("ALMA only supports 10 receiver bands, but enum has more than 10");
		}
	}

	private IfDelayModelRow[] diffIfDelayModels(IfDelayModelRow[] rows) 
	{
		hiliteIfChanges(rows, referenceDelayModel);
		return rows;
	}

	private void hiliteIfChanges(IfDelayModelRow[] rows, DelayModel comparisonDelayModel) 
	{
		// highlight any value changes 
		for(IfDelayModelRow row : rows)
		{
			markIfChanges(row, comparisonDelayModel);
		}
	}

	private void markIfChanges(IfDelayModelRow row, DelayModel comparisonDelayModel) 
	{
	    IFDelay matchingLsbHighPolXDelay = findMatchingIfDelay(row.getBaseband(), PolarizationType.X, IFProcConnectionState.LSB_HIGH, comparisonDelayModel);
	    IFDelay matchingLsbLowPolXDelay = findMatchingIfDelay(row.getBaseband(), PolarizationType.X, IFProcConnectionState.LSB_LOW, comparisonDelayModel);
	    
	    IFDelay matchingUsbHighPolXDelay = findMatchingIfDelay(row.getBaseband(), PolarizationType.X, IFProcConnectionState.USB_HIGH, comparisonDelayModel);
	    IFDelay matchingUsbLowPolXDelay = findMatchingIfDelay(row.getBaseband(), PolarizationType.X, IFProcConnectionState.USB_LOW, comparisonDelayModel);
	    
	    IFDelay matchingLsbHighPolYDelay = findMatchingIfDelay(row.getBaseband(), PolarizationType.Y, IFProcConnectionState.LSB_HIGH, comparisonDelayModel);
	    IFDelay matchingLsbLowPolYDelay = findMatchingIfDelay(row.getBaseband(), PolarizationType.Y, IFProcConnectionState.LSB_LOW, comparisonDelayModel);
	    
	    IFDelay matchingUsbHighPolYDelay = findMatchingIfDelay(row.getBaseband(), PolarizationType.Y, IFProcConnectionState.USB_HIGH, comparisonDelayModel);
	    IFDelay matchingUsbLowPolYDelay = findMatchingIfDelay(row.getBaseband(), PolarizationType.Y, IFProcConnectionState.USB_LOW, comparisonDelayModel);
	    
	    // LSB high polarization X
	    Double rowDelay = row.getLsbHighPolXDelay() == null ? 0.0 : row.getLsbHighPolXDelay().getDelay();
	    Double matchingDelay = matchingLsbHighPolXDelay == null ? 0.0 : matchingLsbHighPolXDelay.getDelay();
	    
		if(!rowDelay.equals(matchingDelay)) {
			row.setLsbHighPolXDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setLsbHighPolXDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setLsbHighPolXDelayImage(null);
			row.setLsbHighPolXDelayFont(null);
		}
		
		// LSB low polarization X
		rowDelay = row.getLsbLowPolXDelay() == null ? 0.0 : row.getLsbLowPolXDelay().getDelay();
	    matchingDelay = matchingLsbLowPolXDelay == null ? 0.0 : matchingLsbLowPolXDelay.getDelay();
	    
	    if(!rowDelay.equals(matchingDelay)) {
			row.setLsbLowPolXDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setLsbLowPolXDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setLsbLowPolXDelayImage(null);
			row.setLsbLowPolXDelayFont(null);
		}
		
		// USB high polarization X
		rowDelay = row.getUsbHighPolXDelay() == null ? 0.0 : row.getUsbHighPolXDelay().getDelay();
	    matchingDelay = matchingUsbHighPolXDelay == null ? 0.0 : matchingUsbHighPolXDelay.getDelay();
	    
	    if(!rowDelay.equals(matchingDelay)) {
			row.setUsbHighPolXDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setUsbHighPolXDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setUsbHighPolXDelayImage(null);
			row.setUsbHighPolXDelayFont(null);
		}
		
		// USB low polarization X
	    rowDelay = row.getUsbLowPolXDelay() == null ? 0.0 : row.getUsbLowPolXDelay().getDelay();
	    matchingDelay = matchingUsbLowPolXDelay == null ? 0.0 : matchingUsbLowPolXDelay.getDelay();
	    
	    if(!rowDelay.equals(matchingDelay)) {
			row.setUsbLowPolXDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setUsbLowPolXDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setUsbLowPolXDelayImage(null);
			row.setUsbLowPolXDelayFont(null);
		}
		
		// LSB high polarization Y
	    rowDelay = row.getLsbHighPolYDelay() == null ? 0.0 : row.getLsbHighPolYDelay().getDelay();
	    matchingDelay = matchingLsbHighPolYDelay == null ? 0.0 : matchingLsbHighPolYDelay.getDelay();
	    
	    if(!rowDelay.equals(matchingDelay)) {
			row.setLsbHighPolYDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setLsbHighPolYDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setLsbHighPolYDelayImage(null);
			row.setLsbHighPolYDelayFont(null);
		}
		
		// LSB low polarization Y
	    rowDelay = row.getLsbLowPolYDelay() == null ? 0.0 : row.getLsbLowPolYDelay().getDelay();
	    matchingDelay = matchingLsbLowPolYDelay == null ? 0.0 : matchingLsbLowPolYDelay.getDelay();
	    
	    if(!rowDelay.equals(matchingDelay)) {
			row.setLsbLowPolYDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setLsbLowPolYDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setLsbLowPolYDelayImage(null);
			row.setLsbLowPolYDelayFont(null);
		}
		
		// USB high polarization Y
	    rowDelay = row.getUsbHighPolYDelay() == null ? 0.0 : row.getUsbHighPolYDelay().getDelay();
	    matchingDelay = matchingUsbHighPolYDelay == null ? 0.0 : matchingUsbHighPolYDelay.getDelay();
	    
	    if(!rowDelay.equals(matchingDelay)) {
			row.setUsbHighPolYDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setUsbHighPolYDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setUsbHighPolYDelayImage(null);
			row.setUsbHighPolYDelayFont(null);
		}
		
		// USB low polarization Y
	    rowDelay = row.getUsbLowPolYDelay() == null ? 0.0 : row.getUsbLowPolYDelay().getDelay();
	    matchingDelay = matchingUsbLowPolYDelay == null ? 0.0 : matchingUsbLowPolYDelay.getDelay();
	    
	    if(!rowDelay.equals(matchingDelay)) {
			row.setUsbLowPolYDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setUsbLowPolYDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setUsbLowPolYDelayImage(null);
			row.setUsbLowPolYDelayFont(null);
		}
	}

	private IFDelay findMatchingIfDelay(BasebandName band, PolarizationType polarization, 
			IFProcConnectionState connectionState, DelayModel comparisonDelayModel) 
	{
		IFDelay retVal = null;
		
		for(IFDelay delay: comparisonDelayModel.getIfDelays()) 
		{
			if(delay.getBaseband().equals(band) && 
			   delay.getPolarization().equals(polarization) && 
			   delay.getIfSwitch().equals(connectionState)) 
			{
				retVal = delay;
				break;
			}
		}
		
		return retVal;
	}

	private LoDelayModelRow[] diffLoDelayModels(LoDelayModelRow[] rows) 
	{
		hiliteLoChanges(rows, referenceDelayModel);
		return rows;
	}

	private void hiliteLoChanges(LoDelayModelRow[] rows, DelayModel comparisonDelayModel) 
	{
		// highlight any value changes 
		for(LoDelayModelRow row : rows)
		{
			markLoChanges(row, comparisonDelayModel);
		}
	}

	private void markLoChanges(LoDelayModelRow row, DelayModel comparisonDelayModel) 
	{
		LODelay matchingDelay = findMatchingLoDelay(row.getDelay().getBaseband(), comparisonDelayModel);
		
		Double rowDelay = row.getDelay() == null ? 0.0 : row.getDelay().getDelay();
	    Double matchingDelayDouble = matchingDelay == null ? 0.0 : matchingDelay.getDelay();
	    
		if(!rowDelay.equals(matchingDelayDouble)) {
			row.setDelayImage(IfDelayModelRow.CHANGED_IMAGE);
			row.setDelayFont(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			row.setDelayImage(null);
			row.setDelayFont(null);
		}
		
	}

	private LODelay findMatchingLoDelay(BasebandName band, DelayModel comparisonDelayModel) 
	{
		LODelay retVal = null;
		
		for(LODelay delay: comparisonDelayModel.getLoDelays()) 
		{
			if(delay.getBaseband().equals(band)) 
			{
				retVal = delay;
				break;
			}
		}
		
		return retVal;
	}

	
	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void setFocus() {
	}

}
