/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.ITableFontProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.obops.tmcdbgui.editors.inputs.HistoricalAntennaToPadEditorInput;
import alma.obops.tmcdbgui.views.providers.AntennaToPadContentsProvider;
import alma.obops.tmcdbgui.views.providers.AntennaToPadLabelProvider;
import alma.obops.tmcdbgui.views.providers.AntennaToPadRow;
import alma.tmcdb.domain.AntennaToPad;

/**
 * @author sharring
 *
 */
public class HistoricalAntennaToPadEditor extends EditorPart 
{
	public static final String ID = "historical-antennatopad.editor";
	
	private AntennaToPad referenceAntennaToPad;
	private AntennaToPad historicalAntennaToPad;
	private TableViewer a2pViewer;

	@Override
	public void doSave(IProgressMonitor monitor) {
		// noop
	}
	
	@Override
	public void doSaveAs() {
		// noop - not allowed
	}
	
	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		HistoricalAntennaToPadEditorInput a2pEdInput = (HistoricalAntennaToPadEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(a2pEdInput.getName());
		historicalAntennaToPad = a2pEdInput.getReferenceAntennaToPad();
		referenceAntennaToPad = a2pEdInput.getPreviousAntennaToPad();
		if(null != a2pViewer) {
			a2pViewer.setInput(populateRows()); // trigger a content reload
		}
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		Composite editorComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		editorComposite.setLayout(gridLayout);
		gridLayout.numColumns = 1;
		createAntennaToPadGroup(editorComposite);
	}
	
	private void createAntennaToPadGroup(Composite editorComposite) 
	{
		Group a2pPositionTableGroup = new Group(editorComposite, SWT.BORDER);
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = false;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.BEGINNING;
		a2pPositionTableGroup.setLayoutData(gdata);
		a2pPositionTableGroup.setLayout(new FillLayout());
		
		a2pViewer = new TableViewer(a2pPositionTableGroup, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Metrology coefficient", "value"};
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(a2pViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.getColumn().pack();
    	}
    	Table table = a2pViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

		a2pViewer.setSorter(new ViewerSorter());
    	a2pViewer.setContentProvider( new HistoricalAntennaToPadContentsProvider() );
    	a2pViewer.setLabelProvider( new HistoricalAntennaToPadLabelProvider() );
    	a2pViewer.setInput(populateRows()); // trigger a content reload
	}

	private AntennaToPadRow[] populateRows() 
	{
		AntennaToPadRow[] retVal = new AntennaToPadRow[2];
		
		retVal[0] = new AntennaToPadRow();
		retVal[0].setCoeffName(AntennaToPadEditor.AN0);
		retVal[0].setCoeffValue(historicalAntennaToPad.getMountMetrologyAN0Coeff());
		
		retVal[1] = new AntennaToPadRow();
		retVal[1].setCoeffName(AntennaToPadEditor.AW0);
		retVal[1].setCoeffValue(historicalAntennaToPad.getMountMetrologyAW0Coeff());
		
		// NOTE: There are essentially two modes for this editor, one for displaying a 
		// historical a2p as it existed at some time in the past, and another
		// for showing the differences between 2 versions of an a2p. 
		// When we are in 'diff' mode the historical and reference a2p 
		// variables will not be identical; else they will be the same. 
		// If they are the same, we merely show the a2p w/o any highlighting; 
		// whereas if they are different, we highlight the differences between them.
		if(referenceAntennaToPad != historicalAntennaToPad)
		{
			retVal = diffAntennaToPads(retVal);
		}
		
		return retVal;
	}

	private AntennaToPadRow[] diffAntennaToPads(AntennaToPadRow[] rows) 
	{
		// AN0
		AntennaToPadRow an0Row = rows[0];
	    Double an0Val = an0Row.getCoeffValue();
	    Double referenceAn0Val = referenceAntennaToPad.getMountMetrologyAN0Coeff();
	    
		if(!an0Val.equals(referenceAn0Val)) {
			an0Row.setAn0Image(AntennaToPadRow.CHANGED_IMAGE);
			an0Row.setAn0Font(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			an0Row.setAn0Image(null);
			an0Row.setAn0Font(null);
		}
		
		// AW0
		AntennaToPadRow aw0Row = rows[1];
	    Double aw0Val = aw0Row.getCoeffValue();
	    Double referenceAw0Val = referenceAntennaToPad.getMountMetrologyAW0Coeff();
	    
		if(!aw0Val.equals(referenceAw0Val)) {
			aw0Row.setAw0Image(AntennaToPadRow.CHANGED_IMAGE);
			aw0Row.setAw0Font(JFaceResources.getFontRegistry().getBold(
					JFaceResources.DEFAULT_FONT));
		}
		else {
			aw0Row.setAw0Image(null);
			aw0Row.setAw0Font(null);
		}
		
		return rows;
	}

	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void setFocus() {
	}
	
	private static class HistoricalAntennaToPadContentsProvider extends AntennaToPadContentsProvider
	{
		// currently this is identical to base class (AntennaToPadContentsProvider),
		// so we could have used it directly but leaving this here just in case...
	}
	
	private static class HistoricalAntennaToPadLabelProvider extends AntennaToPadLabelProvider implements ITableFontProvider
	{
		@Override
		public Image getColumnImage(Object element, int columnIndex) {
			Image retVal = null;
			
			if( !(element instanceof AntennaToPadRow) ) 
			{
				retVal = null;
			}
			else 
			{
				AntennaToPadRow row = (AntennaToPadRow)element;
				switch(columnIndex) 
				{
				case 0:
					if(row.getCoeffName().equalsIgnoreCase("AN0")) {
						retVal = row.getAn0Image();
					} else if(row.getCoeffName().equalsIgnoreCase("AW0")) {
						retVal = row.getAw0Image();
					}
					break;
				case 1:
				default:
					retVal = null;
					break;
				}
			}

			return retVal;
		}

		@Override
		public Font getFont(Object element, int columnIndex) {
			Font retVal = null;

			if( !(element instanceof AntennaToPadRow) ) 
			{
				retVal = null;
			}
			else 
			{
				AntennaToPadRow row = (AntennaToPadRow)element;
				switch(columnIndex) 
				{
				case 1:
					if(row.getCoeffName().equalsIgnoreCase("AN0")) {
						retVal = row.getAn0Font();
					} else if(row.getCoeffName().equalsIgnoreCase("AW0")) {
						retVal = row.getAw0Font();
					}
					break;
				case 0:
				default:
					retVal = null;
					break;
				}
			}
			return retVal;
		}
	}
}
