/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.obops.tmcdbgui.editors.inputs.HistoricalAntennaEditorInput;
import alma.obops.tmcdbgui.widgets.AntennaAttributesComposite;
import alma.tmcdb.domain.Antenna;

/**
 * @author sharring
 *
 */
public class HistoricalAntennaEditor extends EditorPart 
{
	public static final String ID = "historical-antenna.editor";
	
	private Antenna previousAntenna;
	private Antenna historicalAntenna;
	//private TableViewer antennaViewer;
	private AntennaAttributesComposite antennaComposite;

	@Override
	public void doSave(IProgressMonitor monitor) {
		// noop
	}
	
	@Override
	public void doSaveAs() {
		// noop - not allowed
	}
	
	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		HistoricalAntennaEditorInput antennaEdInput = (HistoricalAntennaEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(antennaEdInput.getName());
		historicalAntenna = antennaEdInput.getReferenceAntenna();
		previousAntenna = antennaEdInput.getPreviousAntenna();
	}

	private void diffAntennas() 
	{
		antennaComposite.setAntenna(historicalAntenna);
		
		if((historicalAntenna.getCaiAca() != null && previousAntenna.getCaiAca() != null) && 
		   !historicalAntenna.getCaiAca().equals(previousAntenna.getCaiAca())) 
		{
			// hilite the cai aca field in the antenna composite
			antennaComposite.emphasizeCaiAca();
		}
		else if(historicalAntenna.getCaiAca() == null && previousAntenna.getCaiAca() != null) {
			// hilite the cai aca field in the antenna composite
			antennaComposite.emphasizeCaiAca();
		}
		else if(historicalAntenna.getCaiAca() != null && previousAntenna.getCaiAca() == null) {
			// hilite the cai aca field in the antenna composite
			antennaComposite.emphasizeCaiAca();
		}
		
		if((historicalAntenna.getCaiBaseline() != null && previousAntenna.getCaiBaseline() != null) && 
				!historicalAntenna.getCaiBaseline().equals(previousAntenna.getCaiBaseline())) 
		{
			// hilite the cai baseline field in the antenna composite
			antennaComposite.emphasizeCaiBaseline();
		}
		else if(historicalAntenna.getCaiBaseline() == null && previousAntenna.getCaiBaseline() != null) {
			// hilite the cai baseline field in the antenna composite
			antennaComposite.emphasizeCaiBaseline();
		}
		else if(historicalAntenna.getCaiBaseline() != null && previousAntenna.getCaiBaseline() == null) {
			// hilite the cai baseline field in the antenna composite
			antennaComposite.emphasizeCaiBaseline();
		}
		
		if(!historicalAntenna.getCommissionDate().equals(previousAntenna.getCommissionDate())) {
			// hilite the offset in the antenna composite
			antennaComposite.emphasizeCommissionDate();
		}
		if(!historicalAntenna.getOffset().equals(previousAntenna.getOffset())) {
			// hilite the offset in the antenna composite
			antennaComposite.emphasizeOffset();
		}
		if(!historicalAntenna.getWalshSeq().equals(previousAntenna.getWalshSeq())) {
			// hilite the walsh seq in the antenna composite
			antennaComposite.emphasizeWalshSequence();
		}
		if(!historicalAntenna.getLoOffsettingIndex().equals(previousAntenna.getLoOffsettingIndex())) {
			// hilite the looffsetting index in the antenna composite
			antennaComposite.emphasizeLoOffsetting();
		}
		if(historicalAntenna.getPosition().getX() != previousAntenna.getPosition().getX()) {
			// hilite the X position in the antenna composite
			antennaComposite.emphasizePositionX();
		}
		if(historicalAntenna.getPosition().getY() != previousAntenna.getPosition().getY()) {
			// hilite the Y position in the antenna composite
			antennaComposite.emphasizePositionY();
		}
		if(historicalAntenna.getPosition().getZ()  != previousAntenna.getPosition().getZ()) {
			// hilite the Y position in the antenna composite
			antennaComposite.emphasizePositionZ();
		}
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		Composite editorComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		editorComposite.setLayout(gridLayout);
		gridLayout.numColumns = 1;
		antennaComposite = new AntennaAttributesComposite(editorComposite, SWT.NONE);
		antennaComposite.makeReadOnly();
		
		// NOTE: There are essentially two modes for this editor, one for displaying a 
		// historical antenna as it existed at some time in the past, and another
		// for showing the differences between 2 versions of a antenna. 
		// When we are in 'diff' mode the historical and reference antenna 
		// variables will not be identical; else they will be the same. 
		// If they are the same, we merely show the antenna w/o any highlighting; 
		// whereas if they are different, we highlight the differences between them.
		if(null != antennaComposite && (historicalAntenna != previousAntenna)) 
		{
			diffAntennas();
		} else if(null != antennaComposite) {
			antennaComposite.setAntenna(historicalAntenna);
		}
	}
	
	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void setFocus() {
	}
}
