/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.viewers.ITableFontProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.BasebandNameMod.BasebandName;
import alma.obops.tmcdbgui.editors.inputs.HistoricalAcaCorrDelaysEditorInput;
import alma.obops.tmcdbgui.views.providers.AcaCorrDelaysContentsProvider;
import alma.obops.tmcdbgui.views.providers.AcaCorrDelaysLabelProvider;
import alma.obops.tmcdbgui.views.providers.AcaCorrDelaysRow;
import alma.tmcdb.domain.AcaCorrDelays;

/**
 * @author sharring
 *
 */
public class HistoricalAcaCorrDelaysEditor extends EditorPart 
{
	public static final String ID = "historical-acacorrdelays.editor";
	
	private AcaCorrDelays referenceAcaCorrDelays;
	private AcaCorrDelays historicalAcaCorrDelays;
	private TableViewer acaCorrDelaysViewer;

	@Override
	public void doSave(IProgressMonitor monitor) {
		// noop
	}
	
	@Override
	public void doSaveAs() {
		// noop - not allowed
	}
	
	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		HistoricalAcaCorrDelaysEditorInput acaCorrDelaysEdInput = (HistoricalAcaCorrDelaysEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(acaCorrDelaysEdInput.getName());
		historicalAcaCorrDelays = acaCorrDelaysEdInput.getReferenceAcaCorrDelays();
		referenceAcaCorrDelays = acaCorrDelaysEdInput.getPreviousAcaCorrDelays();
		if(null != acaCorrDelaysViewer) {
			acaCorrDelaysViewer.setInput(populateRows()); // trigger a content reload
		}
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		Composite editorComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		editorComposite.setLayout(gridLayout);
		gridLayout.numColumns = 1;
		createAcaCorrDelaysGroup(editorComposite);
	}
	
	private void createAcaCorrDelaysGroup(Composite editorComposite) 
	{
		Group acaCorrDelaysPositionTableGroup = new Group(editorComposite, SWT.BORDER);
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = false;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.BEGINNING;
		acaCorrDelaysPositionTableGroup.setLayoutData(gdata);
		acaCorrDelaysPositionTableGroup.setLayout(new FillLayout());
		
		acaCorrDelaysViewer = new TableViewer(acaCorrDelaysPositionTableGroup, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Baseband", "Delay (s)"};
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(acaCorrDelaysViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.getColumn().pack();
    	}
    	Table table = acaCorrDelaysViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

		acaCorrDelaysViewer.setSorter(new ViewerSorter());
    	acaCorrDelaysViewer.setContentProvider( new AcaCorrDelaysContentsProvider() );
    	acaCorrDelaysViewer.setLabelProvider( new HistoricalAcaCorrDelaysLabelProvider() );
    	acaCorrDelaysViewer.setInput(populateRows()); // trigger a content reload
	}

	private AcaCorrDelaysRow[] populateRows() 
	{
		AcaCorrDelaysRow[] retVal = new AcaCorrDelaysRow[4];
		
		retVal[0] = new AcaCorrDelaysRow();
		retVal[0].setBaseband(BasebandName.BB_1);
		retVal[0].setDelay(historicalAcaCorrDelays.getDelayBbOne());
		
		retVal[1] = new AcaCorrDelaysRow();
		retVal[1].setBaseband(BasebandName.BB_2);
		retVal[1].setDelay(historicalAcaCorrDelays.getDelayBbTwo());
		
		retVal[2] = new AcaCorrDelaysRow();
		retVal[2].setBaseband(BasebandName.BB_3);
		retVal[2].setDelay(historicalAcaCorrDelays.getDelayBbThree());
		
		retVal[3] = new AcaCorrDelaysRow();
		retVal[3].setBaseband(BasebandName.BB_4);
		retVal[3].setDelay(historicalAcaCorrDelays.getDelayBbFour());
		
		// NOTE: There are essentially two modes for this editor, one for displaying a 
		// historical acaCorrDelays as it existed at some time in the past, and another
		// for showing the differences between 2 versions of an acaCorrDelays. 
		// When we are in 'diff' mode the historical and reference acaCorrDelays 
		// variables will not be identical; else they will be the same. 
		// If they are the same, we merely show the acaCorrDelays w/o any highlighting; 
		// whereas if they are different, we highlight the differences between them.
		if(referenceAcaCorrDelays != historicalAcaCorrDelays)
		{
			retVal = diffAcaCorrDelays(retVal);
		}
		
		return retVal;
	}

	private AcaCorrDelaysRow[] diffAcaCorrDelays(AcaCorrDelaysRow[] rows) 
	{
		AcaCorrDelaysRow row0 = rows[0];
	    Double bb1Del = row0.getDelay();
	    Double referenceBb1Del = referenceAcaCorrDelays.getDelayBbOne();
	    
		if(!bb1Del.equals(referenceBb1Del)) {
			row0.setImage(AcaCorrDelaysRow.CHANGED_IMAGE);
			row0.setFont(JFaceResources.getFontRegistry().getBold(JFaceResources.DEFAULT_FONT));
		}
		else {
			row0.setImage(null);
			row0.setFont(null);
		}
		
		AcaCorrDelaysRow row1 = rows[1];
	    Double bb2Del = row1.getDelay();
	    Double referenceBb2Del = referenceAcaCorrDelays.getDelayBbTwo();
	    
		if(!bb2Del.equals(referenceBb2Del)) {
			row1.setImage(AcaCorrDelaysRow.CHANGED_IMAGE);
			row1.setFont(JFaceResources.getFontRegistry().getBold(JFaceResources.DEFAULT_FONT));
		}
		else {
			row1.setImage(null);
			row1.setFont(null);
		}
		
		AcaCorrDelaysRow row2 = rows[2];
	    Double bb3Del = row2.getDelay();
	    Double referenceBb3Del = referenceAcaCorrDelays.getDelayBbThree();
	    
		if(!bb3Del.equals(referenceBb3Del)) {
			row2.setImage(AcaCorrDelaysRow.CHANGED_IMAGE);
			row2.setFont(JFaceResources.getFontRegistry().getBold(JFaceResources.DEFAULT_FONT));
		}
		else {
			row2.setImage(null);
			row2.setFont(null);
		}
		
		AcaCorrDelaysRow row3 = rows[3];
	    Double bb4Del = row3.getDelay();
	    Double referenceBb4Del = referenceAcaCorrDelays.getDelayBbFour();
	    
		if(!bb4Del.equals(referenceBb4Del)) {
			row3.setImage(AcaCorrDelaysRow.CHANGED_IMAGE);
			row3.setFont(JFaceResources.getFontRegistry().getBold(JFaceResources.DEFAULT_FONT));
		}
		else {
			row3.setImage(null);
			row3.setFont(null);
		}
		
		return rows;
	}

	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void setFocus() {
	}
	
	private static class HistoricalAcaCorrDelaysLabelProvider extends AcaCorrDelaysLabelProvider implements ITableFontProvider
	{
		@Override
		public Image getColumnImage(Object element, int columnIndex) {
			Image retVal = null;
			
			if( !(element instanceof AcaCorrDelaysRow) ) 
			{
				retVal = null;
			}
			else 
			{
				AcaCorrDelaysRow row = (AcaCorrDelaysRow)element;
				switch(columnIndex) 
				{
				case 0:
					retVal = row.getImage();
					break;
				case 1:
				default:
					retVal = null;
					break;
				}
			}

			return retVal;
		}

		@Override
		public Font getFont(Object element, int columnIndex) {
			Font retVal = null;

			if( !(element instanceof AcaCorrDelaysRow) ) 
			{
				retVal = null;
			}
			else 
			{
				AcaCorrDelaysRow row = (AcaCorrDelaysRow)element;
				switch(columnIndex) 
				{
				case 1:
					retVal = row.getFont();
					break;
				case 0:
				default:
					retVal = null;
					break;
				}
			}
			return retVal;
		}
	}
}