/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;

import alma.ReceiverBandMod.ReceiverBand;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdbgui.editors.inputs.FocusModelEditorInput;
import alma.obops.tmcdbgui.editors.inputs.FocusModelHistoryEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.conversation.BackendConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.FocusModelConversationUtils;
import alma.obops.tmcdbgui.views.providers.FocusModelContentsProvider;
import alma.obops.tmcdbgui.views.providers.FocusModelEditingSupport;
import alma.obops.tmcdbgui.views.providers.FocusModelLabelProvider;
import alma.obops.tmcdbgui.views.providers.FocusModelRow;
import alma.obops.tmcdbgui.views.providers.IFocusModelTermUpdateable;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.tmcdb.domain.FocusModel;
import alma.tmcdb.domain.FocusModelCoeff;

/**
 * Editor for Focus model
 * @author sharring
 */
public class FocusModelEditor extends TmcdbObjectEditorPart implements DirtyListener, IFocusModelTermUpdateable
{
	private boolean dirty = false;
	private FocusModel focusModelCopy;
	private FocusModel focusModel;
	private Map<String, FocusModelCoeff> coefficientsToRemove = new HashMap<String, FocusModelCoeff>();
	
	private TableViewer focusModelViewer;
	public static final String ID = "focusmodel.editor";

	// utility method to find a coeff (using id, in case the key was changed) in a focus model
	private Entry<String, FocusModelCoeff> findMatchingEntry(String coeffName, FocusModelCoeff coeff, FocusModel focusModelToSearch) 
	{
		Entry<String, FocusModelCoeff> retVal = null;

		for(Entry<String, FocusModelCoeff> entry : focusModelToSearch.getTerms().entrySet())
		{
			if(entry.getValue().getId() != null && null != coeff.getId() && entry.getValue().getId().equals(coeff.getId())) 
			{
				retVal = entry;
				break;
			} 
			else if(coeff.getId() == null && coeffName.equals(entry.getKey())) {
				retVal = entry;
				break;
			}
		}
		
		return retVal;
	}

	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		InputDialog descriptionInputDialog = new InputDialog(this.getSite().getShell(), "Description", "Please add any comments about your change", "", null);
		if(descriptionInputDialog.open() != Window.OK) 
		{
			return;
		}
		
		try 
		{
			// try to create a new version
			String description = descriptionInputDialog.getValue();
			String userId = System.getProperty("user.name");
			boolean canSave = FocusModelConversationUtils.getInstance().
			prepareFocusModelSave(focusModel, userId, description);

			// if the new version preparation was successful, we can then perform the save
			if(canSave) 
			{			
				// delete coefficients that were removed
				for(Entry<String, FocusModelCoeff> entryToRemove : this.coefficientsToRemove.entrySet()) 
				{
					Entry<String, FocusModelCoeff> entryToDelete = findMatchingEntry(entryToRemove.getKey(), entryToRemove.getValue(), focusModel);
					BackendConversationUtils.getInstance().delete(entryToDelete.getValue(), ConversationToken.CONVERSATION_PENDING, true);
					focusModel.getTerms().remove(entryToDelete.getKey());
				}
				if(coefficientsToRemove.size() > 0) {
					FocusModelConversationUtils.getInstance().saveOrUpdateFocusModel(focusModel, ConversationToken.CONVERSATION_PENDING);
				}
				coefficientsToRemove.clear();

				// get the edited values from the tableviewer
				FocusModelRow[] rows = (FocusModelRow[]) focusModelViewer.getInput();

				// for each row in our editor
				for(FocusModelRow row : rows)
				{
					// find the matching entry in the focus model to be saved; this is done by id first
					// then by name (if id is null e.g. for new coefficients).
					Entry<String, FocusModelCoeff> matchingEntry = findMatchingEntry(row.getCoeffName(), row.getCoeff(), focusModel);

					// for the matching entry, determine if its name was changed. if it was, we have
					// to remove + re-add it to the map because the name is the key of the map.
					if(null != matchingEntry)
					{
						// we found a matching entry; we will copy over the value & offsets from the edited (copy) focus model
						if(matchingEntry.getValue().getValue() != (row.getCoeffValue())) {
							matchingEntry.getValue().setValue(row.getCoeffValue());
						}
						
						Double matchingEntryOffSetVal = matchingEntry.getValue().getOffsets().get(ReceiverBand.ALMA_RB_01);
						if( matchingEntryOffSetVal == null || !matchingEntryOffSetVal.equals(row.getOffset1()))  
						{
							matchingEntry.getValue().getOffsets().put(ReceiverBand.ALMA_RB_01, row.getOffset1());
						}
						
						matchingEntryOffSetVal = matchingEntry.getValue().getOffsets().get(ReceiverBand.ALMA_RB_02);
						if( matchingEntryOffSetVal == null || !matchingEntryOffSetVal.equals(row.getOffset2()))  
						{
							matchingEntry.getValue().getOffsets().put(ReceiverBand.ALMA_RB_02, row.getOffset2());
						}
						
						matchingEntryOffSetVal = matchingEntry.getValue().getOffsets().get(ReceiverBand.ALMA_RB_03);
						if( matchingEntryOffSetVal == null || !matchingEntryOffSetVal.equals(row.getOffset3()))  
						{
							matchingEntry.getValue().getOffsets().put(ReceiverBand.ALMA_RB_03, row.getOffset3());
						}
						
						matchingEntryOffSetVal = matchingEntry.getValue().getOffsets().get(ReceiverBand.ALMA_RB_04);
						if( matchingEntryOffSetVal == null || !matchingEntryOffSetVal.equals(row.getOffset4()))   
						{
							matchingEntry.getValue().getOffsets().put(ReceiverBand.ALMA_RB_04, row.getOffset4());
						}
						
						matchingEntryOffSetVal = matchingEntry.getValue().getOffsets().get(ReceiverBand.ALMA_RB_05);
						if( matchingEntryOffSetVal == null || !matchingEntryOffSetVal.equals(row.getOffset5()))   
						{
							matchingEntry.getValue().getOffsets().put(ReceiverBand.ALMA_RB_05, row.getOffset5());
						}
						
						matchingEntryOffSetVal = matchingEntry.getValue().getOffsets().get(ReceiverBand.ALMA_RB_06);
						if( matchingEntryOffSetVal == null || !matchingEntryOffSetVal.equals(row.getOffset6())) 
						{
							matchingEntry.getValue().getOffsets().put(ReceiverBand.ALMA_RB_06, row.getOffset6());
						}
						
						matchingEntryOffSetVal = matchingEntry.getValue().getOffsets().get(ReceiverBand.ALMA_RB_07);
						if( matchingEntryOffSetVal == null || !matchingEntryOffSetVal.equals(row.getOffset7()))  
						{
							matchingEntry.getValue().getOffsets().put(ReceiverBand.ALMA_RB_07, row.getOffset7());
						}
						
						matchingEntryOffSetVal = matchingEntry.getValue().getOffsets().get(ReceiverBand.ALMA_RB_08);
						if( matchingEntryOffSetVal == null || !matchingEntryOffSetVal.equals(row.getOffset8()))   
						{
							matchingEntry.getValue().getOffsets().put(ReceiverBand.ALMA_RB_08, row.getOffset8());
						}
						
						matchingEntryOffSetVal = matchingEntry.getValue().getOffsets().get(ReceiverBand.ALMA_RB_09);
						if( matchingEntryOffSetVal == null || !matchingEntryOffSetVal.equals(row.getOffset9()))   
						{
							matchingEntry.getValue().getOffsets().put(ReceiverBand.ALMA_RB_09, row.getOffset9());
						}

						matchingEntryOffSetVal = matchingEntry.getValue().getOffsets().get(ReceiverBand.ALMA_RB_10);
						if( matchingEntryOffSetVal == null || !matchingEntryOffSetVal.equals(row.getOffset10()))  
						{
							matchingEntry.getValue().getOffsets().put(ReceiverBand.ALMA_RB_10, row.getOffset10());
						}
					}
					else {
						// a new row was added
						FocusModelCoeff coeff = row.getCoeff();
						focusModel.getTerms().put(row.getCoeffName(), coeff);
					}
				}

				// perform the save
				FocusModelConversationUtils.getInstance().saveOrUpdateFocusModel(focusModel);

				// set the dirty flag to false, as we have just saved
				this.setDirty(false);
			}
			else 
			{
				MessageDialog.openWarning(this.getSite().getShell(), "Unable to save", "Could not save; perhaps someone else is saving now. Try again later.");
			}
		}
		catch(Exception ex) 
		{
			ex.printStackTrace();
			throw new RuntimeException("Could not save focus model", ex);
		} 
		finally {
			try {
				FocusModelConversationUtils.getInstance().endFocusModelSave(focusModel);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		FocusModelEditorInput focusEdInput = (FocusModelEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(focusEdInput.getName());
		focusModel = focusEdInput.getFocusModel();
		makeFocusModelCopy();
		if(null != focusModelViewer) {
			focusModelViewer.setInput(populateRows()); // trigger a content reload
		}
	}

	private void makeFocusModelCopy() 
	{
		focusModelCopy = new FocusModel();
		// for each coefficient in the original, make a copy 
		for(Entry<String, FocusModelCoeff> coeffEntry : focusModel.getTerms().entrySet())
		{
			FocusModelCoeff copyCoeff = new FocusModelCoeff(coeffEntry.getKey(), coeffEntry.getValue().getValue());
			copyCoeff.setId(coeffEntry.getValue().getId());
			for(Entry<ReceiverBand, Double> offsetEntry : coeffEntry.getValue().getOffsets().entrySet())
			{
				// for each offset in the original coeff, make a copy
				Double copyOffsetValue = new Double(offsetEntry.getValue());
				copyCoeff.getOffsets().put(offsetEntry.getKey(), copyOffsetValue);
			}
			focusModelCopy.getTerms().put(coeffEntry.getKey(), copyCoeff);
		}
	}

	@Override
	public void createPartControl(final Composite parent) 
	{
		final Composite editorComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		editorComposite.setLayout(gridLayout);
		gridLayout.numColumns = 1;
		
		Composite tableComposite = new Composite(editorComposite, SWT.NONE);
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = true;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.FILL;
		tableComposite.setLayoutData(gdata);
		tableComposite.setLayout(new FillLayout());
		
		focusModelViewer = new TableViewer(tableComposite, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Coefficient", "Value        ", "b1 offset", "b2 offset", "b3 offset", "b4 offset", "b5 offset", "b6 offset", "b7 offset", "b8 offset", "b9 offset", "b10 offset" };
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(focusModelViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.setEditingSupport(new FocusModelEditingSupport(focusModelViewer, i, this, this));
    		col.getColumn().pack();
    	}
    	Table table = focusModelViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

    	focusModelViewer.setSorter(new ViewerSorter());
    	focusModelViewer.setContentProvider( new FocusModelContentsProvider() );
    	focusModelViewer.setLabelProvider( new FocusModelLabelProvider() );

    	focusModelViewer.setInput(populateRows()); // trigger a content reload
    	
    	Composite buttonComposite = new Composite(editorComposite, SWT.NONE);
    	GridData gridData = new GridData();
    	gridData.grabExcessHorizontalSpace = true;
    	gridData.grabExcessVerticalSpace = false;
    	gridData.horizontalAlignment = SWT.FILL;
    	buttonComposite.setLayoutData(gridData);
    	
    	GridLayout glayout = new GridLayout();
    	glayout.numColumns = 5;
    	glayout.makeColumnsEqualWidth = false;
    	buttonComposite.setLayout(glayout);

    	Button addButton = new Button(buttonComposite, SWT.PUSH | SWT.CENTER);
    	addButton.setText("Add");

    	addButton.addSelectionListener(new SelectionAdapter() {
    		public void widgetSelected(SelectionEvent e) {
    			InputDialog newRowCoeffNameDialog = new InputDialog(getSite().getShell(), "Coefficient name", "Please enter the name of the new coefficient", "", null);
    			if(newRowCoeffNameDialog.open() != Window.OK) 
    			{
    				return;
    			}
    			addRow(newRowCoeffNameDialog.getValue());
    		}
    	});
    	
    	Button deleteButton = new Button(buttonComposite, SWT.PUSH | SWT.CENTER);
    	deleteButton.setText("Delete");

    	deleteButton.addSelectionListener(new SelectionAdapter() {
    		public void widgetSelected(SelectionEvent e) {
    			deleteRow();
    		}
    	});
    	
        //   Create and configure the "history" button
        Button historyButton = new Button(buttonComposite, SWT.PUSH | SWT.CENTER);
        historyButton.setText("History");
        
        historyButton.addSelectionListener(new SelectionAdapter() 
        {
        	public void widgetSelected(SelectionEvent e) 
            {
         	   FocusModelHistoryEditorInput editorInput =  new FocusModelHistoryEditorInput(focusModel);
         	   IWorkbenchWindow win = getSite().getWorkbenchWindow();
         	   try {
         		   win.getActivePage().openEditor(editorInput, FocusModelHistoryEditor.ID);
         	   } 
         	   catch (PartInitException e1) {
         		   throw new RuntimeException("Could not open focus model history editor", e1);
         	   }
            }
        });
        
        //   Create and configure the "units/info" button
        Button helpButton = new Button(buttonComposite, SWT.PUSH | SWT.CENTER);
        helpButton.setText("Units / info");
        
        helpButton.addSelectionListener(new SelectionAdapter() 
        {
        	public void widgetSelected(SelectionEvent e) 
            {
        		MessageDialog.openInformation(getSite().getShell(), "Focus Model Units", "The following units are used: \n\n" +
        				"\tALPHA, BETA:\tradians \n "+
                        "\t{X,Y,Z}TA:\tmeters/degree \n" +
                        "\t{X,Y,Z}?:\tmeters\n\n");
            }
        });
	}
	
	private void deleteRow()
	{
		IStructuredSelection structuredSelection = (IStructuredSelection) focusModelViewer.getSelection();
		FocusModelRow selectedRow = (FocusModelRow)structuredSelection.getFirstElement();
		if(selectedRow != null)
		{
			Entry<String, FocusModelCoeff> selectedEntry = findMatchingEntry(selectedRow.getCoeffName(), selectedRow.getCoeff(), focusModelCopy);
			focusModelCopy.getTerms().remove(selectedEntry.getKey());
			this.coefficientsToRemove.put(selectedEntry.getKey(), selectedEntry.getValue());
			setDirty(true);
			focusModelViewer.setInput(populateRows()); // trigger a content reload
		}
	}
	
	private void addRow(String newCoeffName) 
	{
		FocusModelCoeff coeff = new FocusModelCoeff();
		
		// for bookkeeping, we make up a fake (negative) id
		// this will later (upon save) get overwritten to null
		// so that hibernate can generate a real id upon insert
		coeff.setId(null);
		
		coeff.getOffsets().put(ReceiverBand.ALMA_RB_01, 0d);
		coeff.getOffsets().put(ReceiverBand.ALMA_RB_02, 0d);
		coeff.getOffsets().put(ReceiverBand.ALMA_RB_03, 0d);
		coeff.getOffsets().put(ReceiverBand.ALMA_RB_04, 0d);
		coeff.getOffsets().put(ReceiverBand.ALMA_RB_05, 0d);
		coeff.getOffsets().put(ReceiverBand.ALMA_RB_06, 0d);
		coeff.getOffsets().put(ReceiverBand.ALMA_RB_07, 0d);
		coeff.getOffsets().put(ReceiverBand.ALMA_RB_08, 0d);
		coeff.getOffsets().put(ReceiverBand.ALMA_RB_09, 0d);
		coeff.getOffsets().put(ReceiverBand.ALMA_RB_10, 0d);
		focusModelCopy.getTerms().put(newCoeffName, coeff);
		setDirty(true);
		
		focusModelViewer.setInput(populateRows()); // trigger a content reload
	}

	   
	private FocusModelRow[] populateRows() 
	{
		FocusModelRow[] retVal = new FocusModelRow[focusModelCopy.getTerms().size()];
		
		int count = 0;
		for(Entry<String, FocusModelCoeff> entry : focusModelCopy.getTerms().entrySet()) 
		{
			retVal[count] = new FocusModelRow(focusModelCopy.getAntenna(), entry.getKey(), entry.getValue());
			count++;
		}
		
		return retVal;
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean d) {
		this.dirty = d;
		firePropertyChange(PROP_DIRTY);
	}

	@Override
	public void updateFocusModelCoeffName(String oldCoeffName, String newCoeffName) 
	{
		FocusModelCoeff coeffToModify = this.focusModelCopy.getTerms().remove(oldCoeffName);
		this.focusModelCopy.getTerms().put(newCoeffName, coeffToModify);
	}
}
