/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) NRAO - National Radio Astronomy Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/

package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.acs.tmcdb.DomainsMapping;
import alma.acscommon.ACS_NC_DOMAIN_ALARMSYSTEM;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.editors.inputs.DomainsMappingEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.handlers.NewChannelMappingAction;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.DomainsMappingConversationUtils;
import alma.obops.tmcdbgui.views.SoftwareDeploymentView;

public class DomainsMappingEditor extends TmcdbObjectEditor 
{
	private static final String ALARM_NOTIFY_EVENT_CHANNEL_FACTORY = "AlarmNotifyEventChannelFactory";
	public static final String ID = "domainsmapping.editor";
	private DomainsMapping domainsMapping;
	private DomainsMapping origDomainsMapping;
	private Text nameText;
	private Combo notificationServiceCombo;

	@Override
	public void setFocus() {
		nameText.setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		boolean isNewOption = false;

		if( domainsMapping.getDomainsMappingId() == null ) {
			isNewOption = true;
		}

		// Check for invalid inputs
		if(  (domainsMapping.getName() == null || domainsMapping.getName().toString().trim().equals(""))  ||
			   (domainsMapping.getNotificationService() == null || domainsMapping.getNotificationService().trim().equals("")) )
		{
			MessageDialog.openInformation(getSite().getShell(),
					"Please specify all fields",
					"NS Domain Mapping cannot be saved without all fields defined");
			return;
		}

		// Persist the object
		try {
			DomainsMappingConversationUtils.getInstance().saveOrUpdateDomainsMapping(domainsMapping);
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(getSite().getShell(), "Cannot save NC Domains Mapping", "Error while saving DomainsMapping: " + domainsMapping.getName());
			return;
		}

		setEditedObjectAsOriginalContent();
		setDirty(false);

		// If we're adding a new containerstartupoption, let's refresh the SDV if available
		if( isNewOption ) {
			SoftwareDeploymentView sdv = (SoftwareDeploymentView)RcpUtils.findView( SoftwareDeploymentView.ID );
			sdv.internalModelChange();
		}
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		ScrolledComposite sc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.BORDER);
		sc.setExpandHorizontal(true);
		sc.setExpandVertical(true);

		Composite composite = new Composite(sc, SWT.NONE);
		composite.setLayout(new GridLayout(2, false));

		/* Name */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label nameLabel = new Label(composite, SWT.NONE);
		nameLabel.setText("Name");
		nameLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.grabExcessHorizontalSpace = true;
		gd.horizontalAlignment = SWT.FILL;
		nameText = new Text(composite, SWT.BORDER);
		nameText.setLayoutData(gd);
		
		nameText.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent e) {
				enableOrDisableNsCombo();
			}
		});

		/* Notification Service */
		createNotificationServiceWidgets(composite);
		
		// Data binding
		bind( "name", nameText );
		bind( "notificationService", notificationServiceCombo );
		enableOrDisableNsCombo();
		
		sc.setContent(composite);
	}
	
	private void createNotificationServiceWidgets(Composite composite) 
	{
		Label label3 = new Label(composite, SWT.NONE);
		label3.setText("Select notification service");
		GridData gd7 = new GridData();
		gd7.grabExcessHorizontalSpace = false;
		gd7.grabExcessVerticalSpace = false;
		gd7.horizontalAlignment = SWT.LEFT;
		gd7.verticalAlignment = SWT.CENTER;
		label3.setLayoutData(gd7);
		
		notificationServiceCombo = new Combo(composite, SWT.DROP_DOWN | SWT.READ_ONLY);
		GridData gd8 = new GridData();
		gd8.grabExcessHorizontalSpace = true;
		gd8.grabExcessVerticalSpace = false;
		gd8.horizontalSpan = 1;
		gd8.horizontalAlignment = SWT.FILL;
		notificationServiceCombo.setLayoutData(gd8);
		
		notificationServiceCombo.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
		});
		populateNsComb();
	}

	private void populateNsComb() {
		String[] nsStrings = NewChannelMappingAction.getNotificationServiceStrings(domainsMapping.
				getNotificationServiceMapping().getConfiguration());
		String[] nsStringsPlusAlarmChannel = new String[nsStrings.length + 1];
		nsStringsPlusAlarmChannel[0] = ALARM_NOTIFY_EVENT_CHANNEL_FACTORY;
		int count = 1;
		for(String str : nsStrings) {
			nsStringsPlusAlarmChannel[count++] = str;
		}
		
		notificationServiceCombo.setItems(nsStringsPlusAlarmChannel);
	}

	private void enableOrDisableNsCombo()
	{
		if(nameText.getText() != null && nameText.getText().trim().equals(ACS_NC_DOMAIN_ALARMSYSTEM.value)) 
		{
			notificationServiceCombo.setEnabled(false);
			notificationServiceCombo.select(notificationServiceCombo.indexOf(ALARM_NOTIFY_EVENT_CHANNEL_FACTORY));
		} else {
			notificationServiceCombo.setEnabled(true);
		}
	}
	
	@Override
	protected Object getEditedObject() {
		return domainsMapping;
	}

	@Override
	protected void resetToOriginalContent() {
		domainsMapping.setName(origDomainsMapping.getName());
		domainsMapping.setNotificationService(origDomainsMapping.getNotificationService());
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {
		origDomainsMapping = new DomainsMapping();
		origDomainsMapping.setName(domainsMapping.getName());
		origDomainsMapping.setNotificationService(domainsMapping.getNotificationService());

		setTitleImage(ImageHelper.getImage(origDomainsMapping));
		String partName = LabelHelper.getDomainsMappingLabel(origDomainsMapping);
		setPartName(partName);
		setTitleToolTip(partName);
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		DomainsMappingEditorInput cei = (DomainsMappingEditorInput)input;
		setInput(input);
		setSite(site);

		domainsMapping = cei.getDomainsMapping();
		if( domainsMapping.getDomainsMappingId() == null) {
			setDirty(true);
		}
		setEditedObjectAsOriginalContent();
	}
}
