/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.obops.tmcdbgui.editors.inputs.DelayModelHistoryEditorInput;
import alma.obops.tmcdbgui.editors.inputs.HistoricalDelayModelEditorInput;
import alma.obops.tmcdbgui.utils.conversation.DelaysConversationUtils;
import alma.obops.tmcdbgui.views.providers.DelayModelHistoryTableContentsProvider;
import alma.obops.tmcdbgui.views.providers.DelayModelHistoryTableLabelProvider;
import alma.obops.tmcdbgui.views.providers.HistoryRecordViewerSorter;
import alma.obops.tmcdbgui.views.providers.helpers.config.DelayModel;
import alma.tmcdb.history.HistoryRecord;

public class DelayModelHistoryEditor extends EditorPart 
{
	public static final String ID = "delaymodel-history.editor";
	private TableViewer historyViewer;
	private DelayModel delayModel;

	@Override
	public void createPartControl( Composite parent ) 
	{
		historyViewer = new TableViewer(parent, SWT.BORDER | SWT.FULL_SELECTION | SWT.MULTI);

		// Setup the columns
		String [] titles = { "Version", "Description", "Modifier", "Date" };
		for(int i = 0; i != titles.length; i++) {
			TableViewerColumn col = new TableViewerColumn(historyViewer, SWT.NONE);
			col.getColumn().setText(titles[i]);
			col.getColumn().setMoveable(false);
			col.getColumn().setResizable(true);
			col.getColumn().setWidth(250);
		}
		
		historyViewer.setSorter(new HistoryRecordViewerSorter());
		historyViewer.setContentProvider( new DelayModelHistoryTableContentsProvider() );
		historyViewer.setLabelProvider( new DelayModelHistoryTableLabelProvider() );
		historyViewer.setInput(this.delayModel);
		
		Table table = historyViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		
		MenuManager popupMenu = new MenuManager();
		final CompareDelayModelsAction compareDelayModelsAction = new CompareDelayModelsAction();
		popupMenu.add(compareDelayModelsAction);
		Menu menu = popupMenu.createContextMenu(table);
		table.setMenu(menu);

		historyViewer.addSelectionChangedListener(new ISelectionChangedListener() 
		{
			@Override
			public void selectionChanged(SelectionChangedEvent evt) 
			{
				ISelection selection = evt.getSelection();
				if(selection instanceof IStructuredSelection) {
					IStructuredSelection structuredSelection = (IStructuredSelection) selection;
					Object[] recordsSelected = structuredSelection.toArray();
					if(recordsSelected.length == 2) {
						compareDelayModelsAction.setEnabled(true);
						compareDelayModelsAction.setPreviousRecord((HistoryRecord)recordsSelected[0]);
						compareDelayModelsAction.setReferenceRecord((HistoryRecord)recordsSelected[1]);
					}
					else {
						compareDelayModelsAction.setEnabled(false);
					}
				}

			}
		});

		IDoubleClickListener listener = new GetHistoricalDelayModelDoubleClickListener();
		historyViewer.addDoubleClickListener(listener);
	}

	@Override
	public void setFocus() {
		historyViewer.getControl().setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor arg0) {
		// NOOP
	}

	@Override
	public void doSaveAs() {
		// NOOP
	}

	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException 
	{
		DelayModelHistoryEditorInput editorInput = (DelayModelHistoryEditorInput)input;
		setInput(input);
		if(null != historyViewer) {
			historyViewer.setInput(editorInput.getDelayModel());
		}
		setSite(site);
		setPartName(editorInput.getName());
	}

	@Override 
	public void setInput(IEditorInput input)
	{
		super.setInput(input);
		this.delayModel = ((DelayModelHistoryEditorInput) input).getDelayModel();
	}
	
	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}
	
	private class GetHistoricalDelayModelDoubleClickListener implements IDoubleClickListener
	{
		@Override
		public void doubleClick(DoubleClickEvent evt) 
		{
			ISelection selection = evt.getSelection();
			if(selection instanceof IStructuredSelection) {
				IStructuredSelection structuredSelection = (IStructuredSelection) selection;
				if(structuredSelection.getFirstElement() != null) {
					HistoryRecord clickedRecord = (HistoryRecord) structuredSelection.getFirstElement();
					DelayModel historicalDelayModel = null;
					try {
						historicalDelayModel = DelaysConversationUtils.getInstance().getHistoricalDelayModel(delayModel, clickedRecord);
					} catch (Exception e) {
						throw new RuntimeException("Unable to get historical delay model" + e);
					}
					
					HistoricalDelayModelEditorInput editorInput =  
						new HistoricalDelayModelEditorInput(historicalDelayModel, historicalDelayModel, clickedRecord);
					
					IWorkbenchWindow win = getSite().getWorkbenchWindow();
					try {
						win.getActivePage().openEditor(editorInput, HistoricalDelayModelEditor.ID);
					} 
					catch (PartInitException e1) {
						throw new RuntimeException("Could not open historical delay model editor", e1);
					}
				}
			}
		}
	}
	
	private class CompareDelayModelsAction extends Action
	{
		private HistoryRecord referenceRecord;
		private HistoryRecord previousRecord;
		
		public CompareDelayModelsAction() 
		{
			super("Show differences");
		}
		
		public void setReferenceRecord(HistoryRecord rec)
		{
			this.referenceRecord = rec;
		}
		
		public void setPreviousRecord(HistoryRecord rec)
		{
			this.previousRecord = rec;
		}

		public void run() 
		{
			DelayModel historicalDelayModel = null;
			DelayModel historicalDelayModelPreviousVersion = null;
			try {
				historicalDelayModel = DelaysConversationUtils.getInstance().getHistoricalDelayModel(delayModel, referenceRecord);
				historicalDelayModelPreviousVersion = DelaysConversationUtils.getInstance().getHistoricalDelayModel(delayModel, previousRecord);
			} catch (Exception e) {
				throw new RuntimeException("Unable to get historical delay models" + e);
			}
			
			HistoryRecord junkRecord = new HistoryRecord();
			junkRecord.setVersion(0L - (referenceRecord.getVersion() - previousRecord.getVersion()));
			HistoricalDelayModelEditorInput editorInput =  
				new HistoricalDelayModelEditorInput(historicalDelayModel, historicalDelayModelPreviousVersion, 
						junkRecord, "Diff delay model v." + referenceRecord.getVersion() 
						+ " to v." + previousRecord.getVersion() + " for " + delayModel.getAntenna().getName());
			
			IWorkbenchWindow win = getSite().getWorkbenchWindow();
			try {
				win.getActivePage().openEditor(editorInput, HistoricalDelayModelEditor.ID);
			} 
			catch (PartInitException e1) {
				throw new RuntimeException("Could not open historical delay model editor", e1);
			}
		}
	}
}
