/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.text.DecimalFormat;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;

import alma.BasebandNameMod.BasebandName;
import alma.NetSidebandMod.NetSideband;
import alma.PolarizationTypeMod.PolarizationType;
import alma.ReceiverBandMod.ReceiverBand;
import alma.obops.tmcdbgui.editors.inputs.DelayModelEditorInput;
import alma.obops.tmcdbgui.editors.inputs.DelayModelHistoryEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.editors.sorters.FeDelayViewerSorter;
import alma.obops.tmcdbgui.editors.sorters.IfDelayViewerSorter;
import alma.obops.tmcdbgui.editors.sorters.LoDelayViewerSorter;
import alma.obops.tmcdbgui.utils.DelayEditingUtils;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.DelaysConversationUtils;
import alma.obops.tmcdbgui.views.providers.FeDelayModelContentsProvider;
import alma.obops.tmcdbgui.views.providers.FeDelayModelEditingSupport;
import alma.obops.tmcdbgui.views.providers.FeDelayModelLabelProvider;
import alma.obops.tmcdbgui.views.providers.FeDelayModelRow;
import alma.obops.tmcdbgui.views.providers.IfDelayModelContentsProvider;
import alma.obops.tmcdbgui.views.providers.IfDelayModelEditingSupport;
import alma.obops.tmcdbgui.views.providers.IfDelayModelLabelProvider;
import alma.obops.tmcdbgui.views.providers.IfDelayModelRow;
import alma.obops.tmcdbgui.views.providers.LoDelayModelContentsProvider;
import alma.obops.tmcdbgui.views.providers.LoDelayModelEditingSupport;
import alma.obops.tmcdbgui.views.providers.LoDelayModelLabelProvider;
import alma.obops.tmcdbgui.views.providers.LoDelayModelRow;
import alma.obops.tmcdbgui.views.providers.helpers.config.DelayModel;
import alma.obops.tmcdbgui.widgets.AntennaAttributesComposite;
import alma.obops.tmcdbgui.widgets.PadAttributesComposite;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.FEDelay;
import alma.tmcdb.domain.IFDelay;
import alma.tmcdb.domain.IFProcConnectionState;
import alma.tmcdb.domain.LODelay;

/**
 * Editor for the delays associated with an antenna.
 * @author sharring
 *
 */
public class DelayModelEditor extends TmcdbObjectEditorPart 
   implements DirtyListener
{
	public static int NUM_CHARS_FOR_DELAY = 12;
	 public static final String ANTENNA_DELAY = "Antenna delay";
	 public static final String ANTENNA_DELAY_UNITS = "(s):";

	public static final String ID = "delaymodel.editor";

	private boolean dirty = false;
	private boolean antennaDelayModified = false;
	private DelayModel delayModel;
	private DelayModel delayModelCopy;
	
	// LO delay variables
	private TableViewer loDelayModelViewer;
	
	// IF delay variables
	private TableViewer ifDelayModelViewer;
	
	// FE delay variables
	private TableViewer feDelayModelViewer;
	
	private Text tAntDelayText;

	@Override
	public void doSave(IProgressMonitor progressMonitor) 
	{
		InputDialog descriptionInputDialog = new InputDialog(this.getSite().getShell(), "Description", "Please add any comments about your change", "", null);
		if(descriptionInputDialog.open() != Window.OK) 
		{
			return;
		}

		try 
		{
			// try to create a new version
			String description = descriptionInputDialog.getValue();
			String userId = System.getProperty("user.name");
			boolean canSave = DelaysConversationUtils.getInstance().
			   prepareDelayModelSave(delayModel, userId, description);

			// if the new version preparation was successful, we can then perform the save
			if(canSave) 
			{			
				applyIfDelayEdits();
				applyLoDelayEdits();
				applyFeDelayEdits();

				// save the antenna with all the delays
				try {
					boolean avgDelayChanged = !delayModel.getAntenna().getAvgDelay().equals(Double.valueOf(tAntDelayText.getText()));
					if(avgDelayChanged) {
						// update the antenna's avg delay value
						delayModel.getAntenna().setAvgDelay(Double.valueOf(tAntDelayText.getText()));
					} 	
					BaseElementConversationUtils.getInstance().saveOrUpdateAntenna(delayModel.getAntenna());
					setDirty(false);
				} catch (Exception e) {
					e.printStackTrace();
					throw new RuntimeException("Could not save antenna's delay model", e);
				}
			}
			else 
			{
				MessageDialog.openWarning(this.getSite().getShell(), "Unable to save", "Could not save; perhaps someone else is saving now. Try again later.");
			}
		}
		catch(Exception ex) 
		{
			throw new RuntimeException("Could not save delay model", ex);
		} 
		finally 
		{
			 try {
				DelaysConversationUtils.getInstance().endDelayModelSave(delayModel);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	private void applyFeDelayEdits() {
		for(FeDelayModelRow row : (FeDelayModelRow[]) feDelayModelViewer.getInput())
		{
			// for any 'new' entries in the FEDelay table, add them to our FE delays
			if(row.getLsbPolYDelay().getId() == null)
			{
				// for any 'new' entries in the FEDelay table, add them to our FE delays
				delayModel.getFeDelays().add(row.getLsbPolYDelay());
			}
			else
			{
				// for existing entries, update the delay in our original, so that we can then save it
				FEDelay matchingDelay = findMatchingFEDelay(row.getLsbPolYDelay());
				matchingDelay.setDelay(row.getLsbPolYDelay().getDelay());
				matchingDelay.setPolarization(row.getLsbPolYDelay().getPolarization());
				matchingDelay.setReceiverBand(row.getLsbPolYDelay().getReceiverBand());
				matchingDelay.setSideband(row.getLsbPolYDelay().getSideband());
			}
			if(row.getLsbPolXDelay().getId() == null)
			{
				// for any 'new' entries in the FEDelay table, add them to our FE delays
				delayModel.getFeDelays().add(row.getLsbPolXDelay());	
			}
			else
			{
				// for existing entries, update the delay in our original, so that we can then save it
				FEDelay matchingDelay = findMatchingFEDelay(row.getLsbPolXDelay());
				matchingDelay.setDelay(row.getLsbPolXDelay().getDelay());
				matchingDelay.setPolarization(row.getLsbPolXDelay().getPolarization());
				matchingDelay.setReceiverBand(row.getLsbPolXDelay().getReceiverBand());
				matchingDelay.setSideband(row.getLsbPolXDelay().getSideband());
			}
			if(row.getUsbPolYDelay().getId() == null)
			{
				// for any 'new' entries in the FEDelay table, add them to our FE delays
				delayModel.getFeDelays().add(row.getUsbPolYDelay());	
			}
			else
			{
				// for existing entries, update the delay in our original, so that we can then save it
				FEDelay matchingDelay = findMatchingFEDelay(row.getUsbPolYDelay());
				matchingDelay.setDelay(row.getUsbPolYDelay().getDelay());
				matchingDelay.setPolarization(row.getUsbPolYDelay().getPolarization());
				matchingDelay.setReceiverBand(row.getUsbPolYDelay().getReceiverBand());
				matchingDelay.setSideband(row.getUsbPolYDelay().getSideband());
			}
			if(row.getUsbPolXDelay().getId() == null)
			{
				// for any 'new' entries in the FEDelay table, add them to our FE delays
				delayModel.getFeDelays().add(row.getUsbPolXDelay());
			}
			else
			{
				// for existing entries, update the delay in our original, so that we can then save it
				FEDelay matchingDelay = findMatchingFEDelay(row.getUsbPolXDelay());
				matchingDelay.setDelay(row.getUsbPolXDelay().getDelay());
				matchingDelay.setPolarization(row.getUsbPolXDelay().getPolarization());
				matchingDelay.setReceiverBand(row.getUsbPolXDelay().getReceiverBand());
				matchingDelay.setSideband(row.getUsbPolXDelay().getSideband());
			}
		}
	}

	private void applyLoDelayEdits() {
		for(LoDelayModelRow row : (LoDelayModelRow[])loDelayModelViewer.getInput())
		{
			// for any 'new' entries in the LODelay table, add them to our LO delays
			if(row.getDelay().getId() == null) 
			{
				// for any 'new' entries in the LODelay table, add them to our LO delays
				delayModel.getLoDelays().add(row.getDelay());
			}
			else
			{
				// for existing entries, update the delay in our original, so that we can then save it
				LODelay matchingDelay = findMatchingLODelay(row.getDelay());
				matchingDelay.setBaseband(row.getDelay().getBaseband());
				matchingDelay.setDelay(row.getDelay().getDelay());
			}
		}
	}

	private void applyIfDelayEdits() {
		for(IfDelayModelRow row : (IfDelayModelRow[])ifDelayModelViewer.getInput()) 
		{
			// for each row in the table, there are 8 possible delays, which we check here 
			// by calling the corresponding 8 accessor methods on the row object
			if(row.getUsbHighPolXDelay().getId() == null)
			{
				// for any 'new' entries in the IFDelay table, add them to our IF delays
				delayModel.getIfDelays().add(row.getUsbHighPolXDelay());
			} 
			else 
			{
				// for existing entries, update the delay in our original, so that we can then save it
				IFDelay matchingDelay = this.findMatchingIFDelay(row.getUsbHighPolXDelay());
				matchingDelay.setDelay(row.getUsbHighPolXDelay().getDelay());
			}

			if(row.getUsbHighPolYDelay().getId() == null)
			{
				// for any 'new' entries in the IFDelay table, add them to our IF delays
				delayModel.getIfDelays().add(row.getUsbHighPolYDelay());
			}
			else 
			{
				// for existing entries, update the delay in our original, so that we can then save it
				IFDelay matchingDelay = this.findMatchingIFDelay(row.getUsbHighPolYDelay());
				matchingDelay.setDelay(row.getUsbHighPolYDelay().getDelay());
			}

			if(row.getUsbLowPolXDelay().getId() == null)
			{
				// for any 'new' entries in the IFDelay table, add them to our IF delays
				delayModel.getIfDelays().add(row.getUsbLowPolXDelay());
			}
			else 
			{
				// for existing entries, update the delay in our original, so that we can then save it
				IFDelay matchingDelay = this.findMatchingIFDelay(row.getUsbLowPolXDelay());
				matchingDelay.setDelay(row.getUsbLowPolXDelay().getDelay());
			}

			if(row.getUsbLowPolYDelay().getId() == null)
			{
				// for any 'new' entries in the IFDelay table, add them to our IF delays
				delayModel.getIfDelays().add(row.getUsbLowPolYDelay());
			}
			else 
			{
				// for existing entries, update the delay in our original, so that we can then save it
				IFDelay matchingDelay = this.findMatchingIFDelay(row.getUsbLowPolYDelay());
				matchingDelay.setDelay(row.getUsbLowPolYDelay().getDelay());
			}

			if(row.getLsbHighPolXDelay().getId() == null)
			{
				// for any 'new' entries in the IFDelay table, add them to our IF delays
				delayModel.getIfDelays().add(row.getLsbHighPolXDelay());
			} 
			else 
			{
				// for existing entries, update the delay in our original, so that we can then save it
				IFDelay matchingDelay = this.findMatchingIFDelay(row.getLsbHighPolXDelay());
				matchingDelay.setDelay(row.getLsbHighPolXDelay().getDelay());
			}

			if(row.getLsbHighPolYDelay().getId() == null)
			{
				// for any 'new' entries in the IFDelay table, add them to our IF delays
				delayModel.getIfDelays().add(row.getLsbHighPolYDelay());
			}
			else 
			{
				// for existing entries, update the delay in our original, so that we can then save it
				IFDelay matchingDelay = this.findMatchingIFDelay(row.getLsbHighPolYDelay());
				matchingDelay.setDelay(row.getLsbHighPolYDelay().getDelay());
			}

			if(row.getLsbLowPolXDelay().getId() == null)
			{
				// for any 'new' entries in the IFDelay table, add them to our IF delays
				delayModel.getIfDelays().add(row.getLsbLowPolXDelay());
			}
			else 
			{
				// for existing entries, update the delay in our original, so that we can then save it
				IFDelay matchingDelay = this.findMatchingIFDelay(row.getLsbLowPolXDelay());
				matchingDelay.setDelay(row.getLsbLowPolXDelay().getDelay());
			}

			if(row.getLsbLowPolYDelay().getId() == null)
			{
				// for any 'new' entries in the IFDelay table, add them to our IF delays
				delayModel.getIfDelays().add(row.getLsbLowPolYDelay());
			}
			else 
			{
				// for existing entries, update the delay in our original, so that we can then save it
				IFDelay matchingDelay = this.findMatchingIFDelay(row.getLsbLowPolYDelay());
				matchingDelay.setDelay(row.getLsbLowPolYDelay().getDelay());
			}
		}
	}

	private FEDelay findMatchingFEDelay(FEDelay fedelay) 
	{
		FEDelay retVal = null;
		
		for(FEDelay delay : delayModel.getFeDelays())
		{
			if(delay.getId() != null && delay.getId().equals(fedelay.getId()))
			{
				retVal = delay;
				break;
			}
		}
		return retVal;
	}

	private LODelay findMatchingLODelay(LODelay lodelay) 
	{
		LODelay retVal = null;
	
		for(LODelay delay : delayModel.getLoDelays())
		{
			if(delay.getId() != null && delay.getId().equals(lodelay.getId()))
			{
				retVal = delay;
				break;
			}
		}
		
		return retVal;
	}

	private IFDelay findMatchingIFDelay(IFDelay fedelay) 
	{
		IFDelay retVal = null;
		
		for(IFDelay delay : delayModel.getIfDelays())
		{
			if(delay.getId() != null && delay.getId().equals(fedelay.getId()))
			{
				retVal = delay;
				break;
			}
		}
		
		return retVal;
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		DelayModelEditorInput delayEdInput = (DelayModelEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(delayEdInput.getName());
		delayModel = delayEdInput.getDelayModel();
		makeDelayModelCopy();
	}
	
	private void makeDelayModelCopy() 
	{
		// copy an antenna as a holder for our copied delays
		Antenna antCopy = new Antenna();
		
		// copy the IF delays
		Set<IFDelay> ifdelaysCopy = new HashSet<IFDelay>();
		for(IFDelay origIfDelay : delayModel.getIfDelays())
		{
			IFDelay ifdelayCopy = new IFDelay(origIfDelay.getBaseband(), origIfDelay.getPolarization(), 
					origIfDelay.getIfSwitch(), origIfDelay.getDelay());
			ifdelayCopy.setId(origIfDelay.getId());
			ifdelaysCopy.add(ifdelayCopy);
		}
		antCopy.setIfDelays(ifdelaysCopy);

		// copy the LO delays
		Set<LODelay> lodelaysCopy = new HashSet<LODelay>();
		for(LODelay origLoDelay : delayModel.getLoDelays())
		{
			LODelay lodelayCopy = new LODelay(origLoDelay.getBaseband(), origLoDelay.getDelay());
			lodelayCopy.setId(origLoDelay.getId());
			lodelaysCopy.add(lodelayCopy);
		}
		antCopy.setLoDelays(lodelaysCopy);
		
		// copy the FrontEnd delays
		Set<FEDelay> fedelaysCopy = new HashSet<FEDelay>();
		for(FEDelay origFeDelay : delayModel.getFeDelays())
		{
			FEDelay fedelayCopy = new FEDelay(origFeDelay.getReceiverBand(), origFeDelay.getPolarization(), 
					origFeDelay.getSideband(), origFeDelay.getDelay());
			fedelayCopy.setId(origFeDelay.getId());
			fedelaysCopy.add(fedelayCopy);
		}
		antCopy.setFrontEndDelays(fedelaysCopy);
		
		delayModelCopy = new DelayModel(antCopy, delayModel.getPad());
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		GridLayout gridLayout = new GridLayout();

		ScrolledComposite sc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER);
		sc.setExpandHorizontal(true);
		sc.setExpandVertical(true);

		Composite editorComposite = new Composite(sc, SWT.NONE);
		editorComposite.setLayout(gridLayout);
		gridLayout.numColumns = 1;
		
		createAntDelayComposite(editorComposite);
		createIfDelaysGroup(editorComposite);
		createLoDelaysGroup(editorComposite);
		createFeDelaysGroup(editorComposite);
		createButtonsComposite(editorComposite);

		// Finally, calculate the minimum size so the scroll composite knows
		// when to start its role
		sc.setContent(editorComposite);
		sc.setMinSize(editorComposite.computeSize(SWT.DEFAULT, SWT.DEFAULT));
	}

	private void createAntDelayComposite(Composite editorComposite)
	{
		Composite composite = new Composite(editorComposite, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 4;
		gridLayout.makeColumnsEqualWidth = false;
		composite.setLayout(gridLayout);
		
		Label tAntLabel = new Label(composite, SWT.None);
		tAntLabel.setText(ANTENNA_DELAY + " for antenna " 
				+ delayModel.getAntenna().getName() + " " + ANTENNA_DELAY_UNITS);
		tAntDelayText = new Text(composite, SWT.BORDER);
		GridData gd = GuiUtils.getGridDataForCharWidth(NUM_CHARS_FOR_DELAY, tAntDelayText);
		tAntDelayText.setLayoutData(gd);
		
		DecimalFormat formatter = new DecimalFormat(AntennaAttributesComposite.OFFSET_FORMAT);
		if(null != delayModel.getAntenna().getAvgDelay()) {
		   String formattedDelay = formatter.format(delayModel.getAntenna().getAvgDelay());
		   this.tAntDelayText.setText(formattedDelay);
		} else {
			this.tAntDelayText.setText("");
		}
		
		tAntDelayText.addModifyListener(new ModifyListener() 
		{
			@Override
			public void modifyText(ModifyEvent e) 
			{
				Double newVal = Double.valueOf(tAntDelayText.getText());
				if(null != delayModel.getAntenna().getAvgDelay() && 
				   null !=  newVal && 
				   !newVal.equals(delayModel.getAntenna().getAvgDelay()))  
				{
				  setAntennaDelayModified(true);
				} else {
					setAntennaDelayModified(false);
				}
			}	
		});
		
		Label tPadLabel = new Label(composite, SWT.None);
		if(delayModel.getPad() != null) {
			tPadLabel.setText(PadAttributesComposite.PAD_DELAY + " for pad " 
					+ delayModel.getPad().getName() + " " + PadAttributesComposite.PAD_DELAY_UNITS);
		} else {
			tPadLabel.setText(PadAttributesComposite.PAD_DELAY + " for pad " 
					+ "N/A " + PadAttributesComposite.PAD_DELAY_UNITS);
		}
		Text tPadDelayText = new Text(composite, SWT.BORDER);
		tPadDelayText.setEditable(false);
		tPadDelayText.setEnabled(false);
		gd = GuiUtils.getGridDataForCharWidth(NUM_CHARS_FOR_DELAY, tPadDelayText);
		tPadDelayText.setLayoutData(gd);
		
		formatter = new DecimalFormat(AntennaAttributesComposite.OFFSET_FORMAT);
		if(null != delayModel.getPad() && null != delayModel.getPad().getAvgDelay()) {
		   String formattedDelay = formatter.format(delayModel.getPad().getAvgDelay());
		   tPadDelayText.setText(formattedDelay);
		} else {
			tPadDelayText.setText("");
		}
	}
	
	private void createButtonsComposite(Composite editorComposite) 
	{
		Composite buttonsComposite = new Composite(editorComposite, SWT.NONE);
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = true;
		gdata.horizontalAlignment = SWT.BEGINNING;
		gdata.verticalAlignment = SWT.BEGINNING;
		buttonsComposite.setLayoutData(gdata);
		buttonsComposite.setLayout(new FillLayout());
		
		 //   Create and configure the "history" button
        Button historyButton = new Button(buttonsComposite, SWT.PUSH | SWT.BEGINNING);
        historyButton.setText("History");
        
        historyButton.addSelectionListener(new SelectionAdapter() 
        {
           public void widgetSelected(SelectionEvent e) 
           {
        	   DelayModelHistoryEditorInput editorInput =  new DelayModelHistoryEditorInput(delayModel);
        	   IWorkbenchWindow win = getSite().getWorkbenchWindow();
        	   try {
        		   win.getActivePage().openEditor(editorInput, DelayModelHistoryEditor.ID);
        	   } 
        	   catch (PartInitException e1) {
        		   throw new RuntimeException("Could not open delay model history editor", e1);
        	   }
           }
        });
	}

	private void createFeDelaysGroup(Composite editorComposite) 
	{
		Group feDelayTableGroup = new Group(editorComposite, SWT.BORDER);
		feDelayTableGroup.setText("FE Delays");
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = true;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.FILL;
		feDelayTableGroup.setLayoutData(gdata);
		feDelayTableGroup.setLayout(new FillLayout());
		
		feDelayModelViewer = new TableViewer(feDelayTableGroup, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Receiver band", "USB Pol X (s)", "USB Pol Y (s)", "LSB Pol X (s)", "LSB Pol Y (s)"};
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(feDelayModelViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.setEditingSupport(new FeDelayModelEditingSupport(feDelayModelViewer, i, this));
    		col.getColumn().pack();
    	}
    	Table table = feDelayModelViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

    	feDelayModelViewer.setSorter(new FeDelayViewerSorter());
    	feDelayModelViewer.setContentProvider( new FeDelayModelContentsProvider() );
    	feDelayModelViewer.setLabelProvider( new FeDelayModelLabelProvider() );
    	feDelayModelViewer.setInput(populateFeRows()); // trigger a content reload
	}

	private FeDelayModelRow[] populateFeRows() 
	{
		FeDelayModelRow[] retVal = new FeDelayModelRow[10];

		for(short i = 0; i < 10; i++)
		{
			retVal[i] = new FeDelayModelRow(i);	
		}

		for(FEDelay feDelay : this.delayModelCopy.getFeDelays())
		{
			int basebandVal = DelayEditingUtils.getIntFromReceiverBandEnum(feDelay.getReceiverBand());
			if(feDelay.getSideband().equals(NetSideband.USB)) 
			{
				if(feDelay.getPolarization().equals(PolarizationType.X)) 
				{
					retVal[basebandVal].setUsbPolXDelay(feDelay);
				}
				else if(feDelay.getPolarization().equals(PolarizationType.Y)) 
				{
					retVal[basebandVal].setUsbPolYDelay(feDelay);
				}
			} 
			else if(feDelay.getSideband().equals(NetSideband.LSB))
			{
				if(feDelay.getPolarization().equals(PolarizationType.X)) 
				{
					retVal[basebandVal].setLsbPolXDelay(feDelay);
				}
				else if(feDelay.getPolarization().equals(PolarizationType.Y)) 
				{
					retVal[basebandVal].setLsbPolYDelay(feDelay);
				}
			}
		}
		
		for(short i = 0; i < 10; i++)
		{
			if(retVal[i].getUsbPolXDelay() == null) 
			{
				FEDelay fedelay = new FEDelay();
				assignBaseband(fedelay, i);
				fedelay.setDelay(new Double(0));
				fedelay.setSideband(NetSideband.USB);
				fedelay.setPolarization(PolarizationType.X);
				fedelay.setId(null);
				retVal[i].setUsbPolXDelay(fedelay);
			}
			if(retVal[i].getUsbPolYDelay() == null) 
			{
				FEDelay fedelay = new FEDelay();
				fedelay.setReceiverBand(DelayEditingUtils.getReceiverBandForValue(i)); 
				fedelay.setDelay(new Double(0));
				fedelay.setSideband(NetSideband.USB);
				fedelay.setPolarization(PolarizationType.Y);
				fedelay.setId(null);
				retVal[i].setUsbPolYDelay(fedelay);
			}
			if(retVal[i].getLsbPolYDelay() == null) 
			{
				FEDelay fedelay = new FEDelay();
				fedelay.setReceiverBand(DelayEditingUtils.getReceiverBandForValue(i)); 
				fedelay.setDelay(new Double(0));
				fedelay.setSideband(NetSideband.LSB);
				fedelay.setPolarization(PolarizationType.Y);
				fedelay.setId(null);
				retVal[i].setLsbPolYDelay(fedelay);
			}
			if(retVal[i].getLsbPolXDelay() == null) 
			{
				FEDelay fedelay = new FEDelay();
				fedelay.setReceiverBand(DelayEditingUtils.getReceiverBandForValue(i)); 
				fedelay.setDelay(new Double(0));
				fedelay.setSideband(NetSideband.LSB);
				fedelay.setPolarization(PolarizationType.X);
				fedelay.setId(null);
				retVal[i].setLsbPolXDelay(fedelay);
			}
		}
		
		return retVal;
	}

	private void assignBaseband(FEDelay fedelay, short i) 
	{
		switch(i) 
		{
		case 0:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_01);
			break;
		case 1:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_02);
			break;
		case 2:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_03);
			break;
		case 3:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_04);
			break;
		case 4:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_05);
			break;
		case 5:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_06);
			break;
		case 6:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_07);
			break;
		case 7:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_08);
			break;
		case 8:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_09);
			break;
		case 9:
			fedelay.setReceiverBand(ReceiverBand.ALMA_RB_10);
			break;
		default:
			throw new IllegalStateException("ALMA only supports 10 receiver bands, but enum has more than 10");
		}
	}

	private void createIfDelaysGroup(Composite editorComposite) 
	{
		Group ifDelayTableGroup = new Group(editorComposite, SWT.BORDER);
		ifDelayTableGroup.setText("IF Delays");
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = false;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.BEGINNING;
		ifDelayTableGroup.setLayoutData(gdata);
		ifDelayTableGroup.setLayout(new FillLayout());
		
		ifDelayModelViewer = new TableViewer(ifDelayTableGroup, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Baseband", "USB Low Pol X (s)", "USB Low Pol Y (s)", "USB High Pol X (s)", "USB High Pol Y (s)", 
    			                         "LSB Low Pol X (s)", "LSB Low Pol Y (s)", "LSB High Pol X (s)", "LSB High Pol Y (s)"};
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(ifDelayModelViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.setEditingSupport(new IfDelayModelEditingSupport(ifDelayModelViewer, i, this));
    		col.getColumn().pack();
    	}
    	Table table = ifDelayModelViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

    	ifDelayModelViewer.setSorter(new IfDelayViewerSorter());
    	ifDelayModelViewer.setContentProvider( new IfDelayModelContentsProvider() );
    	ifDelayModelViewer.setLabelProvider( new IfDelayModelLabelProvider() );
    	ifDelayModelViewer.setInput(populateIfRows()); // trigger a content reload
	}

	private IfDelayModelRow[] populateIfRows() 
	{
		IfDelayModelRow[] retVal = new IfDelayModelRow[4];

		for(short i = 0; i < 4; i++)
		{
			retVal[i] = new IfDelayModelRow(this.getBasebandNameForValue(i));	
		}

		for(IFDelay ifDelay : this.delayModelCopy.getIfDelays())
		{
			int basebandInt = getIntFromBasebandEnum(ifDelay.getBaseband());
			switch(ifDelay.getIfSwitch())
			{
			case USB_LOW: 
				if(ifDelay.getPolarization().equals(PolarizationType.X)) 
				{
					retVal[basebandInt].setUsbLowPolXDelay(ifDelay);
				}
				else if(ifDelay.getPolarization().equals(PolarizationType.Y)) 
				{
					retVal[basebandInt].setUsbLowPolYDelay(ifDelay);
				}
				break;
			case USB_HIGH:
				if(ifDelay.getPolarization().equals(PolarizationType.X)) 
				{
					retVal[basebandInt].setUsbHighPolXDelay(ifDelay);
				}
				else if(ifDelay.getPolarization().equals(PolarizationType.Y)) 
				{
					retVal[basebandInt].setUsbHighPolYDelay(ifDelay);
				}
				break;
			case LSB_LOW:
				if(ifDelay.getPolarization().equals(PolarizationType.X)) 
				{
					retVal[basebandInt].setLsbLowPolXDelay(ifDelay);
				}
				else if(ifDelay.getPolarization().equals(PolarizationType.Y)) 
				{
					retVal[basebandInt].setLsbLowPolYDelay(ifDelay);
				}
				break;
			case LSB_HIGH:
				if(ifDelay.getPolarization().equals(PolarizationType.X)) 
				{
					retVal[basebandInt].setLsbHighPolXDelay(ifDelay);
				}
				else if(ifDelay.getPolarization().equals(PolarizationType.Y)) 
				{
					retVal[basebandInt].setLsbHighPolYDelay(ifDelay);
				}
				break;
			}
		}
		
		for(short i = 0; i < 4; i++)
		{
			BasebandName bbname = getBasebandNameForValue(i);
			if(retVal[i].getUsbHighPolXDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.USB_HIGH);
				ifdelay.setPolarization(PolarizationType.X);
				ifdelay.setId(null);
				retVal[i].setUsbHighPolXDelay(ifdelay);
			}
			if(retVal[i].getUsbHighPolYDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.USB_HIGH);
				ifdelay.setPolarization(PolarizationType.Y);
				ifdelay.setId(null);
				retVal[i].setUsbHighPolYDelay(ifdelay);
			}
			if(retVal[i].getUsbLowPolXDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.USB_LOW);
				ifdelay.setPolarization(PolarizationType.X);
				ifdelay.setId(null);
				retVal[i].setUsbLowPolXDelay(ifdelay);
			}
			if(retVal[i].getUsbLowPolYDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.USB_LOW);
				ifdelay.setPolarization(PolarizationType.Y);
				ifdelay.setId(null);
				retVal[i].setUsbLowPolYDelay(ifdelay);
			}
			if(retVal[i].getLsbHighPolXDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.LSB_HIGH);
				ifdelay.setPolarization(PolarizationType.X);
				ifdelay.setId(null);
				retVal[i].setLsbHighPolXDelay(ifdelay);
			}
			if(retVal[i].getLsbHighPolYDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.LSB_HIGH);
				ifdelay.setPolarization(PolarizationType.Y);
				ifdelay.setId(null);
				retVal[i].setLsbHighPolYDelay(ifdelay);
			}
			if(retVal[i].getLsbLowPolXDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.LSB_LOW);
				ifdelay.setPolarization(PolarizationType.X);
				ifdelay.setId(null);
				retVal[i].setLsbLowPolXDelay(ifdelay);
			}
			if(retVal[i].getLsbLowPolYDelay() == null) 
			{
				IFDelay ifdelay = new IFDelay();
				ifdelay.setBaseband(bbname);
				ifdelay.setDelay(new Double(0));
				ifdelay.setIfSwitch(IFProcConnectionState.LSB_LOW);
				ifdelay.setPolarization(PolarizationType.Y);
				ifdelay.setId(null);
				retVal[i].setLsbLowPolYDelay(ifdelay);
			}
		}
		
		return retVal;
	}

	private BasebandName getBasebandNameForValue(short i) 
	{
		BasebandName retVal = null;
		switch(i)
		{
		case 0:
			retVal = BasebandName.BB_1;
			break;
		case 1:
			retVal = BasebandName.BB_2;
			break;
		case 2:
			retVal = BasebandName.BB_3;
			break;
		case 3:
			retVal = BasebandName.BB_4;
			break;
		default:
			throw new IllegalStateException("ALMA only supports 4 basebands; enum attempts to use more than 4");
		}
		return retVal;
	}

	private int getIntFromBasebandEnum(BasebandName baseband) 
	{
		int retVal = -1;
		if(baseband.equals(BasebandName.BB_1))
		{
			retVal = 0;
		}
		else if(baseband.equals(BasebandName.BB_2))
		{
			retVal = 1;
		}
		else if(baseband.equals(BasebandName.BB_3))
		{
			retVal = 2;
		}
		else if(baseband.equals(BasebandName.BB_4))
		{
			retVal = 3;
		}	
		else 
		{
			throw new IllegalStateException("ALMA only supports 4 basebands, but enumeration attempts to use more than 4");
		}
		return retVal;
	}

	private LoDelayModelRow[] populateLoRows() 
	{
		LoDelayModelRow[] retVal = new LoDelayModelRow[4];
		for(LODelay loDelay : this.delayModelCopy.getLoDelays())
		{
			// for each LODelay in the 
			int bb = this.getIntFromBasebandEnum(loDelay.getBaseband());
			retVal[bb] = new LoDelayModelRow(loDelay);
		}
		for(short i = 0; i < 4; i++) 
		{
			BasebandName bbname = getBasebandNameForValue(i);
			if(retVal[i] == null)
			{
				LODelay newDelay = new LODelay();
				newDelay.setBaseband(bbname);
				newDelay.setDelay(new Double(0));
				newDelay.setId(null);
				retVal[i] = new LoDelayModelRow(newDelay);
			}
		}
		
		return retVal;
	}
	
	private void createLoDelaysGroup(Composite editorComposite) {
		Group loDelayTableGroup = new Group(editorComposite, SWT.BORDER);
		loDelayTableGroup.setText("LO Delays");
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = false;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.BEGINNING;
		loDelayTableGroup.setLayoutData(gdata);
		loDelayTableGroup.setLayout(new FillLayout());
		
		loDelayModelViewer = new TableViewer(loDelayTableGroup, SWT.BORDER | SWT.FULL_SELECTION);
		
    	// Setup the columns
    	String [] titles = { "Baseband", "Delay (s)"};
    	for(int i = 0; i != titles.length; i++) {
    		TableViewerColumn col = new TableViewerColumn(loDelayModelViewer, SWT.NONE);
    		col.getColumn().setText(titles[i]);
    		col.getColumn().setMoveable(false);
    		col.getColumn().setResizable(true);
    		col.setEditingSupport(new LoDelayModelEditingSupport(loDelayModelViewer, i, this));
    		col.getColumn().pack();
    	}
    	Table table = loDelayModelViewer.getTable();
    	table.setHeaderVisible(true);
    	table.setLinesVisible(true);

    	loDelayModelViewer.setSorter(new LoDelayViewerSorter());
    	loDelayModelViewer.setContentProvider( new LoDelayModelContentsProvider() );
    	loDelayModelViewer.setLabelProvider( new LoDelayModelLabelProvider() );
    	loDelayModelViewer.setInput(populateLoRows()); // trigger a content reload
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return dirty || antennaDelayModified;
	}
	
	private void setAntennaDelayModified(boolean val) {
		this.antennaDelayModified = val;
		firePropertyChange(PROP_DIRTY);
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean d) {
		if(d == false) {
			setAntennaDelayModified(false);
		}
		this.dirty = d;
		firePropertyChange(PROP_DIRTY);
	}
}
