/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.beans.PropertyChangeEvent;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.acs.tmcdb.DefaultCanAddress;
import alma.obops.tmcdbgui.editors.inputs.DefaultCanAddressEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.DefaultCanAddressConversationUtils;

/**
 * Editor for a {@link DefaultCanAddress} object
 * 
 * @author rtobar, Aug 18th, 2010
 */


public class DefaultCanAddressEditor extends TmcdbObjectEditor {

	public static final String ID = "default-can-address.editor";

	/* The actual DCA being edited */
	private DefaultCanAddress _dca;

	/* Initial contents, used to fallback */
	private DefaultCanAddress _origDca;

	// Need to be here to interact with them
	private Group   canGroup;
	private Text    nodeAddressText;
	private Label   nodeAddressHexLabel;
	private Spinner channelNumberSpinner;

	private Group   ethernetGroup;
	private Text    hostnameText;
	private Spinner portSpinner;
	private Text    macAddressText;
	private Spinner retriesSpinner;
	private Text    timeoutText; // This one is not binded with jface
	private Spinner lingerTimeSpinner;

	@Override
	public void setFocus() {
		hostnameText.setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor monitor) {

		// If it's ethernet, then let's check the timeout value
		String timeout = timeoutText.getText();
		if( timeout != null && timeout.trim().length() != 0  ) {
			try {
				_dca.setTimeOutRxTx(Double.parseDouble(timeout));
			} catch (NumberFormatException e) {
				if( _dca.getIsEthernet() ) {
					MessageDialog.openError(getSite().getShell(), "Cannot save CAN/Ethernet configuration",
							"Error while saving CAN/Ethernet configuration for component " + LabelHelper.getFullPath(_dca.getComponent(), false));
					return;
				}
				// otherwise, just follow, doesn't matter that it's not recorded
			}
		}

		// Persist the object
		try {
			DefaultCanAddressConversationUtils.getInstance().saveOrUpdate(_dca);
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(getSite().getShell(), "Cannot save CAN/Ethernet configuration",
			    "Error while saving CAN/Ethernet configuration for component " + LabelHelper.getFullPath(_dca.getComponent(), false));
			return;
		}

		setEditedObjectAsOriginalContent();
		setDirty(false);
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException {
		DefaultCanAddressEditorInput cei = (DefaultCanAddressEditorInput)input;
		setInput(input);
		setSite(site);

		_dca = cei.getDefaultCanAddress();
		if( _dca.getComponentId() == null ||  _dca.getComponentId() < 0 )
			setDirty(true);

		setEditedObjectAsOriginalContent();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) {

		/* Widgets */
		final Button isEthernetCheck;

		parent.setLayout(new FillLayout());
		ScrolledComposite sc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.BORDER);
		sc.setExpandHorizontal(true);
		sc.setExpandVertical(true);

		Composite composite = new Composite(sc, SWT.NONE);
		composite.setLayout(new GridLayout(2, false));

		/* Is Ethernet? */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cProcTypeLabel = new Label(composite, SWT.NONE);
		cProcTypeLabel.setText("Is Ethernet?");
		cProcTypeLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		isEthernetCheck = new Button(composite, SWT.CHECK);
		isEthernetCheck.setLayoutData(gd);
		isEthernetCheck.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent evt) {
				widgetDefaultSelected(evt);
			}
			public void widgetDefaultSelected(SelectionEvent evt) {
				enableEthernetWidgets( isEthernetCheck.getSelection() );
			}
		});


		/* *CAN GROUP* */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalSpan = 2;
		canGroup = new Group(composite, SWT.BORDER);
		canGroup.setText("CAN configuration");
		canGroup.setLayoutData(gd);
		canGroup.setLayout(new GridLayout(3, false));

		/* Node Address (with hex label) */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cNodeAddressLabel = new Label(canGroup, SWT.NONE);
		cNodeAddressLabel.setText("Node Address");
		cNodeAddressLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		nodeAddressText = new Text(canGroup, SWT.BORDER);
		nodeAddressText.setLayoutData(gd);

		gd = new GridData(SWT.RIGHT, SWT.CENTER, false, false);
		nodeAddressHexLabel = new Label(canGroup, SWT.NONE);
		nodeAddressHexLabel.setLayoutData(gd);
		nodeAddressHexLabel.setAlignment(SWT.RIGHT);
		nodeAddressHexLabel.setText("  0xFF"); // to pack it right

		/* Channel Number (with hex label) */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label channelNumberLabel = new Label(canGroup, SWT.NONE);
		channelNumberLabel.setText("Channel Number");
		channelNumberLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalSpan = 2;
		gd.horizontalIndent = 20;
		channelNumberSpinner = new Spinner(canGroup, SWT.BORDER);
		channelNumberSpinner.setLayoutData(gd);
		channelNumberSpinner.setMinimum(0);
		channelNumberSpinner.setMaximum(18);
		
		/* *ETHERNET GROUP */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalSpan = 2;
		ethernetGroup = new Group(composite, SWT.BORDER);
		ethernetGroup.setText("Ethernet configuration");
		ethernetGroup.setLayoutData(gd);
		ethernetGroup.setLayout(new GridLayout(2, false));

		/* Hostname */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label hostameLabel = new Label(ethernetGroup, SWT.NONE);
		hostameLabel.setText("Hostname");
		hostameLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		hostnameText = new Text(ethernetGroup, SWT.BORDER);
		hostnameText.setLayoutData(gd);

		/* Port */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label portLabel = new Label(ethernetGroup, SWT.NONE);
		portLabel.setText("Port");
		portLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		portSpinner = new Spinner(ethernetGroup, SWT.BORDER);
		portSpinner.setLayoutData(gd);
		portSpinner.setMinimum(1);
		portSpinner.setMaximum(50000);

		/* MAC Address */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label macAddressLabel = new Label(ethernetGroup, SWT.NONE);
		macAddressLabel.setText("MAC Address");
		macAddressLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		macAddressText = new Text(ethernetGroup, SWT.BORDER);
		macAddressText.setLayoutData(gd);

		/* Retries */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label retriesLabel = new Label(ethernetGroup, SWT.NONE);
		retriesLabel.setText("Retries");
		retriesLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		retriesSpinner = new Spinner(ethernetGroup, SWT.BORDER);
		retriesSpinner.setLayoutData(gd);
		retriesSpinner.setMinimum(0);
		retriesSpinner.setMaximum(32767);

		/* MAC Address */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label timeoutLabel = new Label(ethernetGroup, SWT.NONE);
		timeoutLabel.setText("Timeout");
		timeoutLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		timeoutText = new Text(ethernetGroup, SWT.BORDER);
		timeoutText.setLayoutData(gd);

		/* Linger time */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label lingerTimeLabel = new Label(ethernetGroup, SWT.NONE);
		lingerTimeLabel.setText("Linger time");
		lingerTimeLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		lingerTimeSpinner = new Spinner(ethernetGroup, SWT.BORDER);
		lingerTimeSpinner.setLayoutData(gd);
		lingerTimeSpinner.setMinimum(0);
		lingerTimeSpinner.setMaximum(500000);

		// Finally, calculate the minimum size so the scroll composite knows
		// when to start its role
		sc.setContent(composite);
		sc.setMinSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));

		// Data binding and subscription
		bind("isEthernet", isEthernetCheck);
		bind("nodeAddress", nodeAddressText);
		bind("channelNumber", channelNumberSpinner);
		bind("hostname", hostnameText);
		bind("port", portSpinner);
		bind("macAddress", macAddressText);
		bind("retries", retriesSpinner);
		bind("lingerTime", lingerTimeSpinner);

		// Timeout must be handled by hand, since there we have
		// no bind() method for going from string to double (yet)
		// bind("timeout", timeoutText);
		timeoutText.setText( _dca.getTimeOutRxTx() != null ? _dca.getTimeOutRxTx().toString() : "");
		subscribeToChanges(timeoutText);

		// Initial widget states
		nodeAddressHexLabel.setText( String.format("0x%X", Integer.parseInt(_dca.getNodeAddress())) );
		enableEthernetWidgets(_dca.getIsEthernet());
	}

	private void enableEthernetWidgets( boolean isEthernet ) {

		canGroup.setEnabled(!isEthernet);
		nodeAddressText.setEnabled(!isEthernet);
		channelNumberSpinner.setEnabled(!isEthernet);
		nodeAddressHexLabel.setEnabled(!isEthernet);

		ethernetGroup.setEnabled(isEthernet);
		hostnameText.setEnabled(isEthernet);
		portSpinner.setEnabled(isEthernet);
		macAddressText.setEnabled(isEthernet);
		retriesSpinner.setEnabled(isEthernet);
		timeoutText.setEnabled(isEthernet);
		lingerTimeSpinner.setEnabled(isEthernet);
	}


	@Override
	public void propertyChange(PropertyChangeEvent evt) {
		super.propertyChange(evt);
		// Update the hex label for the channel number if necessary
		if( evt.getPropertyName().equals("nodeAddress") )
			nodeAddressHexLabel.setText( String.format("0x%X", Integer.parseInt( _dca.getNodeAddress())) );
	}

	@Override
	protected Object getEditedObject() {
		return _dca;
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {

		_origDca = new DefaultCanAddress();
		_origDca.setComponent(_dca.getComponent());
		_origDca.setChannelNumber(_dca.getChannelNumber());
		_origDca.setHostname(_dca.getHostname());
		_origDca.setIsEthernet(_dca.getIsEthernet());
		_origDca.setLingerTime(_dca.getLingerTime());
		_origDca.setMacAddress(_dca.getMacAddress());
		_origDca.setNodeAddress(_dca.getNodeAddress());
		_origDca.setPort(_dca.getPort());
		_origDca.setRetries(_dca.getRetries());
		_origDca.setTimeOutRxTx(_dca.getTimeOutRxTx());

		setTitleImage(ImageHelper.getImage(_origDca));
		setPartName(_origDca.getComponent().getComponentName());
		setTitleToolTip( LabelHelper.getFullPath(_origDca.getComponent(), false));
	}

	protected void resetToOriginalContent() {

		_dca.setComponent(_origDca.getComponent());
		_dca.setChannelNumber(_origDca.getChannelNumber());
		_dca.setHostname(_origDca.getHostname());
		_dca.setIsEthernet(_origDca.getIsEthernet());
		_dca.setLingerTime(_origDca.getLingerTime());
		_dca.setMacAddress(_origDca.getMacAddress());
		_dca.setNodeAddress(_origDca.getNodeAddress());
		_dca.setPort(_origDca.getPort());
		_dca.setRetries(_origDca.getRetries());
		_dca.setTimeOutRxTx(_origDca.getTimeOutRxTx());
	}

}
