/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.acs.tmcdb.ContStartOptType;
import alma.acs.tmcdb.ContainerStartupOption;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.editors.inputs.ContainerStartupOptionEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.conversation.ContainerConversationUtils;
import alma.obops.tmcdbgui.views.SoftwareDeploymentView;

public class ContainerStartupOptionEditor extends TmcdbObjectEditor 
{
	private static final String ENV_VAR = ContStartOptType.ENV_VAR.toString();

	private static final String EXEC_ARG = ContStartOptType.EXEC_ARG.toString();

	private static final String EXEC_ARG_LANG = ContStartOptType.EXEC_ARG_LANG.toString();

	private static final String CONT_ARG = ContStartOptType.CONT_ARG.toString();

	public static final String[] TYPES = {
		ENV_VAR, EXEC_ARG, EXEC_ARG_LANG, CONT_ARG 
	};
	
	public static final String ID = "containerstartupoption.editor";

	/* The actual computer being edited */
	private ContainerStartupOption _option;

	/* Initial contents, used to fallback */
	private ContainerStartupOption _origOption;
	
	private Text cNameText;
	
	@Override
	public void setFocus() {
		cNameText.setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor monitor) {

		boolean isNewOption = false;

		if( _option.getContStartOptId() == null ) {
			isNewOption = true;
		}

		// Check for invalid inputs
		if(  (_option.getOptionName() == null || _option.getOptionName().trim().equals(""))  ||
			 (_option.getOptionType() == null ) ||
			 (_option.getOptionValue() == null || _option.getOptionValue().trim().equals("")) ) 
		{
			MessageDialog.openInformation(getSite().getShell(),
					"Please specify all fields",
					"A startup option cannot be saved without all fields defined");
			return;
		}

		// Persist the object
		try {
//			ContainerConversationUtils.getInstance().saveOrUpdateContainerStartupOption(_option);
			if(isNewOption) {
				ContainerConversationUtils.getInstance().hydrateContainerStartupOptions(_option.getContainer());
				_option.getContainer().addContainerStartupOptionToContainerStartupOptions(_option);
				ContainerConversationUtils.getInstance().saveOrUpdateContainer(_option.getContainer());
			} else {
				ContainerConversationUtils.getInstance().saveOrUpdateContainerStartupOption(_option);
			}
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(getSite().getShell(), "Cannot save ContainerStartupOption", "Error while saving ContainerStartupOption " + _option.getOptionName());
			return;
		}

		setEditedObjectAsOriginalContent();
		setDirty(false);

		// If we're adding a new containerstartupoption, let's refresh the SDV if available
		if( isNewOption ) {
			SoftwareDeploymentView sdv = (SoftwareDeploymentView)RcpUtils.findView( SoftwareDeploymentView.ID );
			sdv.internalModelChange();
		}
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		ContainerStartupOptionEditorInput cei = (ContainerStartupOptionEditorInput)input;
		setInput(input);
		setSite(site);

		_option = cei.getContainerStartupOption();
		if( _option.getContStartOptId() == null)
			setDirty(true);

		setEditedObjectAsOriginalContent();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) {

		/* Widgets */
		Combo  cTypeCombo;
		Text   cValueText;

		parent.setLayout(new FillLayout());
		ScrolledComposite sc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.BORDER);
		sc.setExpandHorizontal(true);
		sc.setExpandVertical(true);

		Composite composite = new Composite(sc, SWT.NONE);
		composite.setLayout(new GridLayout(2, false));

		/* Name */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cNameLabel = new Label(composite, SWT.NONE);
		cNameLabel.setText("Name");
		cNameLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cNameText = new Text(composite, SWT.BORDER);
		cNameText.setLayoutData(gd);

		/* Type */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cProcTypeLabel = new Label(composite, SWT.NONE);
		cProcTypeLabel.setText("Type");
		cProcTypeLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		Composite c = new Composite(composite, SWT.CHECK);
		c.setLayoutData(gd);
		c.setLayout(new RowLayout());
		cTypeCombo = new Combo(c, SWT.DROP_DOWN | SWT.READ_ONLY );
		cTypeCombo.setData("type", "optionType");
		cTypeCombo.setItems(TYPES);
		selectOption(_option.getOptionType(), cTypeCombo);

		/* Value */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cValueLabel = new Label(composite, SWT.NONE);
		cValueLabel.setText("Value");
		cValueLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cValueText = new Text(composite, SWT.BORDER);
		cValueText.setLayoutData(gd);
		
		// Finally, calculate the minimum size so the scroll composite knows
		// when to start its role
		sc.setContent(composite);
		sc.setMinSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));

		// Data binding and subscription
		bind("optionName", cNameText);
		bindContStartOptTypeCombo("optionType", cTypeCombo);
		bind("optionValue", cValueText);
	}

	public static void selectOption(ContStartOptType type, Combo cTypeCombo) {
		int index = 0;
		String [] items = cTypeCombo.getItems();
		for(int i = 0; i < items.length; i++) 
		{
			if(type.toString().equals(items[i])) 
			{
				index = i;
				break;
			}
		}
		cTypeCombo.select(index);
	}


	@Override
	protected Object getEditedObject() {
		return _option;
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {
		_origOption = new ContainerStartupOption();
		_origOption.setOptionName(_option.getOptionName());
		_origOption.setOptionType(_option.getOptionType());
		_origOption.setOptionValue(_option.getOptionValue());

		setTitleImage(ImageHelper.getImage(_origOption));
		setPartName(_origOption.getOptionName());
		setTitleToolTip(_origOption.getOptionName());
	}

	protected void resetToOriginalContent() {
		_option.setOptionName(_origOption.getOptionName());
		_option.setOptionType(_origOption.getOptionType());
		_option.setOptionValue(_origOption.getOptionValue());
	}

}
