/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * ContainerEditor.java
 */
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerImplLang;
import alma.acs.tmcdb.ContainerRealTimeType;
import alma.acs.tmcdb.LoggingConfig;
import alma.obops.tmcdbgui.editors.inputs.ContainerEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.BackendConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ContainerConversationUtils;
import alma.obops.tmcdbgui.widgets.LoggingConfigComposite;

/**
 * Editor for a {@link Container} object
 * @author rtobar, Feb 25, 2010
 *
 */


public class ContainerEditor extends TmcdbObjectEditor {

	private Container _cont;
	private Container _originalCont;

	public static final String ID = "container.editor";
	private Text    cNameText;
	private Text    cPathText;
	private Button  cRtCheck;
	private Text    cKerModText;
	private Text    cKerModLocText;
	private Text    cTypeModifText;
	private Button  cStartODCheck;
	private Spinner cKATSpin;
	private Spinner cServerThreadsSpin;
	private Spinner cManagerRetrySpin;
	private Spinner cCallTimeoutSpin;
	private Text    cPingInterText;
	private Button  cRecoveryCheck;
	private Text    cAutoloadLibsText;
	private LoggingConfigComposite loggingConfigComposite;


	public ContainerEditor() { }

	@Override
	public void setFocus()
	{
		cNameText.setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor monitor) {

		// Check for invalid inputs
		if( _cont.getContainerName().trim().equals("") ) {
			setDirty(true);
			MessageDialog.openInformation(getSite().getShell(),
					"No container name specified",
					"A container cannot be saved without a name");
			// TODO: focus on the component name widget
			return;
		}
		if( _cont.getImplLang() == null ) {
			setDirty(true);
			MessageDialog.openInformation(getSite().getShell(),
					"No container language specified",
					"A container cannot be saved without specifying the language in which it is implemented");
			// TODO: focus on the component implLang widget
			return;
		}


		// Set all the fields
		_cont.setKeepAliveTime( cKATSpin.getSelection() );
		_cont.setServerThreads( cServerThreadsSpin.getSelection() );
		_cont.setManagerRetry( cManagerRetrySpin.getSelection() );
		_cont.setCallTimeout( cCallTimeoutSpin.getSelection() );

		// Logging config
		LoggingConfig config = _cont.getLoggingConfig();
		if( _cont.getLoggingConfig() == null ) {
			config = new LoggingConfig();
		}
		
		config.setMinLogLevelDefault( loggingConfigComposite.getMinLogLevelDefault() );
		config.setMinLogLevelLocalDefault( loggingConfigComposite.getMinLogLevelLocalDefault() );
		config.setCentralizedLogger( loggingConfigComposite.getCentralizedLogger() );
		config.setDispatchPacketSize( loggingConfigComposite.getDispatchPacketSize() );
		config.setImmediateDispatchLevel( loggingConfigComposite.getImmediateDispatchLevel() );
		config.setFlushPeriodSeconds( loggingConfigComposite.getFlushPeriodSeconds() );
		config.setMaxLogQueueSize( loggingConfigComposite.getMaxLogQueueSize() );
		config.setMaxLogsPerSecond( loggingConfigComposite.getMaxLogsPerSecond() );
		_cont.setLoggingConfig(config);

		// Persist the object
		try {
			// If it is a new computer, check that there are no computers
			// with this configuration and network name already in the DB
			if( BackendConversationUtils.getInstance().exists(_cont) ) {
				MessageDialog.openWarning(getSite().getShell(),
						"Container already exists",
						"The container '" + _cont.getContainerName() + "' " +
						"already exists in the configuration '" + _cont.getConfiguration().getConfigurationName() + "'");
				// TODO: Focus the network name text field
				return;
			}
			ContainerConversationUtils.getInstance().saveOrUpdateContainer(_cont);
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(getSite().getShell(), "Cannot save Container", "Error while saving Container " + _cont.getContainerName());
			setDirty(true);
			return;
		}

		setEditedObjectAsOriginalContent();
		setDirty(false);
	}
	
	@Override
	public void setDirty(boolean dirty)
	{
		super.setDirty(dirty);
		if(null != loggingConfigComposite) {
			loggingConfigComposite.setDirty(dirty);
		}
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		ContainerEditorInput cei = (ContainerEditorInput)input;
		setInput(input);
		setSite(site);

		_cont = cei.getContainer();
		if( _cont.getContainerId() == null || _cont.getContainerId() <= 0 ) {
			setDirty(true);
		}

		setEditedObjectAsOriginalContent();
	}

	@Override
	public void createPartControl(Composite parent) {

		/* Widgets */
		Button cImplLangCppRadio;
		Button cImplLangJavaRadio;
		Button cImplLangPyRadio;

		Button cRtTypeNoneRadio;
		Button cRtTypeAbmRadio;
		Button cRtTypeCorrRadio;


		parent.setLayout(new FillLayout());
		ScrolledComposite sc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.BORDER);
		sc.setExpandHorizontal(true);
		sc.setExpandVertical(true);

		Composite composite = new Composite(sc, SWT.NONE);
		composite.setLayout(new GridLayout(2, false));

		/* Name */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cNameLabel = new Label(composite, SWT.NONE);
		cNameLabel.setText("Name");
		cNameLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cNameText = new Text(composite, SWT.BORDER);
		cNameText.setLayoutData(gd);

		/* Path */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cPathLabel = new Label(composite, SWT.NONE);
		cPathLabel.setText("Path");
		cPathLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cPathText = new Text(composite, SWT.BORDER);
		cPathText.setLayoutData(gd);

		/* Impl. Lang */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cImplLangLabel = new Label(composite, SWT.NONE);
		cImplLangLabel.setText("Implementation Language");
		cImplLangLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		Composite c = new Composite(composite, SWT.CHECK);
		c.setLayoutData(gd);
		c.setLayout(new RowLayout());
		cImplLangCppRadio = new Button(c, SWT.RADIO);
		cImplLangCppRadio.setText(ContainerImplLang.CPP.toString());
		cImplLangJavaRadio = new Button(c, SWT.RADIO);
		cImplLangJavaRadio.setText(ContainerImplLang.JAVA.toString());
		cImplLangPyRadio = new Button(c, SWT.RADIO);
		cImplLangPyRadio.setText(ContainerImplLang.PY.toString());

		/* Real-time */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cRtLabel = new Label(composite, SWT.NONE);
		cRtLabel.setText("Real-time?");
		cRtLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cRtCheck = new Button(composite, SWT.CHECK);
		cRtCheck.setLayoutData(gd);

		/* Real-time type */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cRtTypeLabel = new Label(composite, SWT.NONE);
		cRtTypeLabel.setText("Real-time type");
		cRtTypeLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		Composite c2 = new Composite(composite, SWT.CHECK);
		c2.setLayoutData(gd);
		c2.setLayout(new RowLayout());
		cRtTypeNoneRadio = new Button(c2, SWT.RADIO);
		cRtTypeNoneRadio.setText(ContainerRealTimeType.NONE.toString());
		cRtTypeAbmRadio = new Button(c2, SWT.RADIO);
		cRtTypeAbmRadio.setText(ContainerRealTimeType.ABM.toString());
		cRtTypeCorrRadio = new Button(c2, SWT.RADIO);
		cRtTypeCorrRadio.setText(ContainerRealTimeType.CORR.toString());

		/* Kernel module */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cKernModLabel = new Label(composite, SWT.NONE);
		cKernModLabel.setText("Kernel Module");
		cKernModLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cKerModText = new Text(composite, SWT.BORDER);
		cKerModText.setLayoutData(gd);

		/* Kernel module location */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cKernModLocLabel = new Label(composite, SWT.NONE);
		cKernModLocLabel.setText("Kernel Module location");
		cKernModLocLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cKerModLocText = new Text(composite, SWT.BORDER);
		cKerModLocText.setLayoutData(gd);

		/* Type modifiers */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cTypeModifLabel = new Label(composite, SWT.NONE);
		cTypeModifLabel.setText("Type modifiers");
		cTypeModifLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cTypeModifText = new Text(composite, SWT.BORDER);
		cTypeModifText.setLayoutData(gd);

		/* Start on demand */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cStartODLabel = new Label(composite, SWT.NONE);
		cStartODLabel.setText("Start on demand?");
		cStartODLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cStartODCheck = new Button(composite, SWT.CHECK);
		cStartODCheck.setLayoutData(gd);

		/* Keep Alive Time */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cKATLabel = new Label(composite, SWT.NONE);
		cKATLabel.setText("Keep Alive Time");
		cKATLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		gd.minimumWidth = 50;
		cKATSpin = new Spinner(composite, SWT.NONE);
		cKATSpin.setMinimum(-1);
		cKATSpin.setMaximum(Integer.MAX_VALUE);
		cKATSpin.setLayoutData(gd);

		/* Server threads */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cServerThreadsLabel = new Label(composite, SWT.NONE);
		cServerThreadsLabel.setText("Server Threads");
		cServerThreadsLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		gd.minimumWidth = 50;
		cServerThreadsSpin = new Spinner(composite, SWT.NONE);
		cServerThreadsSpin.setMinimum(0);
		cServerThreadsSpin.setMaximum(Integer.MAX_VALUE);
		cServerThreadsSpin.setLayoutData(gd);

		/* Manager Retry */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cManagerRetryLabel = new Label(composite, SWT.NONE);
		cManagerRetryLabel.setText("Manager Retry");
		cManagerRetryLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		gd.minimumWidth = 50;
		cManagerRetrySpin = new Spinner(composite, SWT.NONE);
		cManagerRetrySpin.setMinimum(0);
		cManagerRetrySpin.setMaximum(Integer.MAX_VALUE);
		cManagerRetrySpin.setLayoutData(gd);

		/* Call Timeout */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cCallTimeoutLabel = new Label(composite, SWT.NONE);
		cCallTimeoutLabel.setText("Call Timeout");
		cCallTimeoutLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		gd.minimumWidth = 50;
		cCallTimeoutSpin = new Spinner(composite, SWT.NONE);
		cCallTimeoutSpin.setMinimum(0);
		cCallTimeoutSpin.setMaximum(Integer.MAX_VALUE);
		cCallTimeoutSpin.setLayoutData(gd);

		/* Ping Interval */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cPingIntervalLabel = new Label(composite, SWT.NONE);
		cPingIntervalLabel.setText("Ping Interval");
		cPingIntervalLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cPingInterText = new Text(composite, SWT.BORDER);
		cCallTimeoutSpin.setMinimum(0);
		cCallTimeoutSpin.setMaximum(Integer.MAX_VALUE);
		cPingInterText.setLayoutData(gd);

		/* Recovery */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cRecoveryLabel = new Label(composite, SWT.NONE);
		cRecoveryLabel.setText("Recovery?");
		cRecoveryLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cRecoveryCheck = new Button(composite, SWT.CHECK);
		cRecoveryCheck.setLayoutData(gd);

		/* Auto-load Shared libs */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cAutoloadLibsLabel = new Label(composite, SWT.NONE);
		cAutoloadLibsLabel.setText("Auto-load Shared libs");
		cAutoloadLibsLabel.setLayoutData(gd);

		gd = new GridData(GridData.FILL, SWT.FILL, true, false);
		gd.horizontalIndent = 20;
		cAutoloadLibsText = new Text(composite, SWT.BORDER | SWT.WRAP | SWT.V_SCROLL );

		GC gc = new GC(cAutoloadLibsText);
		try
		{
		    gc.setFont(cAutoloadLibsText.getFont());
		    FontMetrics fm = gc.getFontMetrics();

		    /* Set the height to 5 rows of characters */
		    gd.heightHint = 5 * fm.getHeight();
		}
		finally
		{
		    gc.dispose();
		}

		cAutoloadLibsText.setLayoutData(gd);

		// Data binding
		bind( "containerName", cNameText );
		bind( "path", cPathText );
		bind( "implLang", new ContainerImplLang[] {ContainerImplLang.CPP, ContainerImplLang.JAVA, ContainerImplLang.PY}, 
			cImplLangCppRadio, cImplLangJavaRadio, cImplLangPyRadio );
		bind( "realTimeType", new ContainerRealTimeType[] {ContainerRealTimeType.NONE, ContainerRealTimeType.ABM, ContainerRealTimeType.CORR}, 
			cRtTypeNoneRadio, cRtTypeAbmRadio, cRtTypeCorrRadio );
		bind( "kernelModule", cKerModText );
		bind( "kernelModuleLocation", cKerModLocText );
		bind( "typeModifiers", cTypeModifText );
		bind( "startOnDemand", cStartODCheck );
		bind( "recovery", cRecoveryCheck );
		bind( "autoloadSharedLibs", cAutoloadLibsText );
		bind( "pingInterval", cPingInterText );
		bind( "realTime", cRtCheck );

		// TODO: bind these other values as well
		cKATSpin.setSelection( nullSafeInteger(_cont.getKeepAliveTime(), -1) );
		cServerThreadsSpin.setSelection( nullSafeInteger(_cont.getServerThreads(), 5) );
		cManagerRetrySpin.setSelection( nullSafeInteger(_cont.getManagerRetry(), 10) );
		cCallTimeoutSpin.setSelection( nullSafeInteger(_cont.getCallTimeout(), 30) );

		subscribeToChanges(cKATSpin, cServerThreadsSpin, cManagerRetrySpin, cCallTimeoutSpin);

		// Now go with the logging config widgets
		loggingConfigComposite = new LoggingConfigComposite(composite, SWT.None, _cont.getLoggingConfig() == null ? new LoggingConfig() : _cont.getLoggingConfig());
		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalSpan = 2;
		loggingConfigComposite.setLayoutData(gd);
		loggingConfigComposite.addDirtyListener(this);

		// Finally, calculate the minimum size so the scroll composite knows
		// when to start its role
		sc.setContent(composite);
		sc.setMinSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {

		_originalCont = new Container();
		_originalCont.setContainerName(_cont.getContainerName());
		_originalCont.setPath(_cont.getPath());
		_originalCont.setImplLang(_cont.getImplLang());
		_originalCont.setRealTime(_cont.getRealTime());
		_originalCont.setRealTimeType(_cont.getRealTimeType());
		_originalCont.setKernelModule(_cont.getKernelModule());
		_originalCont.setKernelModuleLocation(_cont.getKernelModuleLocation());
		_originalCont.setTypeModifiers(_cont.getTypeModifiers());
		_originalCont.setStartOnDemand(_cont.getStartOnDemand());
		_originalCont.setKeepAliveTime(_cont.getKeepAliveTime());
		_originalCont.setServerThreads(_cont.getServerThreads());
		_originalCont.setManagerRetry(_cont.getManagerRetry());
		_originalCont.setCallTimeout(_cont.getCallTimeout());
		_originalCont.setPingInterval(_cont.getPingInterval());
		_originalCont.setRecovery(_cont.getRecovery());
		_originalCont.setAutoloadSharedLibs(_cont.getAutoloadSharedLibs());
		if( _cont.getLoggingConfig() != null ) {
			LoggingConfig lc = new LoggingConfig();
			lc.setMinLogLevelDefault(_cont.getLoggingConfig().getMinLogLevelDefault());
			lc.setMinLogLevelLocalDefault(_cont.getLoggingConfig().getMinLogLevelLocalDefault());
			lc.setCentralizedLogger(_cont.getLoggingConfig().getCentralizedLogger());
			lc.setDispatchPacketSize(_cont.getLoggingConfig().getDispatchPacketSize());
			lc.setImmediateDispatchLevel(_cont.getLoggingConfig().getImmediateDispatchLevel());
			lc.setFlushPeriodSeconds(_cont.getLoggingConfig().getFlushPeriodSeconds());
			lc.setMaxLogQueueSize(_cont.getLoggingConfig().getMaxLogQueueSize());
			lc.setMaxLogsPerSecond(_cont.getLoggingConfig().getMaxLogsPerSecond());
			_originalCont.setLoggingConfig(lc);
		}

		if(null != loggingConfigComposite) {
			loggingConfigComposite.setLoggingConfig(_cont.getLoggingConfig());
		}
		setPartName(_originalCont.getContainerName());		
		setTitleImage(ImageHelper.getImage(_originalCont));
		setTitleToolTip(LabelHelper.getFullPath(_originalCont, false));
	}

	public void resetToOriginalContent() {
		_cont.setContainerName(_originalCont.getContainerName());
		_cont.setPath(_originalCont.getPath());
		_cont.setImplLang(_originalCont.getImplLang());
		_cont.setRealTime(_originalCont.getRealTime());
		_cont.setRealTimeType(_originalCont.getRealTimeType());
		_cont.setKernelModule(_originalCont.getKernelModule());
		_cont.setKernelModuleLocation(_originalCont.getKernelModuleLocation());
		_cont.setTypeModifiers(_originalCont.getTypeModifiers());
		_cont.setStartOnDemand(_originalCont.getStartOnDemand());
		_cont.setKeepAliveTime(_originalCont.getKeepAliveTime());
		_cont.setServerThreads(_originalCont.getServerThreads());
		_cont.setManagerRetry(_originalCont.getManagerRetry());
		_cont.setCallTimeout(_originalCont.getCallTimeout());
		_cont.setPingInterval(_originalCont.getPingInterval());
		_cont.setRecovery(_originalCont.getRecovery());
		_cont.setAutoloadSharedLibs(_originalCont.getAutoloadSharedLibs());
		if( _cont.getLoggingConfig() != null && _originalCont.getLoggingConfig() != null ) {
			LoggingConfig lc = _cont.getLoggingConfig();
			lc.setMinLogLevelDefault(_originalCont.getLoggingConfig().getMinLogLevelDefault());
			lc.setMinLogLevelLocalDefault(_originalCont.getLoggingConfig().getMinLogLevelLocalDefault());
			lc.setCentralizedLogger(_originalCont.getLoggingConfig().getCentralizedLogger());
			lc.setDispatchPacketSize(_originalCont.getLoggingConfig().getDispatchPacketSize());
			lc.setImmediateDispatchLevel(_originalCont.getLoggingConfig().getImmediateDispatchLevel());
			lc.setFlushPeriodSeconds(_originalCont.getLoggingConfig().getFlushPeriodSeconds());
			lc.setMaxLogQueueSize(_originalCont.getLoggingConfig().getMaxLogQueueSize());
			lc.setMaxLogsPerSecond(_originalCont.getLoggingConfig().getMaxLogsPerSecond());
		}
		
	}

	@Override
	protected Object getEditedObject() {
		return _cont;
	}

}
