/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Layout;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.hibernate.criterion.MatchMode;

import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.domain.IModelChangePublisher;
import alma.obops.tmcdbgui.editors.inputs.ConfigurationEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.StartupScenariosView;
import alma.obops.tmcdbgui.widgets.ConfigurationAttributesComposite;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;
import alma.tmcdb.domain.ArrayReference;
import alma.tmcdb.domain.HwConfiguration;

public class ConfigurationEditor extends TmcdbObjectEditorPart implements IModelChangePublisher, DirtyListener, StatusListener
{
	private static List<ConfigurationEditor> openEditors = new ArrayList<ConfigurationEditor>();
	
	private boolean dirty;
	public static final String ID = "configuration.editor";
	private HwConfiguration configuration;
    private List<IModelChangeListener> modelChangeListeners = new ArrayList<IModelChangeListener>();
    private ConfigurationAttributesComposite composite;
    private boolean isComplete;
    
    public static List<ConfigurationEditor> getOpenEditors()
    {
    	return openEditors;
    }
    
    @Override 
    public void dispose() 
    {
		openEditors.remove(this);
    	this.composite.dispose();
    	super.dispose();
		this.composite = null;
    }

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		ConfigurationEditorInput configEdInput = (ConfigurationEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(configEdInput.getName());
		this.configuration = configEdInput.getConfiguration();
	}
	
	@Override
	public void setInput( IEditorInput input ) 
    {
		super.setInput(input);
		ConfigurationEditorInput configEdInput = ((ConfigurationEditorInput)input);
		this.configuration = configEdInput.getConfiguration();
		if(null != this.composite) {
			this.composite.setConfiguration(configuration);
		}
		this.modelChangeListeners.clear();
		this.addModelChangeListener(configEdInput.getModelChangeListener());
    }

	 @Override
	 public void createPartControl(Composite parent) {
		 Layout layout = new GridLayout( 1, false );
		 parent.setLayout( layout );
		 composite = new ConfigurationAttributesComposite(parent, SWT.NONE, this, configuration, this);
		 openEditors.add(this);
	 }
	 
	 private boolean configAlreadyExists() 
	 {
		 boolean configAlreadyExists = false;
		 if(!configuration.getName().equals(composite.getConfigurationName()))
		 {
			 List<HwConfiguration> configs;
			 try {
				 configs = HwConfigurationConversationUtils.getInstance().findConfigurationsByName(composite.getConfigurationName(), MatchMode.EXACT);
			 } catch (Exception e1) {
				 throw new RuntimeException("Unable to query configurations" + e1);
			 }
			 if(configs.size() > 0) 
			 {
				 configAlreadyExists = true;
			 }
		 }
		 return configAlreadyExists;
	 }

	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		if(configAlreadyExists()) {
			GuiUtils.showErrorDialog(getSite().getShell(), "Could not save changes", "Configuration with name '" + configuration.getName() + "' already exists");
		}
		else if(this.isComplete)
		{
			if(configuration.getArrayReference() == null) {
				configuration.setArrayReference(new ArrayReference());
			}
			String configName = composite.getConfigurationName();
			String configDescription = composite.getDescription();
			String configFullName = composite.getFullName();
			String configTelescopeName = composite.getTelescopeName();
			HwConfiguration globalConfig = composite.getGlobalConfiguration();
			Boolean configActive = composite.getActive();
			Double arrayPositionX = composite.getArrayReference().getX();
			Double arrayPositionY = composite.getArrayReference().getY();
			Double arrayPositionZ = composite.getArrayReference().getZ();
			
			if((!configName.equals(configuration.getName()) ||
					 !configDescription.equals(configuration.getDescription()) ||
					 !configFullName.equals(configuration.getFullName()) ||
					 !safeEquals(globalConfig, configuration.getGlobalConfiguration()) ||
					 !configTelescopeName.equals(configuration.getTelescopeName()) ||
					 !configActive.equals(configuration.getActive()) ||
					 !arrayPositionX.equals(configuration.getArrayReference().getX()) ||
					 !arrayPositionY.equals(configuration.getArrayReference().getY()) ||
					 !arrayPositionZ.equals(configuration.getArrayReference().getZ())))
			{
				try {
					configuration.setTelescopeName(configTelescopeName);
					configuration.getSwConfiguration().setConfigurationName(configName);
					configuration.setGlobalConfiguration(globalConfig);
					configuration.getSwConfiguration().setFullName(configFullName);
					configuration.getSwConfiguration().setDescription(configDescription);
					configuration.getSwConfiguration().setActive(configActive);
					configuration.getArrayReference().setX(arrayPositionX);
					configuration.getArrayReference().setY(arrayPositionY);
					configuration.getArrayReference().setZ(arrayPositionZ);
					HwConfigurationConversationUtils.getInstance().updateConfiguration(configuration);
					this.setPartName(configuration.getName());
					this.composite.setConfiguration(configuration);
					this.modelShouldBeReloaded();
				} catch (Exception e) {
					GuiUtils.showErrorDialog(getSite().getShell(), "Could not save changes", e.getMessage());
					e.printStackTrace();
				}
			}
		}
		else {
			this.setContentDescription("Please specify all attributes");
		}
	}

	private boolean safeEquals(HwConfiguration conf1,
			HwConfiguration conf2) 
	{
		boolean retVal = false;
		
		if(conf1 == null && conf2 == null) {
			retVal = true;
		} 
		else if(conf1 == null && conf2 != null) {
			retVal = false;
		} 
		else if(conf1 != null && conf2 == null) {
			retVal = false;
		} 
		else if(conf1 != null && conf2 != null) 
		{
			if(conf1.getId().equals(conf2.getId())) 
			{
				retVal = true;
			} else 
			{
				retVal = false;
			}
		}
		
		return retVal;
	}

	@Override
	public void modelChanged() 
	{
		for(IModelChangeListener listener: modelChangeListeners )
		{
			// make sure configurations view is refreshed
			listener.internalModelChange();
			
			// and make sure hw startups view refreshes to reflect the new name
			StartupScenariosView startupsView = (StartupScenariosView) RcpUtils.findView(StartupScenariosView.ID);
			startupsView.refresh();
		}
	}

	@Override
	public void modelShouldBeReloaded() {
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.externalModelChange();
		}
	}
	
	@Override
	public void addModelChangeListener(IModelChangeListener listener) {
		if(null != listener)
		{
			this.modelChangeListeners.add(listener);
		}
	}

	@Override
	public void removeModelChangeListener(IModelChangeListener listener) {
		this.modelChangeListeners.remove(listener);
	}

	@Override
	public void notifyOfCompletion(boolean complete) {
		this.isComplete = complete;
		this.setContentDescription("");
	}

	@Override
	public void updateErrorStatus(String newStatusMessage) 
	{
		String partName = getPartName();
		if(newStatusMessage != null && !newStatusMessage.trim().equals(""))
		{
			if(!partName.startsWith("!")) {
				setPartName("!" + partName);
			}
		} else {
			if(partName.startsWith("!")) {
				String cleanedPartName = partName.substring(1);
				setPartName(cleanedPartName);				
			}
		}
	}

	@Override
	public void doSaveAs() {
		// noop; not supported
	}

	@Override
	public boolean isDirty() {
		return this.dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean dirty) {
		this.dirty = dirty;
		this.firePropertyChange(PROP_DIRTY);
	}

	public HwConfiguration getConfiguration() {
		return this.configuration;
	}
}
