/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * ComputerEditor.java
 */
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.ComputerProcessorType;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.editors.inputs.ComputerEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.BackendConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComputerConversationUtils;
import alma.obops.tmcdbgui.views.SoftwareDeploymentView;

/**
 * Editor for a {@link Computer} object
 * 
 * @author rtobar, Mar 4, 2010
 */


public class ComputerEditor extends TmcdbObjectEditor {

	public static final String ID = "computer.editor";

	/* The actual computer being edited */
	private Computer _comp;

	/* Initial contents, used to fallback */
	private Computer _origComputer;

	private Text   cNameText;
	
	@Override 
	public void setFocus() {
		cNameText.setFocus();
	}

	@Override
	public void doSave(IProgressMonitor monitor) {

		boolean isNewComputer = false;

		if( _comp.getNetworkDeviceId() == null )
			isNewComputer = true;

		// Check for invalid inputs
		if( _comp.getNetworkName() == null || _comp.getNetworkName().trim().equals("") ) {
			MessageDialog.openInformation(getSite().getShell(),
					"No network name specified",
					"A computer cannot be saved without a network name");
			// TODO: Focus the network name text field
			return;
		}
		if( _comp.getProcessorType() == null ) {
			MessageDialog.openInformation(getSite().getShell(),
					"No processor type name specified",
					"A computer cannot be saved without specifying its processor type");
			return;
		}

		// Persist the object
		try {
			// Check that there are no computers with this configuration and network name already in the DB
			if( BackendConversationUtils.getInstance().exists(_comp) ) {
				MessageDialog.openWarning(getSite().getShell(),
						"Computer already exists",
						"The computer '" + _comp.getNetworkName() + "' " +
						"already exists in the configuration '" + _comp.getConfiguration().getConfigurationName() + "'");
				// TODO: Focus the network name text field
				return;
			}
			ComputerConversationUtils.getInstance().saveOrUpdateComputer(_comp);
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(getSite().getShell(), "Cannot save Computer", "Error while saving Computer " + _comp.getName());
			return;
		}

		setEditedObjectAsOriginalContent();
		setDirty(false);

		// If we're adding a new computer, let's refresh the SDV if available
		if( isNewComputer ) {
			SoftwareDeploymentView sdv = (SoftwareDeploymentView)RcpUtils.findView( SoftwareDeploymentView.ID );
			sdv.internalModelChange();
		}
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException {
		ComputerEditorInput cei = (ComputerEditorInput)input;
		setInput(input);
		setSite(site);

		_comp = cei.getComputer();
		if( _comp.getNetworkDeviceId() == null || _comp.getNetworkDeviceId() < 0 )
			setDirty(true);

		setEditedObjectAsOriginalContent();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) {

		/* Widgets */
		Text   cNetworkNameText;
		Text   cLocationText;
		Button cRtCheck;
		Button cDisklessCheck;
		Button cSmpProcRadio;
		Button cUniProcRadio;

		parent.setLayout(new FillLayout());
		ScrolledComposite sc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.BORDER);
		sc.setExpandHorizontal(true);
		sc.setExpandVertical(true);

		Composite composite = new Composite(sc, SWT.NONE);
		composite.setLayout(new GridLayout(2, false));

		/* Name */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cNameLabel = new Label(composite, SWT.NONE);
		cNameLabel.setText("Name");
		cNameLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cNameText = new Text(composite, SWT.BORDER);
		cNameText.setLayoutData(gd);

		/* Network Name */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cNetworkNameLabel = new Label(composite, SWT.NONE);
		cNetworkNameLabel.setText("Network name");
		cNetworkNameLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cNetworkNameText = new Text(composite, SWT.BORDER);
		cNetworkNameText.setLayoutData(gd);

		/* Location */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cLocationLabel = new Label(composite, SWT.NONE);
		cLocationLabel.setText("Location");
		cLocationLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cLocationText = new Text(composite, SWT.BORDER);
		cLocationText.setLayoutData(gd);

		/* Real-time */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cRtLabel = new Label(composite, SWT.NONE);
		cRtLabel.setText("Real-time?");
		cRtLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cRtCheck = new Button(composite, SWT.CHECK);
		cRtCheck.setLayoutData(gd);

		/* Diskless */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cDisklessLabel = new Label(composite, SWT.NONE);
		cDisklessLabel.setText("Is diskless?");
		cDisklessLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		cDisklessCheck = new Button(composite, SWT.CHECK);
		cDisklessCheck.setLayoutData(gd);

		/* Processor type */
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label cProcTypeLabel = new Label(composite, SWT.NONE);
		cProcTypeLabel.setText("Processor type");
		cProcTypeLabel.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.horizontalIndent = 20;
		Composite c = new Composite(composite, SWT.CHECK);
		c.setLayoutData(gd);
		c.setLayout(new RowLayout());
		cUniProcRadio = new Button(c, SWT.RADIO);
		cUniProcRadio.setText(ComputerProcessorType.UNI.toString());
		cSmpProcRadio = new Button(c, SWT.RADIO);
		cSmpProcRadio.setText(ComputerProcessorType.SMP.toString());

		// Finally, calculate the minimum size so the scroll composite knows
		// when to start its role
		sc.setContent(composite);
		sc.setMinSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));

		// Data binding and subscription
		bind("name", cNameText);
		bind("networkName", cNetworkNameText);
		bind("physicalLocation", cLocationText);
		bind("realTime", cRtCheck);
		bind("diskless", cDisklessCheck);
		bind("processorType", new ComputerProcessorType[] { ComputerProcessorType.UNI, ComputerProcessorType.SMP }, cUniProcRadio, cSmpProcRadio);

	}


	@Override
	protected Object getEditedObject() {
		return _comp;
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {
		_origComputer = new Computer();
		_origComputer.setNetworkName(_comp.getNetworkName());
		_origComputer.setName(_comp.getName());
		_origComputer.setPhysicalLocation(_comp.getPhysicalLocation());
		_origComputer.setRealTime(_comp.getRealTime());
		_origComputer.setDiskless(_comp.getDiskless());
		_origComputer.setProcessorType(_comp.getProcessorType());

		setTitleImage(ImageHelper.getImage(_origComputer));
		setPartName(_origComputer.getNetworkName());
		setTitleToolTip( LabelHelper.getComputerLabel(_origComputer) );
	}

	protected void resetToOriginalContent() {

		_comp.setNetworkName(_origComputer.getNetworkName());
		_comp.setName(_origComputer.getName());
		_comp.setPhysicalLocation(_origComputer.getPhysicalLocation());
		_comp.setRealTime(_origComputer.getRealTime());
		_comp.setDiskless(_origComputer.getDiskless());
		_comp.setProcessorType(_origComputer.getProcessorType());
	}

}
