/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * ComponentEditor.java
 */
package alma.obops.tmcdbgui.editors;

import java.beans.PropertyChangeEvent;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.obops.tmcdbgui.editors.inputs.ComponentEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.BackendConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComponentConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.ComponentTypeConversationUtils;
import alma.obops.tmcdbgui.widgets.ComponentEditingComposite;

/**
 * Editor for a {@link Component} object
 *
 * @author rtobar, Mar 1, 2010
 */


public class ComponentEditor extends TmcdbObjectEditor  {

	private Component _comp;
	private Component _originalComp;
	private String _originalIDL;
	private ComponentEditingComposite editingComposite;

	public static final String ID = "component.editor";

	public ComponentEditor() { }
	
	@Override 
	public void setFocus() {
		editingComposite.setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor monitor) {

		boolean createComponentType = false;

		// Check for invalid inputs
		if( _comp.getComponentName() == null || _comp.getComponentName().trim().equals("") ) {
			setDirty(true);
			MessageDialog.openInformation(getSite().getShell(),
					"No component name specified",
					"A component cannot be saved without a component name");
			// TODO: focus on the component name widget
			return;
		}
		if( _comp.getPath() == null || _comp.getPath().trim().equals("") ) {
			setDirty(true);
			MessageDialog.openInformation(getSite().getShell(),
					"No path specified",
					"A component cannot be saved without a path (use / to specify a component without path)");
			// TODO: focus on the component path widget
			return;
		}
		if( _comp.getCode() == null || _comp.getCode().trim().equals("") ) {
			setDirty(true);
			MessageDialog.openInformation(getSite().getShell(),
					"No component code specified",
					"A component cannot be saved without specifying its code");
			// TODO: focus on the component code widget
			return;
		}
		if( _comp.getImplLang() == null ) {
			setDirty(true);
			MessageDialog.openInformation(getSite().getShell(),
					"No component language specified",
					"A component cannot be saved without specifying the language in which it is implemented");
			// TODO: focus on the component implLang widget
			return;
		}
		if( _comp.getIsControl() == null ) {
			setDirty(true);
			MessageDialog.openInformation(getSite().getShell(),
					"Missing specification",
					"A component cannot be saved without specifying if it is a CONTROL component or not");
			// TODO: focus on the isControl widget
			return;
		}
		if( _comp.getKeepAliveTime() == null ) {
			setDirty(true);
			MessageDialog.openInformation(getSite().getShell(),
					"No component Keep Alive Time specified",
					"A component cannot be saved without specifying its Keep Alive Time");
			// TODO: focus on the component KAT widget
			return;
		}

		// Oracle doesn't like empty string for XML fields, let's null it if empty
		if( _comp.getXMLDoc() != null && _comp.getXMLDoc().length() == 0 )
			_comp.setXMLDoc(null);


		// Persist the object
		try {
			// 1st: Check that a component with same path/name doesn't exist for this configuration
			if( BackendConversationUtils.getInstance().exists(_comp) ) {
				MessageDialog.openWarning(getSite().getShell(),
						"Component already exists",
						"The component '" + LabelHelper.getFullPath(_comp, false) + "' " +
						"already exists in the configuration '" + _comp.getConfiguration().getConfigurationName() + "'");
				// TODO: Focus the component name text field
				return;
			}

			// 2nd: Check if the component type exists. If not, ask whether the user wants to create it
			//      or correct it. Here we are using a ilike() comparison, and we assume that we won't find
			//      more than 1 result with this.
			String possiblyEditedIdl = _comp.getComponentType().getIDL();
			List<ComponentType> cts = ComponentTypeConversationUtils.getInstance().findByNameExactMatch( possiblyEditedIdl );
			if(cts.size() == 1) 
			{
				// There was an existing componenttype for the (possibly edited) IDL string.
				
				// So, reset the IDL string for the component's current componenttype (in case it had been edited).
				// In other words, any edits to the IDL string of the componenttype 
				// should _not_ remain (rather, we reassign the componenttype reference, below, to the actual
				// componenttype that has that IDL string, which we found in our query)
				_comp.getComponentType().setIDL(_originalIDL);
				
				// reassign the componenttype reference for our component (to reflect the desired IDL string)
				// to the componenttype with that IDL string, which was found in our query:
				_comp.setComponentType(cts.get(0));
			}
			else {
				// the component type with the IDL string given by the user doesn't exist; 
				// so, ask user if she wants to create it:
				if( MessageDialog.openConfirm(getSite().getShell(), "New IDL type",
					"The IDL type '" + possiblyEditedIdl + "' does not exist. Do you want to create it?") ) {
					_comp.setComponentType( new ComponentType() );
					_comp.getComponentType().setIDL(possiblyEditedIdl);
					createComponentType = true;
				}
			}

			// 3rd: finally save the component (and component type if needed)
			if( createComponentType ) {
				ComponentTypeConversationUtils.getInstance().saveOrUpdate(_comp.getComponentType());
			}
			ComponentConversationUtils.getInstance().saveOrUpdateComponent(_comp);
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(getSite().getShell(), "Cannot save Component", "Error while saving Component "
			   + _comp.getComponentName() + ": " + e);
			setDirty(true);
			return;
		}

		setEditedObjectAsOriginalContent();
		setDirty(false);
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		ComponentEditorInput cei = (ComponentEditorInput)input;
		setInput(input);
		setSite(site);

		// Hydrate component type if necessary (when editing existing components)
		_comp = cei.getComponent();
		if( _comp.getComponentId() != null && _comp.getComponentId() >= 0 ) {
			try {
				ComponentConversationUtils.getInstance().hydrateComponentType(_comp);
			} catch (Exception e) {
				throw new PartInitException("Error while hydrating Component Type", e);
			}
		}
		else
			setDirty(true);

		setEditedObjectAsOriginalContent();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		ScrolledComposite sc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.BORDER);
		sc.setExpandHorizontal(true);
		sc.setExpandVertical(true);
		
		editingComposite = new ComponentEditingComposite(this, sc, SWT.NONE, _comp, this.getSite().getWorkbenchWindow(), null);
		
		// Finally, calculate the minimum size so the scroll composite knows
		// when to start its role
		sc.setContent(editingComposite);
		sc.setMinSize(editingComposite.computeSize(SWT.DEFAULT, SWT.DEFAULT));
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {

		_originalComp = new Component();
		_originalComp.setComponentName(_comp.getComponentName());
		_originalComp.setPath(_comp.getPath());
		_originalComp.setCode(_comp.getCode());
		_originalComp.setImplLang(_comp.getImplLang());
		_originalComp.setRealTime(_comp.getRealTime());
		_originalComp.setIsControl(_comp.getIsControl());
		_originalComp.setIsAutostart(_comp.getIsAutostart());
		_originalComp.setIsDefault(_comp.getIsDefault());
		_originalComp.setIsStandaloneDefined(_comp.getIsStandaloneDefined());
		_originalComp.setKeepAliveTime(_comp.getKeepAliveTime());
		_originalComp.setMinLogLevel(_comp.getMinLogLevel());
		_originalComp.setMinLogLevelLocal(_comp.getMinLogLevelLocal());
		_originalComp.setXMLDoc(_comp.getXMLDoc());
		_originalComp.setComponentType(_comp.getComponentType());
		_originalComp.setContainer(_comp.getContainer());
		_originalIDL = _comp.getComponentType().getIDL();

		setPartName( _originalComp.getComponentName() );
		setTitleImage( ImageHelper.getImage(_originalComp) );
		setTitleToolTip( LabelHelper.getFullPath(_originalComp, false) );
	}

	public void resetToOriginalContent() {

		_comp.setComponentName(_originalComp.getComponentName());
		_comp.setPath(_originalComp.getPath());
		_comp.setCode(_originalComp.getCode());
		_comp.setImplLang(_originalComp.getImplLang());
		_comp.setRealTime(_originalComp.getRealTime());
		_comp.setIsControl(_originalComp.getIsControl());
		_comp.setIsAutostart(_originalComp.getIsAutostart());
		_comp.setIsDefault(_originalComp.getIsDefault());
		_comp.setIsStandaloneDefined(_originalComp.getIsStandaloneDefined());
		_comp.setKeepAliveTime(_originalComp.getKeepAliveTime());
		_comp.setMinLogLevel(_originalComp.getMinLogLevel());
		_comp.setMinLogLevelLocal(_originalComp.getMinLogLevelLocal());
		_comp.setXMLDoc(_originalComp.getXMLDoc());
		_comp.setContainer(_originalComp.getContainer());
		_comp.setComponentType(_originalComp.getComponentType());
		_comp.getComponentType().setIDL(_originalIDL);
	}

	@Override
	protected Object getEditedObject() {
		return _comp;
	}
	
	@Override
	public void propertyChange(PropertyChangeEvent evt) 
	{
		super.propertyChange(evt);
		if( evt.getPropertyName().equals("implLang") ) {
			setTitleImage( ImageHelper.getImage(_comp) );
		}
	}
}
