/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) NRAO - National Radio Astronomy Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/

package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.acs.tmcdb.ChannelMapping;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.widgets.ChannelMappingEventChannelNameComposite;
import alma.obops.tmcdbgui.editors.inputs.ChannelMappingEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.ChannelMappingConversationUtils;
import alma.obops.tmcdbgui.views.SoftwareDeploymentView;

public class ChannelMappingEditor extends TmcdbObjectEditor
{
	public static final String ID = "channelmapping.editor";
	private ChannelMapping channelMapping;
	private ChannelMapping origChannelMapping;
	private ChannelMappingEventChannelNameComposite composite;

	@Override 
	public void setFocus() {
		composite.setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		boolean isNewOption = false;

		if( channelMapping.getChannelMappingId() == null ) {
			isNewOption = true;
		}
		
		channelMapping.setName(composite.getChannelName());
		channelMapping.setNotificationService(composite.getNotificationService());

		// Check for invalid inputs
		if(  (channelMapping.getName() == null || channelMapping.getName().toString().trim().equals(""))  ||
			   (channelMapping.getNotificationService() == null || channelMapping.getNotificationService().trim().equals("")) )
		{
			MessageDialog.openInformation(getSite().getShell(),
					"Please specify all fields",
					"NS Channel Mapping cannot be saved without all fields defined");
			return;
		}

		// Persist the object
		try {
			ChannelMappingConversationUtils.getInstance().saveOrUpdateChannelMapping(channelMapping);
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(getSite().getShell(), "Cannot save NS Channel Mapping", "Error while saving ChannelMapping: " + channelMapping.getName());
			return;
		}

		setEditedObjectAsOriginalContent();
		setDirty(false);
		composite.setDirty(false);

		// If we're adding a new containerstartupoption, let's refresh the SDV if available
		if( isNewOption ) {
			SoftwareDeploymentView sdv = (SoftwareDeploymentView)RcpUtils.findView( SoftwareDeploymentView.ID );
			sdv.internalModelChange();
		}
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		ScrolledComposite sc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.BORDER);
		sc.setExpandHorizontal(true);
		sc.setExpandVertical(true);

		// Finally, calculate the minimum size so the scroll composite knows
		// when to start its role
		composite = new ChannelMappingEventChannelNameComposite(sc, SWT.NONE);
		composite.setInput((ChannelMappingEditorInput) this.getEditorInput());
		composite.addDirtyListener(this);
		sc.setContent(composite);
		sc.setMinSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));
	}

	@Override
	protected Object getEditedObject() {
		return channelMapping;
	}

	@Override
	protected void resetToOriginalContent() {
		channelMapping.setName(origChannelMapping.getName());
		channelMapping.setNotificationService(origChannelMapping.getNotificationService());
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {
		origChannelMapping = new ChannelMapping();
		origChannelMapping.setName(channelMapping.getName());
		origChannelMapping.setNotificationService(channelMapping.getNotificationService());

		setTitleImage(ImageHelper.getImage(origChannelMapping));
		String partName = LabelHelper.getChannelMappingLabel(origChannelMapping);
		setPartName(partName);
		setTitleToolTip(partName);
	}
	
	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		ChannelMappingEditorInput cei = (ChannelMappingEditorInput)input;
		setInput(input);
		setSite(site);

		channelMapping = cei.getChannelMapping();
		if( channelMapping.getChannelMappingId() == null) {
			setDirty(true);
		}

		setEditedObjectAsOriginalContent();
	}
}
