/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.beans.PropertyChangeEvent;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.acs.tmcdb.BACIProperty;
import alma.obops.tmcdbgui.editors.inputs.BACIPropertyEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.BACIPropertyConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.BackendConversationUtils;
import alma.obops.tmcdbgui.widgets.BACIPropertyEditingComposite;

/**
 * Editor for baci properties.
 * @author sharring
 */
public class BACIPropertyEditor extends TmcdbObjectEditor
{
	private BACIProperty baciProp;
	private BACIProperty originalProp;
	private BACIPropertyEditingComposite composite; 

	public static final String ID = "baciproperty.editor";

	public BACIPropertyEditor() { }
	
	@Override
	public void setFocus() {
		composite.setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor monitor) {

		// Check for invalid inputs
		if( invalidInput(baciProp.getPropertyName(), "property name") ||
		    invalidInput(baciProp.getDescription(), "description") ||
		    invalidInput(baciProp.getFormat(), "format") ||
		    invalidInput(baciProp.getUnits(), "units") ||
		    invalidInput(baciProp.getResolution(), "resolution") ||
		    invalidInput(baciProp.getArchive_priority(), "archive priority") ||
		    invalidInput(baciProp.getArchive_max_int(), "archive max int") ||
		    invalidInput(baciProp.getArchive_min_int(), "archive min int") ||
		    invalidInput(baciProp.getArchive_mechanism(), "archive mechanism") ||
		    invalidInput(baciProp.getArchive_suppress(), "archive supress") ||
		    invalidInput(baciProp.getDefault_timer_trig(), "default timer trig") ||
		    invalidInput(baciProp.getMin_timer_trig(), "min timer trig") ||
		    invalidInput(baciProp.getInitialize_devio(), "initialize DevIO") ||
		    invalidInput(baciProp.getDefault_value(), "default value") ||
		    invalidInput(baciProp.getArchive_delta(), "archive delta") )
			return;

		// Persist the object
		try {
			// 1st: Check that a baci property with same name doesn't exist for this component
			if( BackendConversationUtils.getInstance().exists(baciProp) ) {
				MessageDialog.openWarning(getSite().getShell(),
						"BACIProperty already exists",
						"The baciproperty '" + LabelHelper.getFullPath(baciProp, false) + "' " +
						"already exists in the component '" + baciProp.getComponent().getComponentName() + "'");
				// TODO: Focus the component name text field
				return;
			}

			BACIPropertyConversationUtils.getInstance().saveOrUpdate(baciProp);
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(getSite().getShell(), "Cannot save Component", "Error while saving Component "
			   + baciProp.getPropertyName() + ": " + e);
			setDirty(true);
			return;
		}

		setEditedObjectAsOriginalContent();
		setDirty(false);
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		BACIPropertyEditorInput bpei = (BACIPropertyEditorInput)input;
		baciProp = bpei.getBACIProperty();
		setInput(input);
		setSite(site);

		if( baciProp.getBACIPropertyId() == null || baciProp.getBACIPropertyId() < 0 )
			setDirty(true);

		setEditedObjectAsOriginalContent();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		ScrolledComposite sc = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER);
		sc.setExpandHorizontal(true);
		sc.setExpandVertical(true);

		composite = new BACIPropertyEditingComposite(this, sc, SWT.NONE, baciProp, null);
		
		sc.setContent(composite);
		sc.setMinSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));
	}

	@Override
	protected void setEditedObjectAsOriginalContent() 
	{
		originalProp = new BACIProperty();
		
		originalProp.setAlarm_fault_family(baciProp.getAlarm_fault_family());
		originalProp.setAlarm_fault_member(baciProp.getAlarm_fault_member());
		
		originalProp.setAlarm_high_off(baciProp.getAlarm_high_off());
		originalProp.setAlarm_high_on(baciProp.getAlarm_high_on());
		
		originalProp.setAlarm_level(baciProp.getAlarm_level());
		
		originalProp.setAlarm_low_off(baciProp.getAlarm_low_off());
		originalProp.setAlarm_low_on(baciProp.getAlarm_low_on());
		
		originalProp.setAlarm_off(baciProp.getAlarm_off());
		originalProp.setAlarm_on(baciProp.getAlarm_on());
		
		originalProp.setAlarm_timer_trig(baciProp.getAlarm_timer_trig());
		originalProp.setArchive_delta(baciProp.getArchive_delta());
		originalProp.setArchive_max_int(baciProp.getArchive_max_int());
		originalProp.setArchive_min_int(baciProp.getArchive_min_int());
		originalProp.setArchive_suppress(baciProp.getArchive_suppress());
		originalProp.setArchive_mechanism(baciProp.getArchive_mechanism());
		originalProp.setArchive_priority(baciProp.getArchive_priority());
		
		originalProp.setBitDescription(baciProp.getBitDescription());
		originalProp.setCondition(baciProp.getCondition());
		originalProp.setData(baciProp.getData());
		originalProp.setDefault_timer_trig(baciProp.getDefault_timer_trig());
		originalProp.setDefault_value(baciProp.getDefault_value());
		originalProp.setDescription(baciProp.getDescription());
		originalProp.setFormat(baciProp.getFormat());
		originalProp.setGraph_max(baciProp.getGraph_max());
		originalProp.setGraph_min(baciProp.getGraph_min());
		originalProp.setInitialize_devio(baciProp.getInitialize_devio());
		originalProp.setMax_value(baciProp.getMax_value());
		originalProp.setMin_delta_trig(baciProp.getMin_delta_trig());
		originalProp.setMin_step(baciProp.getMin_step());
		originalProp.setMin_timer_trig(baciProp.getMin_timer_trig());
		originalProp.setMin_value(baciProp.getMin_value());
		originalProp.setPropertyName(baciProp.getPropertyName());
		originalProp.setResolution(baciProp.getResolution());
		originalProp.setStatesDescription(baciProp.getStatesDescription());
		originalProp.setUnits(baciProp.getUnits());
		originalProp.setWhenCleared(baciProp.getWhenCleared());
		originalProp.setWhenSet(baciProp.getWhenSet());
		
		setPartName( originalProp.getPropertyName());
		setTitleImage( ImageHelper.getImage(originalProp) );
		setTitleToolTip( LabelHelper.getFullPath(originalProp, false) );
	}

	public void resetToOriginalContent() {
		baciProp.setAlarm_fault_family(originalProp.getAlarm_fault_family());
		baciProp.setAlarm_fault_member(originalProp.getAlarm_fault_member());
		
		baciProp.setAlarm_high_off(originalProp.getAlarm_high_off());
		baciProp.setAlarm_high_on(originalProp.getAlarm_low_on());
		
		baciProp.setAlarm_level(originalProp.getAlarm_level());
		
		baciProp.setAlarm_low_off(originalProp.getAlarm_low_off());
		baciProp.setAlarm_low_on(originalProp.getAlarm_low_on());
		
		baciProp.setAlarm_off(originalProp.getAlarm_off());
		baciProp.setAlarm_on(originalProp.getAlarm_on());
		
		baciProp.setAlarm_timer_trig(originalProp.getAlarm_timer_trig());
		baciProp.setArchive_delta(originalProp.getArchive_delta());
		baciProp.setArchive_max_int(originalProp.getArchive_max_int());
		baciProp.setArchive_min_int(originalProp.getArchive_min_int());
		baciProp.setArchive_suppress(originalProp.getArchive_suppress());
		baciProp.setArchive_mechanism(originalProp.getArchive_mechanism());
		baciProp.setArchive_priority(originalProp.getArchive_priority());
		
		baciProp.setBitDescription(originalProp.getBitDescription());
		baciProp.setCondition(originalProp.getCondition());
		baciProp.setData(originalProp.getData());
		baciProp.setDefault_timer_trig(originalProp.getDefault_timer_trig());
		baciProp.setDefault_value(originalProp.getDefault_value());
		baciProp.setDescription(originalProp.getDescription());
		baciProp.setFormat(originalProp.getFormat());
		baciProp.setGraph_max(originalProp.getGraph_max());
		baciProp.setGraph_min(originalProp.getGraph_min());
		baciProp.setInitialize_devio(originalProp.getInitialize_devio());
		baciProp.setMax_value(originalProp.getMax_value());
		baciProp.setMin_delta_trig(originalProp.getMin_delta_trig());
		baciProp.setMin_step(originalProp.getMin_step());
		baciProp.setMin_timer_trig(originalProp.getMin_timer_trig());
		baciProp.setMin_value(originalProp.getMin_value());
		baciProp.setPropertyName(originalProp.getPropertyName());
		baciProp.setResolution(originalProp.getResolution());
		baciProp.setStatesDescription(originalProp.getStatesDescription());
		baciProp.setUnits(originalProp.getUnits());
		baciProp.setWhenCleared(originalProp.getWhenCleared());
		baciProp.setWhenSet(originalProp.getWhenSet());

	}

	@Override
	protected Object getEditedObject() {
		return baciProp;
	}

	@Override
	public void propertyChange(PropertyChangeEvent evt) {
		super.propertyChange(evt);
		if( evt.getPropertyName().equals("propertyName") )
			setPartName( (String) evt.getNewValue() );
	}
}
