/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.logging.Logger;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.sax.SAXSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.beans.PojoObservables;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.databinding.swt.SWTObservables;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import alma.obops.tmcdbgui.TmcdbGui;
import alma.obops.tmcdbgui.editors.inputs.AssemblyEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AssemblyConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.AssemblyTypeConversationUtils;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.HwSchema;

public class AssemblyEditor extends TmcdbObjectEditor 
{
	public static final String ID = "assembly.editor";
	
	private Assembly assembly;
	private Assembly originalAssembly;
    private DataBindingContext ctx;
	
	private Text dataTextArea;
	
	@Override
	protected Object getEditedObject() {
		return assembly;
	}
	
	@Override
	public void setFocus()
	{
		dataTextArea.setFocus();
	}
	
	@Override 
	public void setInput(IEditorInput input)
	{
		super.setInput(input);
	}
	
	@Override
	protected void resetToOriginalContent() {
		assembly.setSerialNumber(originalAssembly.getSerialNumber());
		assembly.setData(originalAssembly.getData());
		assembly.setAssemblyType(originalAssembly.getAssemblyType());
		assembly.setConfiguration(originalAssembly.getConfiguration());
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {
		originalAssembly = new Assembly(assembly.getSerialNumber(), assembly.getData(), assembly.getAssemblyType());
		originalAssembly.setConfiguration(assembly.getConfiguration());
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		AssemblyEditorInput assemblyEdInput = (AssemblyEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(assemblyEdInput.getName());
		this.assembly = assemblyEdInput.getAssembly();
		setEditedObjectAsOriginalContent();
	}

	@Override
	public void doSave(IProgressMonitor arg0) 
	{
		String assemblySerialNum = assembly.getSerialNumber();
		String assemblyData = assembly.getData();
		AssemblyType assemblyType = assembly.getAssemblyType();
		
		if( (!assemblySerialNum.equals(originalAssembly.getSerialNumber()) ||
						!assemblyData.equals(originalAssembly.getData()) ||
						!assemblyType.equals(originalAssembly.getAssemblyType())) )
		{
			boolean valid = validateDocument();
			if(valid)
			{
				try {
					AssemblyConversationUtils.getInstance().saveOrUpdateAssembly(assembly);
					setEditedObjectAsOriginalContent();
					setDirty(false);
				}
				catch (Exception e) {
					GuiUtils.showErrorDialog(getSite().getShell(), "Could not save changes", e.getMessage());
					e.printStackTrace();
				}
			}
		}
	}

	private boolean validateDocument()
	{
		boolean retVal = false;
		
		Logger logger = TmcdbGui.getLogger();

		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		SchemaFactory schemaFactory = SchemaFactory.newInstance(javax.xml.XMLConstants.W3C_XML_SCHEMA_NS_URI);

		String schemaString = "";
		try {
		   List<HwSchema> schemas = AssemblyTypeConversationUtils.getInstance().findHwSchemasForAssemblyType(assembly.getAssemblyType());
		   if( schemas != null && schemas.size() > 0 )
			   schemaString = schemas.get(0).getSchema();
		} catch(Exception ex) {
			String msg = "Unable to find assembly or its corresponding schema";
			logger.severe(msg);
		} 
		
		File tmpFile = null;

		try {
			tmpFile = File.createTempFile(assembly.getSerialNumber() + "-" + System.currentTimeMillis(), ".xsd");
			FileOutputStream fos = new FileOutputStream(tmpFile.getAbsolutePath());
			fos.write(schemaString.getBytes());
			fos.close();
			Schema schema = schemaFactory.newSchema(tmpFile);

			factory.setNamespaceAware(true);
			factory.setValidating(false);
			factory.setSchema(schema);

			DocumentBuilder parser = factory.newDocumentBuilder();

			ByteArrayInputStream bs = new ByteArrayInputStream(assembly.getData().getBytes());
			parser.parse(bs);

			Validator validator = schema.newValidator();
			validator.setErrorHandler(new LineNumberErrorHandler());
			validator.validate(new SAXSource(new InputSource(new ByteArrayInputStream(assembly.getData().getBytes()))));
			retVal = true;
		} 
		catch (FileNotFoundException e) {
			String msg = "Temporary schema file not found; could not validate xml";
			logger.severe(msg);
			GuiUtils.showErrorDialog(getSite().getShell(), "Could not save changes", msg);
			e.printStackTrace();
		} catch (IOException e) {
			String msg = "IOException attempting to validate xml";
			logger.severe(msg);
			e.printStackTrace();
		} 
		catch(SAXParseException e) 
		{
			// noop; handled by our error handler already...
		}
		catch (SAXException e) {
			String msg = "Sax exception attempting to validate xml; could not save changes to assembly";
			GuiUtils.showErrorDialog(getSite().getShell(), "Could not save changes", msg);
			logger.warning(msg);
			e.printStackTrace();
		} catch (ParserConfigurationException e) {
			String msg = "Parser not properly configured while attempting to validate xml";
			logger.severe(msg);
			GuiUtils.showErrorDialog(getSite().getShell(), "Could not save changes", msg);
			e.printStackTrace();
		} catch (Exception e) {
			String msg = "Unexpected exception attempting to validate xml";
			logger.severe(msg);
			GuiUtils.showErrorDialog(getSite().getShell(), "Could not save changes", msg);
			e.printStackTrace();
		}
		finally 
		{
			if(tmpFile != null) 
			{
				if( !tmpFile.delete() )
					logger.warning("Temporary file '" + tmpFile.getAbsolutePath() + "' couldn't be deleted");
			}
		}
		return retVal;
	}
	
	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		dataTextArea = new Text(parent, SWT.MULTI | SWT.V_SCROLL | SWT.H_SCROLL);
		initializeDataBinding();
	}

	private void initializeDataBinding() {
		if( ctx != null ) {
			ctx.dispose();
		}

		ctx = new DataBindingContext();

		ctx.bindValue( SWTObservables.observeText( dataTextArea, SWT.Modify ),
				PojoObservables.observeValue( assembly, "data" ), 
				null, null );
		
		// This should be done by the model-specific publisher/listener mechanism and not by manually
		// subscribing to changes in the widgets
		this.subscribeToChanges(dataTextArea);
	}
	
	private class LineNumberErrorHandler implements ErrorHandler 
	{
		@Override
		public void error(SAXParseException e) throws SAXException {
			String msg = "Validation warning at line: " + e.getLineNumber() + " column: " + e.getColumnNumber() + " due to: " + e.getMessage();
			GuiUtils.showErrorDialog(getSite().getShell(), "Could not save changes", msg);
			throw e;
		}

		@Override
		public void fatalError(SAXParseException e) throws SAXException {
			String msg = "Validation warning at line: " + e.getLineNumber() + " column: " + e.getColumnNumber() + " due to: " + e.getMessage();
			GuiUtils.showErrorDialog(getSite().getShell(), "Could not save changes", msg);
			throw e;
		}

		@Override
		public void warning(SAXParseException e) throws SAXException {
			String msg = "Validation warning at line: " + e.getLineNumber() + " column: " + e.getColumnNumber() + " due to: " + e.getMessage();
			GuiUtils.showErrorDialog(getSite().getShell(), "Could not save changes", msg);
			throw e;
		}
	}
}
