/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;

import alma.obops.tmcdbgui.editors.inputs.AntennaToPadEditorInput;
import alma.obops.tmcdbgui.editors.inputs.AntennaToPadHistoryEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.conversation.AntennaToPadConversationUtils;
import alma.obops.tmcdbgui.views.providers.AntennaToPadContentsProvider;
import alma.obops.tmcdbgui.views.providers.AntennaToPadEditingSupport;
import alma.obops.tmcdbgui.views.providers.AntennaToPadLabelProvider;
import alma.obops.tmcdbgui.views.providers.AntennaToPadRow;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.tmcdb.domain.AntennaToPad;

/**
 * Simple editor for an antennaToPad assignment.
 * @author sharring
 */
public class AntennaToPadEditor extends TmcdbObjectEditorPart implements DirtyListener 
{
	public static final String AW0 = "AW0";
	public static final String AN0 = "AN0";
	public static final String ID = "antennatopad.editor";
	private TableViewer tableViewer;
	private AntennaToPad antennaToPad;
	private boolean dirty;

	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		InputDialog descriptionInputDialog = new InputDialog(this.getSite().getShell(), "Description", "Please add any comments about your change", "", null);
		if(descriptionInputDialog.open() != Window.OK) 
		{
			return;
		}

		// try to create a new version
		String description = descriptionInputDialog.getValue();
		String userId = System.getProperty("user.name");

		// if the new version preparation was successful, we can then perform the save
		try 
		{
			boolean canSave = AntennaToPadConversationUtils.getInstance().prepareAntennaToPadSave(antennaToPad, userId, description);

			if(canSave) 
			{
				AntennaToPadRow[] rows = (AntennaToPadRow[]) tableViewer.getInput();
				for(AntennaToPadRow row: rows)
				{
					if(row.getCoeffName().equals(AN0))
					{
						antennaToPad.setMountMetrologyAN0Coeff(row.getCoeffValue());
					}
					else if(row.getCoeffName().endsWith(AW0))
					{
						antennaToPad.setMountMetrologyAW0Coeff(row.getCoeffValue());
					}
				}

				try {
					AntennaToPadConversationUtils.getInstance().saveOrUpdateAntennaToPad(antennaToPad);
					this.setDirty(false);
				} catch (Exception e) {
					throw new RuntimeException("Could not save AntennaToPad object", e);
				}
			}
		}
		catch(Exception ex) 
		{
			ex.printStackTrace();
			throw new RuntimeException("Could not save AntennaToPad", ex);
		} 
		finally 
		{
			try {
				AntennaToPadConversationUtils.getInstance().endAntennaToPadSave(antennaToPad);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
	throws PartInitException 
	{
		AntennaToPadEditorInput editorInput = (AntennaToPadEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(editorInput.getName());
		antennaToPad = editorInput.getAntennaToPad();
		if(null != tableViewer) {
			tableViewer.setInput(populateRows()); // trigger a content reload
		}
	}

	private AntennaToPadRow[] populateRows() 
	{
		AntennaToPadRow[] retVal = new AntennaToPadRow[2];

		retVal[0] = new AntennaToPadRow();
		retVal[0].setCoeffName(AN0);
		retVal[0].setCoeffValue(antennaToPad.getMountMetrologyAN0Coeff());

		retVal[1] = new AntennaToPadRow();
		retVal[1].setCoeffName(AW0);
		retVal[1].setCoeffValue(antennaToPad.getMountMetrologyAW0Coeff());

		return retVal;
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		// not allowed
		return false;
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());

		Composite editorComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		editorComposite.setLayout(gridLayout);
		gridLayout.numColumns = 1;

		Composite tableComposite = new Composite(editorComposite, SWT.NONE);
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = true;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.FILL;
		tableComposite.setLayoutData(gdata);
		tableComposite.setLayout(new FillLayout());

		tableViewer = new TableViewer(tableComposite, SWT.BORDER | SWT.FULL_SELECTION);

		// Setup the columns
		String [] titles = { "Metrology coefficient", "Value"};
		for(int i = 0; i != titles.length; i++) {
			TableViewerColumn col = new TableViewerColumn(tableViewer, SWT.NONE);
			col.getColumn().setText(titles[i]);
			col.getColumn().setMoveable(false);
			col.getColumn().setResizable(true);
			col.setEditingSupport(new AntennaToPadEditingSupport(tableViewer, i, this));
			col.getColumn().pack();
		}
		Table table = tableViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		tableViewer.setSorter(new ViewerSorter());
		tableViewer.setContentProvider( new AntennaToPadContentsProvider() );
		tableViewer.setLabelProvider( new AntennaToPadLabelProvider() );
		tableViewer.setInput(populateRows()); // trigger a content reload

		Composite buttonComposite = new Composite(editorComposite, SWT.NONE);
		GridData gridData = new GridData();
		gridData.grabExcessHorizontalSpace = false;
		gridData.grabExcessVerticalSpace = false;
		buttonComposite.setLayoutData(gridData);
		buttonComposite.setLayout(new FillLayout());

		//   Create and configure the "history" button
		Button historyButton = new Button(buttonComposite, SWT.PUSH | SWT.CENTER);
		historyButton.setText("History");

		historyButton.addSelectionListener(new SelectionAdapter() 
		{
			public void widgetSelected(SelectionEvent e) 
			{
				AntennaToPadHistoryEditorInput editorInput =  new AntennaToPadHistoryEditorInput(antennaToPad);
				IWorkbenchWindow win = getSite().getWorkbenchWindow();
				try {
					win.getActivePage().openEditor(editorInput, AntennaToPadHistoryEditor.ID);
				} 
				catch (PartInitException e1) {
					throw new RuntimeException("Could not open antenna to pad history editor", e1);
				}
			}
		});
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean dirty) {
		this.dirty = dirty;firePropertyChange(PROP_DIRTY);

	}
}
