/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.obops.tmcdbgui.editors.inputs.AntennaPadAssignmentHistoryEditorInput;
import alma.obops.tmcdbgui.editors.inputs.HistoricalAntennaToPadEditorInput;
import alma.obops.tmcdbgui.views.providers.AntennaPadAssignmentHistoryTableContentsProvider;
import alma.obops.tmcdbgui.views.providers.AntennaPadAssignmentHistoryTableLabelProvider;
import alma.obops.tmcdbgui.views.providers.AntennaPadAssignmentHistoryViewerSorter;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.history.HistoryRecord;

/**
 * @author sharring
 *
 */
public class AntennaPadAssignmentHistoryEditor extends EditorPart 
{
	public static final String ID = "antenna-pad-assignment-history.editor";
	private TableViewer historyViewer;
	private Antenna antenna;

	@Override
	public void createPartControl( Composite parent ) 
	{
		historyViewer = new TableViewer(parent, SWT.BORDER | SWT.FULL_SELECTION | SWT.SINGLE);

		// Setup the columns
		String [] titles = { "Pad", "Start time", "End time"};
		for(int i = 0; i != titles.length; i++) {
			TableViewerColumn col = new TableViewerColumn(historyViewer, SWT.NONE);
			col.getColumn().setText(titles[i]);
			col.getColumn().setMoveable(false);
			col.getColumn().setResizable(true);
			if(i != 1) {
				col.getColumn().setWidth(150);
			} else {
				col.getColumn().setWidth(500);
			}
		}
		
		historyViewer.setSorter(new AntennaPadAssignmentHistoryViewerSorter());
		historyViewer.setContentProvider( new AntennaPadAssignmentHistoryTableContentsProvider() );
		historyViewer.setLabelProvider( new AntennaPadAssignmentHistoryTableLabelProvider(antenna.getConfiguration()) );
		historyViewer.setInput(this.antenna);
		
		Table table = historyViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		
		IDoubleClickListener listener = new GetHistoricalAntennaToPadDoubleClickListener();
		historyViewer.addDoubleClickListener(listener);
	}

	@Override
	public void setFocus() {
		historyViewer.getControl().setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor arg0) {
		// NOOP
	}

	@Override
	public void doSaveAs() {
		// NOOP
	}

	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException 
	{
		AntennaPadAssignmentHistoryEditorInput editorInput = (AntennaPadAssignmentHistoryEditorInput)input;
		setInput(input);
		if(null != historyViewer) {
			historyViewer.setInput(editorInput.getAntenna());
		}
		setSite(site);
		setPartName(editorInput.getName());
	}

	@Override 
	public void setInput(IEditorInput input)
	{
		super.setInput(input);
		this.antenna = ((AntennaPadAssignmentHistoryEditorInput) input).getAntenna();
	}
	
	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}
	
	private class GetHistoricalAntennaToPadDoubleClickListener implements IDoubleClickListener
	{
		@Override
		public void doubleClick(DoubleClickEvent evt) 
		{
			ISelection selection = evt.getSelection();
			if(selection instanceof IStructuredSelection) {
				IStructuredSelection structuredSelection = (IStructuredSelection) selection;
				if(structuredSelection.getFirstElement() != null) {
					AntennaToPad clickedA2p = (AntennaToPad) structuredSelection.getFirstElement();
					
					HistoryRecord dummyHistoryRecord = new HistoryRecord();
					dummyHistoryRecord.setDescription("Historical pad assignment of antenna " 
							+ clickedA2p.getAntenna().getName() + " on antenna " + clickedA2p.getAntenna().getName());
					
					long count = historyViewer.getTable().getSelectionIndex();
					dummyHistoryRecord.setVersion(count);
					HistoricalAntennaToPadEditorInput editorInput =  
						new HistoricalAntennaToPadEditorInput(clickedA2p, clickedA2p, dummyHistoryRecord);
					
					IWorkbenchWindow win = getSite().getWorkbenchWindow();
					try {
						win.getActivePage().openEditor(editorInput, HistoricalAntennaToPadEditor.ID);
					} 
					catch (PartInitException e1) {
						e1.printStackTrace();
						throw new RuntimeException("Could not open historical antennaToPad editor", e1);
					}
				}
			}
		}
	}
	
}