/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.obops.tmcdbgui.editors.inputs.AntennaHistoryEditorInput;
import alma.obops.tmcdbgui.editors.inputs.HistoricalAntennaEditorInput;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.views.providers.AntennaHistoryTableContentsProvider;
import alma.obops.tmcdbgui.views.providers.AntennaHistoryTableLabelProvider;
import alma.obops.tmcdbgui.views.providers.HistoryRecordViewerSorter;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.history.HistoryRecord;

/**
 * @author sharring
 *
 */
public class AntennaHistoryEditor extends EditorPart 
{
	public static final String ID = "antenna-history.editor";
	private TableViewer historyViewer;
	private Antenna antenna;

	@Override
	public void createPartControl( Composite parent ) 
	{
		historyViewer = new TableViewer(parent, SWT.BORDER | SWT.FULL_SELECTION | SWT.MULTI);

		// Setup the columns
		String [] titles = { "Version", "Description", "Modifier", "Date" };
		for(int i = 0; i != titles.length; i++) {
			TableViewerColumn col = new TableViewerColumn(historyViewer, SWT.NONE);
			col.getColumn().setText(titles[i]);
			col.getColumn().setMoveable(false);
			col.getColumn().setResizable(true);
			if(i != 1) {
				col.getColumn().setWidth(150);
			} else {
				col.getColumn().setWidth(500);
			}
		}
		
		historyViewer.setSorter(new HistoryRecordViewerSorter());
		historyViewer.setContentProvider( new AntennaHistoryTableContentsProvider() );
		historyViewer.setLabelProvider( new AntennaHistoryTableLabelProvider() );
		historyViewer.setInput(this.antenna);
		
		Table table = historyViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		
		MenuManager popupMenu = new MenuManager();
		final CompareAntennasAction compareAntennasAction = new CompareAntennasAction();
		popupMenu.add(compareAntennasAction);
		Menu menu = popupMenu.createContextMenu(table);
		table.setMenu(menu);

		historyViewer.addSelectionChangedListener(new ISelectionChangedListener() 
		{
			@Override
			public void selectionChanged(SelectionChangedEvent evt) 
			{
				ISelection selection = evt.getSelection();
				if(selection instanceof IStructuredSelection) {
					IStructuredSelection structuredSelection = (IStructuredSelection) selection;
					Object[] recordsSelected = structuredSelection.toArray();
					if(recordsSelected.length == 2) {
						compareAntennasAction.setEnabled(true);
						compareAntennasAction.setPreviousRecord((HistoryRecord)recordsSelected[0]);
						compareAntennasAction.setReferenceRecord((HistoryRecord)recordsSelected[1]);
					}
					else {
						compareAntennasAction.setEnabled(false);
					}
				}

			}
		});

		IDoubleClickListener listener = new GetHistoricalAntennaDoubleClickListener();
		historyViewer.addDoubleClickListener(listener);
	}

	@Override
	public void setFocus() {
		historyViewer.getControl().setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor arg0) {
		// NOOP
	}

	@Override
	public void doSaveAs() {
		// NOOP
	}

	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException 
	{
		AntennaHistoryEditorInput editorInput = (AntennaHistoryEditorInput)input;
		setInput(input);
		if(null != historyViewer) {
			historyViewer.setInput(editorInput.getAntenna());
		}
		setSite(site);
		setPartName(editorInput.getName());
	}

	@Override 
	public void setInput(IEditorInput input)
	{
		super.setInput(input);
		this.antenna = ((AntennaHistoryEditorInput) input).getAntenna();
	}
	
	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}
	
	private class GetHistoricalAntennaDoubleClickListener implements IDoubleClickListener
	{
		@Override
		public void doubleClick(DoubleClickEvent evt) 
		{
			ISelection selection = evt.getSelection();
			if(selection instanceof IStructuredSelection) {
				IStructuredSelection structuredSelection = (IStructuredSelection) selection;
				if(structuredSelection.getFirstElement() != null) {
					HistoryRecord clickedRecord = (HistoryRecord) structuredSelection.getFirstElement();
					Antenna historicalAntenna = null;
					try {
						historicalAntenna = BaseElementConversationUtils.getInstance().getHistoricalAntenna(antenna, clickedRecord);
					} catch (Exception e) {
						throw new RuntimeException("Unable to get historical antenna" + e);
					}
					
					HistoricalAntennaEditorInput editorInput =  
						new HistoricalAntennaEditorInput(historicalAntenna, historicalAntenna, clickedRecord);
					
					IWorkbenchWindow win = getSite().getWorkbenchWindow();
					try {
						win.getActivePage().openEditor(editorInput, HistoricalAntennaEditor.ID);
					} 
					catch (PartInitException e1) {
						throw new RuntimeException("Could not open historical antenna editor", e1);
					}
				}
			}
		}
	}
	
	private class CompareAntennasAction extends Action
	{
		private HistoryRecord referenceRecord;
		private HistoryRecord previousRecord;
		
		public CompareAntennasAction() 
		{
			super("Show differences");
		}
		
		public void setReferenceRecord(HistoryRecord rec)
		{
			this.referenceRecord = rec;
		}
		
		public void setPreviousRecord(HistoryRecord rec)
		{
			this.previousRecord = rec;
		}

		public void run() 
		{
			Antenna historicalAntenna = null;
			Antenna historicalAntennaPreviousVersion = null;
			try {
				historicalAntenna = BaseElementConversationUtils.getInstance().getHistoricalAntenna(antenna, referenceRecord);
				historicalAntennaPreviousVersion = BaseElementConversationUtils.getInstance().getHistoricalAntenna(antenna, previousRecord);
			} catch (Exception e) {
				throw new RuntimeException("Unable to get historical antennas" + e);
			}
			
			HistoryRecord junkRecord = new HistoryRecord();
			junkRecord.setVersion(0L - (referenceRecord.getVersion() - previousRecord.getVersion()));
			HistoricalAntennaEditorInput editorInput =  
				new HistoricalAntennaEditorInput(historicalAntenna, historicalAntennaPreviousVersion, 
						junkRecord, "Diff antenna v." + referenceRecord.getVersion() 
						+ " to v." + previousRecord.getVersion() + " for " + antenna.getName());
			
			IWorkbenchWindow win = getSite().getWorkbenchWindow();
			try {
				win.getActivePage().openEditor(editorInput, HistoricalAntennaEditor.ID);
			} 
			catch (PartInitException e1) {
				throw new RuntimeException("Could not open historical antenna editor", e1);
			}
		}
	}
}
