/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.hibernate.exception.ConstraintViolationException;

import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.domain.IModelChangePublisher;
import alma.obops.tmcdbgui.editors.inputs.AntennaEditorInput;
import alma.obops.tmcdbgui.editors.inputs.AntennaHistoryEditorInput;
import alma.obops.tmcdbgui.editors.inputs.AntennaPadAssignmentHistoryEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.BaseElementConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.HwConfigurationConversationUtils;
import alma.obops.tmcdbgui.widgets.AntennaAttributesComposite;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaType;

/**
 * Editor for antenna object.
 * @author sharring
 */
public class AntennaEditor extends TmcdbObjectEditor 
   implements IModelChangePublisher
{
	public static final String ID = "antenna.editor";
	private static final String CHANGES_NOT_SAVED = "Changes not saved";

	private List<IModelChangeListener> modelChangeListeners = new ArrayList<IModelChangeListener>();
	private boolean shouldNotifyListeners;
	private AntennaAttributesComposite downcastControl;
	private Antenna antenna;
	private AntennaType originalType;
	private String originalNumber;
	private String originalName;

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		AntennaEditorInput antennaEditorInput = (AntennaEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(antennaEditorInput.getName());
		antenna = antennaEditorInput.getAntenna();
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		
		ScrolledComposite sc1 = new ScrolledComposite(parent,SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		sc1.setExpandHorizontal(true);
		sc1.setExpandVertical(true);
        
		Composite comp = new Composite(sc1, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 1;
		comp.setLayout(gridLayout);
		comp.setLayoutData(new GridData(GridData.FILL_BOTH));
		downcastControl = new AntennaAttributesComposite(comp, SWT.NONE, this);
		
		Composite buttonComposite = new Composite(comp, SWT.NONE);
    	buttonComposite.setLayout(new FillLayout());

		 //   Create and configure the "history" button
        Button historyButton = new Button(buttonComposite, SWT.PUSH | SWT.CENTER);
        historyButton.setText("History");
        
        historyButton.addSelectionListener(new SelectionAdapter() 
        {
        	public void widgetSelected(SelectionEvent e) 
            {
         	   AntennaHistoryEditorInput editorInput =  new AntennaHistoryEditorInput(antenna);
         	   IWorkbenchWindow win = getSite().getWorkbenchWindow();
         	   try {
         		   win.getActivePage().openEditor(editorInput, AntennaHistoryEditor.ID);
         	   } 
         	   catch (PartInitException e1) {
         		   throw new RuntimeException("Could not open antenna history editor", e1);
         	   }
            }
        });
        
        //   Create and configure the "history" button
        Button padHistoryButton = new Button(buttonComposite, SWT.PUSH | SWT.CENTER);
        padHistoryButton.setText("Pad history");
        
        padHistoryButton.addSelectionListener(new SelectionAdapter() 
        {
        	public void widgetSelected(SelectionEvent e) 
            {
         	   AntennaPadAssignmentHistoryEditorInput editorInput =  new AntennaPadAssignmentHistoryEditorInput(antenna);
         	   IWorkbenchWindow win = getSite().getWorkbenchWindow();
         	   try {
         		   win.getActivePage().openEditor(editorInput, AntennaPadAssignmentHistoryEditor.ID);
         	   } 
         	   catch (PartInitException e1) {
         		   throw new RuntimeException("Could not open antenna-to-pad assignment history editor", e1);
         	   }
            }
        });
        
		sc1.setMinSize(comp.computeSize(SWT.DEFAULT, SWT.DEFAULT));
		sc1.setContent(comp);
		
		configure();
	}

	@Override
	public void setFocus() {
		// TODO Auto-generated method stub

	}
	
	@Override
	public void setInput( IEditorInput input ) 
	{
		super.setInput(input);
		AntennaEditorInput antEdInput = ((AntennaEditorInput)input);
		Antenna ant = (antEdInput).getAntenna();
		this.modelChangeListeners.clear();
		this.addModelChangeListener(antEdInput.getModelChangeListener());
		this.antenna = ant;
		this.originalName = ant.getName();
		if(null != downcastControl) 
		{
			configure();
		}
	}
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		if((!downcastControl.getAntennaNumber().equals(originalNumber) || !downcastControl.getAntennaType().equals(originalType)) &&
				(downcastControl.getStatus() != null && downcastControl.getStatus().trim().length() > 0)) 
		{
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, downcastControl.getStatus());
			setPartName(originalName);
		} 
		else 
		{ 
			InputDialog descriptionInputDialog = new InputDialog(this.getSite().getShell(), "Description", "Please add any comments about your change", "", null);
    		if(descriptionInputDialog.open() != Window.OK) 
    		{
    			return;
    		}
    		
    		// try to create a new version
			String description = descriptionInputDialog.getValue();
			String userId = System.getProperty("user.name");
			
    		try 
    		{
    			boolean canSave = BaseElementConversationUtils.getInstance().prepareAntennaSave(antenna, userId, description);

    			// if the new version preparation was successful, we can then perform the save
    			if(canSave) 
    			{	
    				try {
    					this.getSite().getShell().setCursor(this.getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
    					if(!checkForDuplicateWalshFunction(downcastControl.getWalshSequence()) 
    							|| !checkForDuplicateLoOffset(downcastControl.getLoOffsetting()))
    					{
    						return;
    					}

    					applyChangesAndSave();
    					this.originalNumber = downcastControl.getAntennaNumber();
    					this.originalName = downcastControl.getAntennaName();
    					this.originalType = downcastControl.getAntennaType();
    					setDirty(false);
    				} catch (Exception e) {
    					GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
    					setDirty(true);
    					e.printStackTrace();
    				} 
    				finally {
    					this.getSite().getShell().setCursor(null);
    				}
    			}
			}
    		catch(Exception ex) 
    		{
    			ex.printStackTrace();
    			throw new RuntimeException("Could not save antenna", ex);
    		} 
    		finally 
    		{
    			 try {
    				BaseElementConversationUtils.getInstance().endAntennaSave(antenna);
    			} catch (Exception e) {
    				e.printStackTrace();
    			}
    		}
		}
		
		if(shouldNotifyListeners) {
			this.modelChanged();
			this.shouldNotifyListeners = false;
		}
	}

	private void applyChangesAndSave() 
	{
		String newAntennaName = downcastControl.getAntennaName();
		if(!this.antenna.getName().equals(newAntennaName)) {
			shouldNotifyListeners = true;
			this.setPartName(newAntennaName);
		} else {
			shouldNotifyListeners = false;
		}
		this.antenna.setName(newAntennaName);
		this.antenna.setAntennaType(downcastControl.getAntennaType());
		this.antenna.setCommissionDate(downcastControl.getCommissionDate().getTime());
		this.antenna.setDiameter(downcastControl.getDiameter());
		this.antenna.setPosition(downcastControl.getPosition());
		this.antenna.setOffset(downcastControl.getOffset());
		this.antenna.setLoOffsettingIndex(downcastControl.getLoOffsetting());
		this.antenna.setWalshSeq(downcastControl.getWalshSequence());
		this.antenna.setCaiAca(downcastControl.getCaiAca());
		this.antenna.setCaiBaseline(downcastControl.getCaiBaseline());

		try {
			BaseElementConversationUtils.getInstance().saveOrUpdateAntenna(antenna);
		} catch (ConstraintViolationException e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, "Antenna already exists: antenna name (prefix + number) must be unique within configuration");
			antenna.setName(originalName);
		} catch (Exception e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
		}
		
		this.downcastControl.setAntenna(this.antenna);
		this.downcastControl.setDirty(false);
	}

	private boolean checkForDuplicateLoOffset(Integer loOffset) 
	{
		boolean retVal = true;
		try {
			Antenna[] antennasWithLoOffset = HwConfigurationConversationUtils.getInstance().findAntennaByLoOffsetInConfig(loOffset, antenna.getConfiguration());
			if(antennasWithLoOffset.length > 0) {
				StringBuffer antennaStrBuf = new StringBuffer();
				int count = 0;
				boolean duplicateFound = false;
				for(Antenna antennaIterated: antennasWithLoOffset) 
				{
					if(!antennaIterated.getId().equals(antenna.getId())) 
					{
						duplicateFound = true;
						antennaStrBuf.append(antennaIterated.getName());
						if(++count < antennasWithLoOffset.length) {
							antennaStrBuf.append(", ");
						}
					} else {
						count++;
					}
				}
				if(duplicateFound) {
					retVal = MessageDialog.openConfirm(downcastControl.getShell(), 
							"Duplicate LO offset", "The following antennas have the same LO offset: " + antennaStrBuf.toString() + " are you sure?");
				}
			}
		} catch(Exception ex) {
			MessageDialog.openError(downcastControl.getShell(), "Problem encountered", "Cannot query for LO offset duplicates...");
			ex.printStackTrace();
			retVal = false;
		}
		
		return retVal;
	}

	private boolean checkForDuplicateWalshFunction(Integer walshSequence) 
	{
		boolean retVal = true;
		try {
			Antenna[] antennasWithWalsh = HwConfigurationConversationUtils.getInstance().findAntennaByWalshFunctionInConfig(walshSequence, antenna.getConfiguration());
			if(antennasWithWalsh.length > 0) {
				StringBuffer antennaStrBuf = new StringBuffer();
				int count = 0;
				boolean duplicateFound = false;
				for(Antenna antennaIterated: antennasWithWalsh) 
				{
					if(!antennaIterated.getId().equals(antenna.getId())) 
					{
						duplicateFound = true;
						antennaStrBuf.append(antennaIterated.getName());
						if(++count < antennasWithWalsh.length) {
							antennaStrBuf.append(", ");
							int remainder = count % 10;
							if(remainder == 0) {
								antennaStrBuf.append("\n\t");
							}
						}
					} else {
						count++;
					}
				}
				if(duplicateFound) {
					retVal = MessageDialog.openConfirm(downcastControl.getShell(), 
							"Duplicate Walsh sequence", "The following antennas have the same Walsh sequence: " + antennaStrBuf.toString() + " are you sure?");
				}
			}
		} catch(Exception ex) {
			MessageDialog.openError(downcastControl.getShell(), "Problem encountered", "Cannot query for Walsh sequence duplicates...");
			ex.printStackTrace();
			retVal = false;
		}
		return retVal;
	}

	private void configure() 
	{
		this.downcastControl.setAntenna(antenna);
		this.originalType = downcastControl.getAntennaType();
		this.originalNumber = downcastControl.getAntennaNumber();
		this.originalName = antenna.getName();
	}
	
	@Override
	protected Object getEditedObject() {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	protected void resetToOriginalContent() {
		// TODO Auto-generated method stub
		
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {
		// TODO Auto-generated method stub
		
	}

	@Override
	public void addModelChangeListener(IModelChangeListener listener) {
		if(null != listener)
		{
			this.modelChangeListeners.add(listener);
		}
	}

	@Override
	public void modelChanged() 
	{
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.internalModelChange();
		}
	}

	@Override
	public void modelShouldBeReloaded() {
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.externalModelChange();
		}
	}

	@Override
	public void removeModelChangeListener(IModelChangeListener listener) {
		this.modelChangeListeners.remove(listener);
	}
}
