/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.AcsServiceServiceType;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.editors.inputs.AcsServiceEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.ImageHelper;
import alma.obops.tmcdbgui.utils.LabelHelper;
import alma.obops.tmcdbgui.utils.conversation.AcsServiceConversationUtils;
import alma.obops.tmcdbgui.views.SoftwareDeploymentView;
import alma.obops.tmcdbgui.widgets.AcsServiceComposite;

public class AcsServiceEditor extends TmcdbObjectEditor 
{
	public final static String NOTIFICATION_SERVICE_TYPE = AcsServiceServiceType.NOTIFICATION.toString();
	private final static String NAMING_SERVICE_TYPE = AcsServiceServiceType.NAMING.toString();
	private final static String IFR_SERVICE_TYPE = AcsServiceServiceType.IFR.toString();
	private final static String CDB_SERVICE_TYPE = AcsServiceServiceType.CDB.toString();
	private final static String LOGGING_SERVICE_TYPE = AcsServiceServiceType.LOGGING.toString();
	private final static String MANAGER_SERVICE_TYPE = AcsServiceServiceType.MANAGER.toString();
	private final static String ALARM_SERVICE_TYPE = AcsServiceServiceType.ALARM.toString();
	private final static String LOGPROXY_SERVICE_TYPE = AcsServiceServiceType.LOGPROXY.toString();

	public final static String[] SERVICE_TYPES = { NAMING_SERVICE_TYPE, IFR_SERVICE_TYPE, CDB_SERVICE_TYPE, 
		 NOTIFICATION_SERVICE_TYPE, LOGGING_SERVICE_TYPE, MANAGER_SERVICE_TYPE, 
		 ALARM_SERVICE_TYPE, LOGPROXY_SERVICE_TYPE };
	
	public static final String ID = "acsservice.editor";

	/* The actual computer being edited */
	private AcsService _service;

	/* Initial contents, used to fallback */
	private AcsService _origService;
	
	private AcsServiceComposite composite;
	
	@Override
	public void setFocus() {
		composite.setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor monitor) {

		// Check for invalid inputs
		if(  (_service.getServiceType() == null || _service.getServiceType().toString().trim().equals(""))  ||
			 ( _service.getServiceType().equals(AcsServiceServiceType.NOTIFICATION) && 
			   (_service.getServiceInstanceName() == null || _service.getServiceInstanceName().trim().equals(""))) )
		{
			MessageDialog.openInformation(getSite().getShell(),
					"Please specify all fields",
					"ACS service cannot be saved without all fields defined");
			return;
		}

		// Persist the object
		try {
			if(!_service.getServiceType().equals(AcsServiceServiceType.NOTIFICATION)) {
				_service.setServiceInstanceName(null);
			}
			AcsServiceConversationUtils.getInstance().saveOrUpdateAcsService(_service);
		} catch(Exception e) {
			e.printStackTrace();
			MessageDialog.openError(getSite().getShell(), "Cannot save ACS Service", "Error while saving AcsService of type: " + _service.getServiceType());
			return;
		}

		setEditedObjectAsOriginalContent();
		setDirty(false);

		SoftwareDeploymentView sdv = (SoftwareDeploymentView)RcpUtils.findView( SoftwareDeploymentView.ID );
		sdv.internalModelChange();
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
			throws PartInitException 
	{
		AcsServiceEditorInput cei = (AcsServiceEditorInput)input;
		setInput(input);
		setSite(site);

		_service = cei.getAcsService();
		if( _service.getAcsServiceId() == null) {
			setDirty(true);
		}

		setEditedObjectAsOriginalContent();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	@Override
	public void createPartControl(Composite parent) 
	{
		composite = new AcsServiceComposite(parent, SWT.NONE, _service);
		// Data binding and subscription
		bind("serviceInstanceName", composite.getInstanceNameControl());
		bind("serviceType", composite.getTypeControl());
	}

	@Override
	protected Object getEditedObject() {
		return _service;
	}

	@Override
	protected void setEditedObjectAsOriginalContent() {
		_origService = new AcsService();
		_origService.setServiceType(_service.getServiceType());
		_origService.setServiceInstanceName(_service.getServiceInstanceName());

		setTitleImage(ImageHelper.getImage(_origService));
		String partName = LabelHelper.getAcsServiceLabel(_origService);
		setPartName(partName);
		setTitleToolTip(partName);
	}

	protected void resetToOriginalContent() {
		_service.setServiceType(_origService.getServiceType());
		_service.setServiceInstanceName(_origService.getServiceInstanceName());
	}
}
