/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.obops.tmcdbgui.editors.inputs.AcaCorrDelaysHistoryEditorInput;
import alma.obops.tmcdbgui.editors.inputs.HistoricalAcaCorrDelaysEditorInput;
import alma.obops.tmcdbgui.utils.conversation.AcaCorrDelaysConversationUtils;
import alma.obops.tmcdbgui.views.providers.AcaCorrDelaysHistoryTableContentsProvider;
import alma.obops.tmcdbgui.views.providers.AcaCorrDelaysHistoryTableLabelProvider;
import alma.obops.tmcdbgui.views.providers.HistoryRecordViewerSorter;
import alma.tmcdb.domain.AcaCorrDelays;
import alma.tmcdb.history.HistoryRecord;

/**
 * "Editor" (but actually read-only) for the history of an acacorrdelay object.
 * @author sharring
 */
public class AcaCorrDelaysHistoryEditor extends EditorPart 
{
	public static final String ID = "acacorrdelays-history.editor";
	private TableViewer historyViewer;
	private AcaCorrDelays acaCorrDelays;

	@Override
	public void createPartControl( Composite parent ) 
	{
		historyViewer = new TableViewer(parent, SWT.BORDER | SWT.FULL_SELECTION | SWT.MULTI);

		// Setup the columns
		String [] titles = { "Version", "Description", "Modifier", "Date" };
		for(int i = 0; i != titles.length; i++) {
			TableViewerColumn col = new TableViewerColumn(historyViewer, SWT.NONE);
			col.getColumn().setText(titles[i]);
			col.getColumn().setMoveable(false);
			col.getColumn().setResizable(true);
			if(i != 1) {
				col.getColumn().setWidth(150);
			} else {
				col.getColumn().setWidth(500);
			}
		}
		
		historyViewer.setSorter(new HistoryRecordViewerSorter());
		historyViewer.setContentProvider(new AcaCorrDelaysHistoryTableContentsProvider());
		historyViewer.setLabelProvider(new AcaCorrDelaysHistoryTableLabelProvider());
		historyViewer.setInput(this.acaCorrDelays);
		
		Table table = historyViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		
		MenuManager popupMenu = new MenuManager();
		final CompareAcaCorrDelaysAction compareAcaCorrDelayssAction = new CompareAcaCorrDelaysAction();
		popupMenu.add(compareAcaCorrDelayssAction);
		Menu menu = popupMenu.createContextMenu(table);
		table.setMenu(menu);

		historyViewer.addSelectionChangedListener(new ISelectionChangedListener() 
		{
			@Override
			public void selectionChanged(SelectionChangedEvent evt) 
			{
				ISelection selection = evt.getSelection();
				if(selection instanceof IStructuredSelection) {
					IStructuredSelection structuredSelection = (IStructuredSelection) selection;
					Object[] recordsSelected = structuredSelection.toArray();
					if(recordsSelected.length == 2) {
						compareAcaCorrDelayssAction.setEnabled(true);
						compareAcaCorrDelayssAction.setPreviousRecord((HistoryRecord)recordsSelected[0]);
						compareAcaCorrDelayssAction.setReferenceRecord((HistoryRecord)recordsSelected[1]);
					}
					else {
						compareAcaCorrDelayssAction.setEnabled(false);
					}
				}

			}
		});

		IDoubleClickListener listener = new GetHistoricalAcaCorrDelaysDoubleClickListener();
		historyViewer.addDoubleClickListener(listener);
	}

	@Override
	public void setFocus() {
		historyViewer.getControl().setFocus();
	}
	
	@Override
	public void doSave(IProgressMonitor arg0) {
		// NOOP
	}

	@Override
	public void doSaveAs() {
		// NOOP
	}

	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException 
	{
		AcaCorrDelaysHistoryEditorInput editorInput = (AcaCorrDelaysHistoryEditorInput)input;
		setInput(input);
		if(null != historyViewer) {
			historyViewer.setInput(editorInput.getAcaCorrDelays());
		}
		setSite(site);
		setPartName(editorInput.getName());
	}

	@Override 
	public void setInput(IEditorInput input)
	{
		super.setInput(input);
		this.acaCorrDelays = ((AcaCorrDelaysHistoryEditorInput) input).getAcaCorrDelays();
	}
	
	@Override
	public boolean isDirty() {
		return false;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}
	
	private class GetHistoricalAcaCorrDelaysDoubleClickListener implements IDoubleClickListener
	{
		@Override
		public void doubleClick(DoubleClickEvent evt) 
		{
			ISelection selection = evt.getSelection();
			if(selection instanceof IStructuredSelection) {
				IStructuredSelection structuredSelection = (IStructuredSelection) selection;
				if(structuredSelection.getFirstElement() != null) {
					HistoryRecord clickedRecord = (HistoryRecord) structuredSelection.getFirstElement();
					AcaCorrDelays historicalAcaCorrDelays = null;
					try {
						historicalAcaCorrDelays = AcaCorrDelaysConversationUtils.getInstance().getHistoricalAcaCorrDelays(acaCorrDelays, clickedRecord);
					} catch (Exception e) {
						e.printStackTrace();
						throw new RuntimeException("Unable to get historical ACA correlator delays" + e);
					}
					
					HistoricalAcaCorrDelaysEditorInput editorInput =  
						new HistoricalAcaCorrDelaysEditorInput(historicalAcaCorrDelays, historicalAcaCorrDelays, clickedRecord);
					
					IWorkbenchWindow win = getSite().getWorkbenchWindow();
					try {
						win.getActivePage().openEditor(editorInput, HistoricalAcaCorrDelaysEditor.ID);
					} 
					catch (PartInitException e1) {
						e1.printStackTrace();
						throw new RuntimeException("Could not open historical ACA correlator delays editor", e1);
					}
				}
			}
		}
	}
	
	private class CompareAcaCorrDelaysAction extends Action
	{
		private HistoryRecord referenceRecord;
		private HistoryRecord previousRecord;
		
		public CompareAcaCorrDelaysAction() 
		{
			super("Show differences");
		}
		
		public void setReferenceRecord(HistoryRecord rec)
		{
			this.referenceRecord = rec;
		}
		
		public void setPreviousRecord(HistoryRecord rec)
		{
			this.previousRecord = rec;
		}

		public void run() 
		{
			AcaCorrDelays historicalAcaCorrDelays = null;
			AcaCorrDelays historicalAcaCorrDelaysPreviousVersion = null;
			try {
				historicalAcaCorrDelays = AcaCorrDelaysConversationUtils.getInstance().getHistoricalAcaCorrDelays(acaCorrDelays, referenceRecord);
				historicalAcaCorrDelaysPreviousVersion = AcaCorrDelaysConversationUtils.getInstance().getHistoricalAcaCorrDelays(acaCorrDelays, previousRecord);
			} catch (Exception e) {
				throw new RuntimeException("Unable to get historical ACA correlator delays" + e);
			}
			
			HistoryRecord junkRecord = new HistoryRecord();
			junkRecord.setVersion(0L - (referenceRecord.getVersion() - previousRecord.getVersion()));
			
			String identifier = acaCorrDelays.getAntenna().getName();
			HistoricalAcaCorrDelaysEditorInput editorInput =  
				new HistoricalAcaCorrDelaysEditorInput(historicalAcaCorrDelays, historicalAcaCorrDelaysPreviousVersion, 
						junkRecord, "Diff ACA correlator delays v." + referenceRecord.getVersion() 
						+ " to v." + previousRecord.getVersion() + " for " + identifier);
			
			IWorkbenchWindow win = getSite().getWorkbenchWindow();
			try {
				win.getActivePage().openEditor(editorInput, HistoricalAcaCorrDelaysEditor.ID);
			} 
			catch (PartInitException e1) {
				e1.printStackTrace();
				throw new RuntimeException("Could not open historical ACA correlator delays editor", e1);
			}
		}
	}
}
