/*******************************************************************************
  * ALMA - Atacama Large Millimeter Array
  * Copyright (c) NRAO - National Radio Astronomy Observatory, 2012
  * (in the framework of the ALMA collaboration).
  * All rights reserved.
  *
  * This library is free software; you can redistribute it and/or
  * modify it under the terms of the GNU Lesser General Public
  * License as published by the Free Software Foundation; either
  * version 2.1 of the License, or (at your option) any later version.
  *
  * This library is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this library; if not, write to the Free Software
  * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 
*******************************************************************************/
package alma.obops.tmcdbgui.editors;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;

import alma.BasebandNameMod.BasebandName;
import alma.obops.tmcdbgui.editors.inputs.AcaCorrDelaysEditorInput;
import alma.obops.tmcdbgui.editors.inputs.AcaCorrDelaysHistoryEditorInput;
import alma.obops.tmcdbgui.editors.inputs.TmcdbObjectEditorInput;
import alma.obops.tmcdbgui.utils.conversation.AcaCorrDelaysConversationUtils;
import alma.obops.tmcdbgui.views.providers.AcaCorrDelaysContentsProvider;
import alma.obops.tmcdbgui.views.providers.AcaCorrDelaysEditingSupport;
import alma.obops.tmcdbgui.views.providers.AcaCorrDelaysLabelProvider;
import alma.obops.tmcdbgui.views.providers.AcaCorrDelaysRow;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.tmcdb.domain.AcaCorrDelays;

/**
 * @author sharring
 *
 */
public class AcaCorrDelaysEditor extends TmcdbObjectEditorPart implements DirtyListener 
{
	public static final String ID = "acacorrdelays.editor";
	private TableViewer tableViewer;
	private AcaCorrDelays acaCorrDelays;
	private boolean dirty;

	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		InputDialog descriptionInputDialog = new InputDialog(this.getSite().getShell(), "Description", "Please add any comments about your change", "", null);
		if(descriptionInputDialog.open() != Window.OK) 
		{
			return;
		}

		// try to create a new version
		String description = descriptionInputDialog.getValue();
		String userId = System.getProperty("user.name");

		// if the new version preparation was successful, we can then perform the save
		try 
		{
			boolean canSave = AcaCorrDelaysConversationUtils.getInstance().prepareAcaCorrDelaysSave(acaCorrDelays, userId, description);

			if(canSave) 
			{
				try {
					AcaCorrDelaysRow[] rows = (AcaCorrDelaysRow[]) tableViewer.getInput();
					for(AcaCorrDelaysRow row: rows) {
						if(row.getBaseband().equals(BasebandName.BB_1)) {
							acaCorrDelays.setDelayBbOne(row.getDelay());	
						} else if(row.getBaseband().equals(BasebandName.BB_2)) {
							acaCorrDelays.setDelayBbTwo(row.getDelay());
						} else if(row.getBaseband().equals(BasebandName.BB_3)) {
							acaCorrDelays.setDelayBbThree(row.getDelay());
						} else if(row.getBaseband().equals(BasebandName.BB_4)) {
							acaCorrDelays.setDelayBbFour(row.getDelay());
						}
					}
					AcaCorrDelaysConversationUtils.getInstance().saveOrUpdateAcaCorrDelays(acaCorrDelays);
					this.setDirty(false);
				} catch (Exception e) {
					throw new RuntimeException("Could not save AcaCorrDelays object", e);
				}
			}
		}
		catch(Exception ex) 
		{
			ex.printStackTrace();
			throw new RuntimeException("Could not save AcaCorrDelays", ex);
		} 
		finally 
		{
			try {
				AcaCorrDelaysConversationUtils.getInstance().endAcaCorrDelaysSave(acaCorrDelays);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	@Override
	public void specializedInit(IEditorSite site, TmcdbObjectEditorInput input)
	throws PartInitException 
	{
		AcaCorrDelaysEditorInput editorInput = (AcaCorrDelaysEditorInput)input;
		setInput(input);
		setSite(site);
		setPartName(editorInput.getName());
		acaCorrDelays = editorInput.getAcaCorrDelays();
		if(null != tableViewer) {
			tableViewer.setInput(populateRows()); // trigger a content reload
		}
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		// not allowed
		return false;
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());

		Composite editorComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		editorComposite.setLayout(gridLayout);
		gridLayout.numColumns = 1;

		Composite tableComposite = new Composite(editorComposite, SWT.NONE);
		GridData gdata = new GridData();
		gdata.grabExcessHorizontalSpace = true;
		gdata.grabExcessVerticalSpace = true;
		gdata.horizontalAlignment = SWT.FILL;
		gdata.verticalAlignment = SWT.FILL;
		tableComposite.setLayoutData(gdata);
		tableComposite.setLayout(new FillLayout());

		tableViewer = new TableViewer(tableComposite, SWT.BORDER | SWT.FULL_SELECTION);

		// Setup the columns
		String [] titles = { "Baseband", "Delay (s)"};
		for(int i = 0; i != titles.length; i++) {
			TableViewerColumn col = new TableViewerColumn(tableViewer, SWT.NONE);
			col.getColumn().setText(titles[i]);
			col.getColumn().setMoveable(false);
			col.getColumn().setResizable(true);
			col.setEditingSupport(new AcaCorrDelaysEditingSupport(tableViewer, i, this));
			col.getColumn().pack();
		}
		Table table = tableViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		tableViewer.setSorter(new ViewerSorter());
		tableViewer.setContentProvider( new AcaCorrDelaysContentsProvider() );
		tableViewer.setLabelProvider( new AcaCorrDelaysLabelProvider() );
		tableViewer.setInput(populateRows()); // trigger a content reload

		Composite buttonComposite = new Composite(editorComposite, SWT.NONE);
		GridData gridData = new GridData();
		gridData.grabExcessHorizontalSpace = false;
		gridData.grabExcessVerticalSpace = false;
		buttonComposite.setLayoutData(gridData);
		buttonComposite.setLayout(new FillLayout());

		//   Create and configure the "history" button
		Button historyButton = new Button(buttonComposite, SWT.PUSH | SWT.CENTER);
		historyButton.setText("History");

		historyButton.addSelectionListener(new SelectionAdapter() 
		{
			public void widgetSelected(SelectionEvent e) 
			{
				AcaCorrDelaysHistoryEditorInput editorInput =  new AcaCorrDelaysHistoryEditorInput(acaCorrDelays);
				IWorkbenchWindow win = getSite().getWorkbenchWindow();
				try {
					win.getActivePage().openEditor(editorInput, AcaCorrDelaysHistoryEditor.ID);
				} 
				catch (PartInitException e1) {
					e1.printStackTrace();
					throw new RuntimeException("Could not open ACA correlator delays history editor", e1);
				}
			}
		});
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean dirty) {
		this.dirty = dirty;firePropertyChange(PROP_DIRTY);
	}
	
	private AcaCorrDelaysRow[] populateRows() 
	{
		AcaCorrDelaysRow[] retVal = new AcaCorrDelaysRow[4];

		retVal[0] = new AcaCorrDelaysRow();
		retVal[0].setBaseband(BasebandName.BB_1);
		retVal[0].setDelay(acaCorrDelays.getDelayBbOne());

		retVal[1] = new AcaCorrDelaysRow();
		retVal[1].setBaseband(BasebandName.BB_2);
		retVal[1].setDelay(acaCorrDelays.getDelayBbTwo());
		
		retVal[2] = new AcaCorrDelaysRow();
		retVal[2].setBaseband(BasebandName.BB_3);
		retVal[2].setDelay(acaCorrDelays.getDelayBbThree());
		
		retVal[3] = new AcaCorrDelaysRow();
		retVal[3].setBaseband(BasebandName.BB_4);
		retVal[3].setDelay(acaCorrDelays.getDelayBbFour());

		return retVal;
	}
}
