/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdbgui.domain;

/**
 * Interface which can be implemented by classes which can make changes to
 * (or reload) the domain model. Typically, it will be implemented by 
 * actions which modify / reload the domain model. This will allow views
 * to be decoupled from actions, such that interested views can register
 * to be notified when actions change or reload the domain model.
 * 
 * @author sharring
 * 
 * @see IModelChangeListener
 */
public interface IModelChangePublisher 
{
	/**
	 * Adds a model change listener to the list of 'interested parties'
	 * which will be notified upon a change to (or reload of) the model.
	 * 
	 * @param listener the listener which wishes to be notified when
	 * the domain model changes or is reloaded.
	 */
	public void addModelChangeListener(IModelChangeListener listener);
	
	/**
	 * Removes a model change listener to the list of 'interested parties'
	 * which will be notified upon a change to (or reload of) the model.
	 * 
	 * @param listener the listener which wishes to be notified when
	 * the domain model changes or is reloaded.
	 */
	public void removeModelChangeListener(IModelChangeListener listener);
	
	/**
	 * When the domain model has been changed; this method can be invoked 
	 * in order to update / notify the listeners, if any, of the change.
	 */
	public void modelChanged();

	/**
	 * Called when the domain model should be completely reloaded (e.g. from the database);
	 * interested listeners, if any, will be notified when this method is called.
	 */
	public void modelShouldBeReloaded();
}
