/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.VerifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.acs.tmcdb.ReductionThreshold;
import alma.obops.tmcdb.alarms.ui.wizards.support.IntegerVerifyListener;
import alma.obops.tmcdbgui.widgets.StatusPublishingComposite;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;

public class ReductionThresholdAttributesComposite extends StatusPublishingComposite
{
	private Text thresholdText;
	private ReductionThreshold threshold;
	private VerifyListener integerVerifyListener = new IntegerVerifyListener();
	private KeyListener verifyKeyListener = new SetDirtyKeyListener();

	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 * @param threshold the threshold that is being "dealt" with.
	 */
	public ReductionThresholdAttributesComposite(Composite parent, int style, ReductionThreshold threshold, StatusListener statusListener, DirtyListener dirtyListener)
	{
		super(parent, style);
		this.addStatusListener(statusListener);
		this.addDirtyListener(dirtyListener);
		this.threshold = threshold;
		createControl();
	}

	private void createControl() 
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;  // label, entry
		setLayout( layout );
		GridData gridData = new GridData(SWT.FILL, SWT.NONE, true, true, 1, 1);
		setLayoutData(gridData);

		GridData gridData2 = new GridData();
		gridData2.horizontalAlignment = GridData.FILL;
		gridData2.grabExcessHorizontalSpace = false;
		Label label = new Label(this, SWT.NONE);
		label.setText("Threshold");
		label.setLayoutData(gridData2);
		
		thresholdText = new Text(this, SWT.BORDER);
		GridData gridData3 = new GridData();
		gridData3.horizontalAlignment = GridData.FILL;
		gridData3.grabExcessHorizontalSpace = true;
		String thresholdString = this.threshold != null ? this.threshold.getValue().toString() : "";
		thresholdText.setText(thresholdString);
		thresholdText.setLayoutData(gridData3);
		thresholdText.addVerifyListener(integerVerifyListener);
		thresholdText.addKeyListener(verifyKeyListener);
	}
	
	private class SetDirtyKeyListener implements KeyListener 
	{
		@Override
		public void keyPressed(KeyEvent e) {
			setDirty(true);
		}

		@Override
		public void keyReleased(KeyEvent e) {
		}
	}
	
	public void setThreshold(ReductionThreshold newThreshold) 
	{
		if(null == newThreshold || this.threshold == newThreshold) 
		{
			return;
		}
		this.threshold = newThreshold;
		String newThresholdStr = newThreshold.getValue().toString();
		thresholdText.removeVerifyListener(integerVerifyListener);
		this.thresholdText.setText(newThresholdStr);
		thresholdText.addVerifyListener(integerVerifyListener);
	}
	
	public ReductionThreshold getThreshold()
	{
		Integer value = thresholdText.getText().trim().length() > 0 ? Integer.parseInt(thresholdText.getText()) : 0;
		if(null == this.threshold) {
			this.threshold = new ReductionThreshold();
		}
		this.threshold.setValue(value);
		return this.threshold;
	}
}
