/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import java.util.List;

import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import alma.acs.tmcdb.Location;
import alma.obops.tmcdb.alarms.ui.actions.add.AddLocationAction;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewLocationListener;
import alma.obops.tmcdb.alarms.ui.tree.helpers.LocationHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;

public class LocationSelectionDialog extends ElementListSelectionDialog implements INewLocationListener
{
	private List<Location> allLocations;
	
	public LocationSelectionDialog(Shell shell, ILabelProvider renderer) 
	{
		super(shell, renderer);
		loadLocations();
		setIgnoreCase(true);
		setMessage("Select a location");
		setMultipleSelection(false);
	}

	protected Control createDialogArea(Composite parent) 
	{
		final Composite comp = new Composite(parent, SWT.NONE);
		comp.setLayout(new GridLayout(2, false));
		comp.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		GridData gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		gd.horizontalSpan = 2;
		Control control = super.createDialogArea(comp);
		control.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 10;
		Label iconLabel = new Label(comp, SWT.NONE);
		iconLabel.setImage(RcpUtils.getImage("icons/location-new.png"));
		iconLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Link newLocationLink = new Link(comp, SWT.NONE);
		newLocationLink.setText("<a href=\"create\">Create a new location</a>");
		newLocationLink.setLayoutData(gd);
		final INewLocationListener locationlistener = this;
		newLocationLink.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				new AddLocationAction(getShell(), locationlistener).run();
			}
		});
		newLocationLink.setLayoutData(gd);

		getShell().setText("Location selection");
		return control;
	}

	private synchronized void loadLocations() 
	{
		allLocations = LocationHelper.getLocations();
		if(allLocations != null) 
		{
			Location[] locationsArray = allLocations.toArray(new Location[0]);
			setElements(locationsArray);
		} 
		else 
		{
			setElements(null);
		}
	}

	@Override
	public void update(Location newLocation) {
		addLocation(newLocation);
		this.setListElements(allLocations.toArray(new Location[0]));
		this.setFilter(newLocation.getMnemonic());
	}

	private synchronized void addLocation(Location newLocation) 
	{
		LocationHelper.findLocation(newLocation);
		allLocations.add(newLocation);
	}
}
