/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.acs.tmcdb.Location;
import alma.obops.tmcdb.alarms.ui.tree.helpers.LocationHelper;
import alma.obops.tmcdbgui.widgets.StatusPublishingComposite;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;

public class LocationAttributesComposite extends StatusPublishingComposite 
{
	public static final String LOCATION_ALREADY_EXISTS = "Location already exists: name must be unique";
	private Location location;

	private Text mnemonicText;
	private Text buildingText;
	private Text roomText;
	private Text positionText;
	private Text floorText;
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 * @param location the faultcode that is being "dealt" with.
	 */
	public LocationAttributesComposite(Composite parent, int style, Location location, StatusListener statusListener, DirtyListener dirtyListener)
	{
		super(parent, style);
		this.addStatusListener(statusListener);
		this.addDirtyListener(dirtyListener);
		createControl(location);
	}
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public LocationAttributesComposite(Composite parent, int style, DirtyListener dirtyListener) 
	{
		this(parent, style, null, null, dirtyListener);
	}

	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public LocationAttributesComposite(Composite parent, int style, StatusListener statusListener) 
	{
		this(parent, style, null, statusListener, null);
	}

	private void createControl(Location loc) 
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;  // label, entry
		setLayout( layout );
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		setLayoutData(gridData);

		createMnemonicControl();
		createBuildingControl();
		createFloorControl();
		createRoomControl();
		createPositionControl();
        setLocation(loc);
        addKeyListeners();
	}	
	
	private void createPositionControl() 
	{
        Label label = new Label(this, SWT.NONE);
        label.setText("Position");
        positionText = new Text(this, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        positionText.setLayoutData(gridData);
        positionText.addKeyListener(new SetDirtyKeyListener());
	}
	
	private void createMnemonicControl() 
	{
		Label label = new Label(this, SWT.NONE);
		label.setText("Mnemonic");
		mnemonicText = new Text(this, SWT.BORDER);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		mnemonicText.setLayoutData(gridData);
		mnemonicText.addKeyListener(new SetDirtyKeyListener());
	}

	private void createBuildingControl() {
		Label label = new Label(this, SWT.NONE);
		label.setText("Building");
		buildingText = new Text(this, SWT.BORDER);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		buildingText.setLayoutData(gridData);
		buildingText.addKeyListener(new SetDirtyKeyListener());
	}

	private void createFloorControl() 
	{
        Label label = new Label(this, SWT.NONE);
        label.setText("Floor");
        floorText = new Text(this, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        floorText.setLayoutData(gridData);
        floorText.addKeyListener(new SetDirtyKeyListener());
	}
	
	private void createRoomControl() 
	{
        Label label = new Label(this, SWT.NONE);
        label.setText("Room");
        roomText = new Text(this, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        roomText.setLayoutData(gridData);
        roomText.addKeyListener(new SetDirtyKeyListener());
	}

	private void addKeyListeners() 
	{
		// At each keystroke computes whether this page is complete
		KeyListener completionKL = new KeyListener() 
		{
			public void keyPressed( KeyEvent e ) {
				// ignore
			}

			public void keyReleased( KeyEvent e ) {
				isComplete();
			}
		};
		mnemonicText.addKeyListener(completionKL);
		buildingText.addKeyListener(completionKL);
		roomText.addKeyListener(completionKL);
		floorText.addKeyListener(completionKL);
		positionText.addKeyListener(completionKL);
	}
	
	private boolean locationExists()
	{
		boolean retVal = false;

		retVal = foundCorrespondingLocation();
		
		if(retVal == true) {
			this.setStatus(LOCATION_ALREADY_EXISTS);
		} else {
			this.setStatus(null);
		}
		return retVal;
	}
	
	private boolean foundCorrespondingLocation() 
	{
		boolean retVal = false;
		for(Location con: LocationHelper.getLocations()) 
		{
			if(con.getMnemonic().equals(getMnemonic())) 
			{
				retVal = true;
				break;
			}
		}
		return retVal;
	}
	
	
	/** @return <code>true</code> when all required fields are populated */
	public boolean isComplete() 
	{	
		boolean complete = 
			!locationExists() &&
			(mnemonicText.getText().length() > 0) &&
			(buildingText.getText().length() > 0) &&
			(roomText.getText().length() > 0) &&
			(positionText.getText().length() > 0) &&
			(floorText.getText().length() > 0); 

		notifyListenersOfCompletion(complete);
		return complete;
	}

	private class SetDirtyKeyListener implements KeyListener 
	{
		@Override
		public void keyPressed(KeyEvent e) {
			setDirty(true);
		}

		@Override
		public void keyReleased(KeyEvent e) {
		}
	}

	public void setLocation(Location cont) 
	{
		if(null == cont || this.location == cont) 
		{
			return;
		}
		this.location = cont;
		
		this.mnemonicText.setText(cont.getMnemonic());
	}

	public String getMnemonic() {
		return this.mnemonicText.getText();
	}
	
	public String getBuilding() {
		return this.buildingText.getText();
	}

	public String getFloor() {
		return this.floorText.getText();
	}
	
	public String getPosition() {
		return this.positionText.getText();
	}
	
	public String getRoom() {
		return this.roomText.getText();
	}
}
