/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import java.util.Collection;
import java.util.HashSet;

import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdb.alarms.ui.actions.add.AddFaultMemberAction;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewFaultMemberListener;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultMemberHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

public class FaultMemberSelectionDialog extends ElementListSelectionDialog implements INewFaultMemberListener
{
	private Collection<FaultMember> allFaultMembers;
	private IWorkbenchWindow window;
	private FaultFamily owningFamily;
	
	public FaultMemberSelectionDialog(IWorkbenchWindow window, ILabelProvider renderer, FaultFamily family)
	{
		super(window.getShell(), renderer);
		this.owningFamily = family;
		this.window = window;
		setIgnoreCase(true);
		setMessage("Select a fault member");
		setMultipleSelection(false);
		FaultMember[] fms = new FaultMember[0];
		allFaultMembers = new HashSet<FaultMember>();
		
		try {
			fms = AlarmConversationUtils.getInstance().getFaultMembersForFaultFamily(family, ConversationToken.CONVERSATION_PENDING);
		} catch (Exception e) {
			GuiUtils.showErrorDialog(getShell(), "Error", "Could not query fault members for fault family.");
			e.printStackTrace();
		}
		
		for(FaultMember member: fms) 
		{
			allFaultMembers.add(FaultMemberHelper.findFaultMember(member));
		}
		setElements(fms);
	}
	
	protected Control createDialogArea(Composite parent) 
	{
		final Composite comp = new Composite(parent, SWT.NONE);
		comp.setLayout(new GridLayout(2, false));
		comp.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		GridData gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		gd.horizontalSpan = 2;
		Control control = super.createDialogArea(comp);
		control.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 10;
		Label iconLabel = new Label(comp, SWT.NONE);
		iconLabel.setImage(RcpUtils.getImage("icons/faultmember-new.png"));
		iconLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Link newFaultMemberLink = new Link(comp, SWT.NONE);
		newFaultMemberLink.setText("<a href=\"create\">Create a new fault member</a>");
		newFaultMemberLink.setLayoutData(gd);
		final INewFaultMemberListener fmlistener = this;
		newFaultMemberLink.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				new AddFaultMemberAction(window, fmlistener, owningFamily ).run();
			}
		});
		newFaultMemberLink.setLayoutData(gd);

		getShell().setText("Fault member selection");
		return control;
	}

	@Override
	public void update(FaultMember newFaultMember) {
		allFaultMembers.add(newFaultMember);
		this.setListElements(allFaultMembers.toArray(new FaultMember[0]));
		this.setFilter(newFaultMember.getMemberName());
	}
}
