/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import java.util.Set;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.obops.tmcdbgui.widgets.StatusPublishingComposite;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;

public class FaultMemberAttributesComposite extends StatusPublishingComposite 
{
	public static final String FAULTMEMBER_ALREADY_EXISTS = "Fault member already exists: name must be unique";
	private FaultMember faultMember;
	private Text nameText;
	private Set<FaultMember> existingFaultMembers;
    private FaultFamily owningFamily;
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 * @param fm the fault member that is being "dealt" with.
	 */
	public FaultMemberAttributesComposite(Composite parent, int style, FaultMember fm, StatusListener statusListener, DirtyListener dirtyListener)
	{
		super(parent, style);
		this.addStatusListener(statusListener);
		this.addDirtyListener(dirtyListener);
		if(null != fm) {
			this.owningFamily = fm.getFaultFamily();
		}
		createControl(fm);
	}
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public FaultMemberAttributesComposite(Composite parent, int style, DirtyListener dirtyListener) 
	{
		this(parent, style, null, null, dirtyListener);
	}

	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public FaultMemberAttributesComposite(Composite parent, int style, StatusListener statusListener, FaultFamily owningFamily) 
	{
		this(parent, style, null, statusListener, null);
		this.owningFamily = owningFamily;
	}

	private void createControl(FaultMember fm) 
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;  // label, entry
		setLayout( layout );
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		setLayoutData(gridData);

		createNameControl();
        setFaultMember(fm);
        addKeyListeners();
	}	
	
	private void createNameControl() 
	{
        Label nameLabel = new Label(this, SWT.NONE);
        nameLabel.setText("Name");
        nameText = new Text(this, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        nameText.setLayoutData(gridData);
        nameText.addKeyListener(new SetDirtyKeyListener());
	}

	private void addKeyListeners() 
	{
		// At each keystroke computes whether this page is complete
		KeyListener completionKL = new KeyListener() 
		{
			public void keyPressed( KeyEvent e ) {
				// ignore
			}

			public void keyReleased( KeyEvent e ) {
				isComplete();
			}
		};
		nameText.addKeyListener(completionKL);
	}
	
	/**
	 * Getter for the new fault member's name.
	 * @return the new fault member's name.
	 */
	public String getMemberName() 
	{
		String retVal = null;
		retVal = nameText.getText();
		return retVal;
	}
	
	private boolean faultMemberExistsInConfig()
	{
		boolean retVal = false;

		if(null == existingFaultMembers) 
		{
			this.existingFaultMembers = owningFamily.getFaultMembers();
		} 	

		try	{
			retVal = foundCorrespondingFaultMember();
		}
		catch(Exception ex) {
			throw new RuntimeException("Unable to get the fault members for the fault family", ex);
		}
		
		if(retVal == true) {
			this.setStatus(FAULTMEMBER_ALREADY_EXISTS);
		} else {
			this.setStatus(null);
		}
		return retVal;
	}
	
	private boolean foundCorrespondingFaultMember() {
		boolean retVal = false;
		for(FaultMember fm: existingFaultMembers) 
		{
			if(fm.getMemberName().equals(getMemberName())) 
			{
				retVal = true;
				break;
			}
		}
		return retVal;
	}
	
	
	/** @return <code>true</code> when all required fields are populated */
	public boolean isComplete() 
	{	
		boolean complete = 
			!faultMemberExistsInConfig() &&
			(nameText.getText().length() > 0);

		notifyListenersOfCompletion(complete);
		return complete;
	}

	private class SetDirtyKeyListener implements KeyListener 
	{
		@Override
		public void keyPressed(KeyEvent e) {
			setDirty(true);
		}

		@Override
		public void keyReleased(KeyEvent e) {
		}
	}

	public void setFaultMember(FaultMember fm) 
	{
		if(null == fm || this.faultMember == fm) 
		{
			return;
		}
		this.faultMember = fm;
		
		// name
		this.nameText.setText(fm.getMemberName());
	}
}
