/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import java.util.Collection;
import java.util.HashSet;

import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.FaultFamily;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewFaultFamilyListener;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultFamilyHelper;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

public class FaultFamilySelectionDialog extends ElementListSelectionDialog implements INewFaultFamilyListener
{
	private Collection<FaultFamily> allFaultFamilies;
	
	public FaultFamilySelectionDialog(IWorkbenchWindow window, ILabelProvider renderer, Configuration config, String title)
	{
		super(window.getShell(), renderer);
		this.setTitle(title);
		this.setMessage(title);
		setIgnoreCase(true);
		setMultipleSelection(false);
		FaultFamily[] ffs = populateFaultFamilies(config);
		setElements(ffs);
	}

	private FaultFamily[] populateFaultFamilies(Configuration config) 
	{
		allFaultFamilies = new HashSet<FaultFamily>();
		FaultFamily[] ffs = new FaultFamily[0];
		try {
			ffs = AlarmConversationUtils.getInstance().getFaultFamilies(config, ConversationToken.CONVERSATION_PENDING);
		} catch (Exception e) {
			GuiUtils.showErrorDialog(getShell(), "Could not hydrate fault families", "Problem hydrating fault families for configuration");
			e.printStackTrace();
		}
		for(FaultFamily family : ffs) {
			allFaultFamilies.add(FaultFamilyHelper.findFaultFamily(family));
		}
		return ffs;
	}
	
	protected Control createDialogArea(Composite parent) 
	{
		final Composite comp = new Composite(parent, SWT.NONE);
		comp.setLayout(new GridLayout(2, false));
		comp.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		GridData gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		gd.horizontalSpan = 2;
		Control control = super.createDialogArea(comp);
		control.setLayoutData(gd);

		getShell().setText("Fault family selection");
		return control;
	}

	@Override
	public void update(FaultFamily newFaultFamily) {
		allFaultFamilies.add(newFaultFamily);
		this.setListElements(allFaultFamilies.toArray(new FaultFamily[0]));
		this.setFilter(newFaultFamily.getFamilyName());
	}
}
