/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.FaultFamily;
import alma.obops.tmcdbgui.widgets.StatusPublishingComposite;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;

/**
 * Used to define/edit the basic attributes of a fault family.
 * @author sharring
 */
public class FaultFamilyAttributesComposite extends StatusPublishingComposite 
{
	public static final String FAULTFAMILY_ALREADY_EXISTS = "Fault family already exists: name must be unique";
	private Text nameText;
	private Text alarmSourceText;
	private Text helpUrlText;
    private Configuration configuration;
	
    /**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 * @param weatherstation the weatherstation that is being "dealt" with.
	 */
	public FaultFamilyAttributesComposite(Composite parent, int style, FaultFamily ff, StatusListener statusListener, DirtyListener dirtyListener, Configuration config)
	{
		super(parent, style);
		this.addStatusListener(statusListener);
		this.addDirtyListener(dirtyListener);
		this.configuration = config;
		createControl(ff);
	}
	
	private void createControl(FaultFamily ff) 
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;  // label, entry
		setLayout( layout );
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		setLayoutData(gridData);

		createNameControl();
		createAlarmSourceControl();
		createHelpUrlControl();
        setFaultFamily(ff);
        addKeyListeners();
	}	
	
	private void createHelpUrlControl() 
	{
		Label helpUrlLabel = new Label(this, SWT.NONE);
		helpUrlLabel.setText("Help URL");
        helpUrlText = new Text(this, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        helpUrlText.setLayoutData(gridData);
        helpUrlText.addKeyListener(new SetDirtyKeyListener());
	}

	private void createAlarmSourceControl() {
		Label alarmSourceLabel = new Label(this, SWT.NONE);
		alarmSourceLabel.setText("Alarm Source");
        alarmSourceText = new Text(this, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        alarmSourceText.setLayoutData(gridData);
        alarmSourceText.addKeyListener(new SetDirtyKeyListener());
	}

	private void createNameControl() 
	{
        Label nameLabel = new Label(this, SWT.NONE);
        nameLabel.setText("Name");
        nameText = new Text(this, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        nameText.setLayoutData(gridData);
        nameText.addKeyListener(new SetDirtyKeyListener());
	}

	private void addKeyListeners() 
	{
		// At each keystroke computes whether this page is complete
		KeyListener completionKL = new KeyListener() 
		{
			public void keyPressed( KeyEvent e ) {
				// ignore
			}

			public void keyReleased( KeyEvent e ) {
				isComplete();
			}
		};
		nameText.addKeyListener(completionKL);
		helpUrlText.addKeyListener(completionKL);
		alarmSourceText.addKeyListener(completionKL);
	}
	
	/**
	 * Getter for the new antenna's name.
	 * @return the new antenna's name.
	 */
	public String getFamilyName() 
	{
		String retVal = null;
		retVal = nameText.getText();
		return retVal;
	}
	
	private boolean faultFamilyExistsInConfig()
	{
		boolean retVal = false;

		try	{
			retVal = foundCorrespondingFaultFamily();
		}
		catch(Exception ex) {
			throw new RuntimeException("Unable to get the fault families for the configuration", ex);
		}
		
		if(retVal == true) {
			this.setStatus(FAULTFAMILY_ALREADY_EXISTS);
		} else {
			this.setStatus(null);
		}
		return retVal;
	}
	
	private boolean foundCorrespondingFaultFamily() {
		boolean retVal = false;
		for(FaultFamily ff: configuration.getFaultFamilies())
		{
			if(ff.getFamilyName().equals(getFamilyName())) 
			{
				retVal = true;
				break;
			}
		}
		return retVal;
	}
	
	
	/** @return <code>true</code> when all required fields are populated */
	public boolean isComplete() 
	{	
		boolean complete = 
			(configuration == null || !faultFamilyExistsInConfig()) &&
			(nameText.getText().length() > 0) &&
			(helpUrlText.getText().length() > 0) &&
			(alarmSourceText.getText().length() > 0);

		notifyListenersOfCompletion(complete);
		return complete;
	}

	private class SetDirtyKeyListener implements KeyListener 
	{
		@Override
		public void keyPressed(KeyEvent e) {
			setDirty(true);
		}

		@Override
		public void keyReleased(KeyEvent e) {
		}
	}

	public void setFaultFamily(FaultFamily ff) 
	{
		if(null == ff) 
		{
			return;
		}

		this.nameText.setText(ff.getFamilyName());
		String helpUrlStr = ff.getHelpURL() == null ? "" : ff.getHelpURL();
		this.helpUrlText.setText(helpUrlStr);
		String alarmSrcStr = ff.getAlarmSource() == null ? "" : ff.getAlarmSource();
		this.alarmSourceText.setText(alarmSrcStr);
	}

	public String getAlarmSource() {
		return this.alarmSourceText.getText();
	}

	public String getHelpUrl() {
		return this.helpUrlText.getText();
	}
}
