/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import java.util.Collection;
import java.util.HashSet;

import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.tmcdb.alarms.ui.actions.add.AddFaultCodeAction;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewFaultCodeListener;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultCodeHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

public class FaultCodeSelectionDialog extends ElementListSelectionDialog implements INewFaultCodeListener
{
	private Collection<FaultCode> allFaultCodes;
	private FaultFamily faultFamily;
	private IWorkbenchWindow window;

	public FaultCodeSelectionDialog(IWorkbenchWindow window, ILabelProvider renderer, FaultFamily faultFamily)
	{
		super(window.getShell(), renderer);
		this.window = window;
		this.faultFamily = faultFamily;
		FaultCode[] fcs = new FaultCode[0];
		
		try {
			fcs = AlarmConversationUtils.getInstance().getFaultCodesForFaultFamily(faultFamily, ConversationToken.CONVERSATION_PENDING);
		} catch (Exception e1) {
			GuiUtils.showErrorDialog(getShell(), "Error", "Could not load fault codes for fault family");
			e1.printStackTrace();
		}

		allFaultCodes = new HashSet<FaultCode>();
		for(FaultCode fc: fcs)
		{
			allFaultCodes.add(FaultCodeHelper.findFaultCode(fc));
		}
		
		setElements(fcs);
		setIgnoreCase(true);
		setMessage("Select a fault code");
		setMultipleSelection(false);
	}
	
	public void setFaultFamily(FaultFamily family)
	{
		this.faultFamily = family;
	}

	protected Control createDialogArea(Composite parent) 
	{
		final Composite comp = new Composite(parent, SWT.NONE);
		comp.setLayout(new GridLayout(2, false));
		comp.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		GridData gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		gd.horizontalSpan = 2;
		Control control = super.createDialogArea(comp);
		control.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 10;
		Label iconLabel = new Label(comp, SWT.NONE);
		iconLabel.setImage(RcpUtils.getImage("icons/faultcode-new.png"));
		iconLabel.setLayoutData(gd);

		if(null != faultFamily)
		{
			gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
			Link newFaultCodeLink = new Link(comp, SWT.NONE);
			newFaultCodeLink.setText("<a href=\"create\">Create a new fault code</a>");
			newFaultCodeLink.setLayoutData(gd);
			final INewFaultCodeListener fmlistener = this;
			newFaultCodeLink.addListener(SWT.Selection, new Listener() {
				public void handleEvent(Event event) {
					new AddFaultCodeAction(window, fmlistener, faultFamily).run();
				}
			});
			newFaultCodeLink.setLayoutData(gd);
		}
		
		getShell().setText("Fault code selection");
		return control;
	}

	@Override
	public void update(FaultCode newFaultCode) 
	{
		addFaultCode(newFaultCode);
		this.setListElements(allFaultCodes.toArray(new FaultCode[0]));
		this.setFilter(newFaultCode.getCodeValue().toString());
	}

	private synchronized void addFaultCode(FaultCode newFaultCode) 
	{
		allFaultCodes.add(newFaultCode);
	}
}
