/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import java.util.Set;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.obops.tmcdb.alarms.ui.wizards.support.IntegerVerifyListener;
import alma.obops.tmcdbgui.widgets.StatusPublishingComposite;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;

public class FaultCodeAttributesComposite extends StatusPublishingComposite 
{
	public static final String FAULTCODE_ALREADY_EXISTS = "Fault code already exists: value must be unique";

	private Text codeValueText;
	private Combo priorityCombo;
	private Text causeText;
	private Text actionText;
	private Text consequenceText;
	private Text problemDescriptionText;
	private Button instantButton;
	private Set<FaultCode> existingFaultCodes;
    private FaultFamily owningFamily;
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 * @param faultCode the faultcode that is being "dealt" with.
	 */
	public FaultCodeAttributesComposite(Composite parent, int style, FaultCode fc, StatusListener statusListener, DirtyListener dirtyListener)
	{
		super(parent, style);
		if(fc != null) {
			this.owningFamily = fc.getFaultFamily();
		}
		this.addStatusListener(statusListener);
		this.addDirtyListener(dirtyListener);
		createControl(fc);
	}
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public FaultCodeAttributesComposite(Composite parent, int style, DirtyListener dirtyListener) 
	{
		this(parent, style, null, null, dirtyListener);
	}

	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public FaultCodeAttributesComposite(Composite parent, int style, StatusListener statusListener, FaultFamily owningFamily) 
	{
		this(parent, style, null, statusListener, null);
		this.owningFamily = owningFamily;
	}

	private void createControl(FaultCode fc) 
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;  // label, entry
		setLayout( layout );
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		setLayoutData(gridData);

		createCodeValueControl();
		createPriorityControl();
		createCauseControl();
		createActionControl();
		createConsequenceControl();
		createProblemDescriptionControl();
		creatInstantControl();
        setFaultCode(fc);
        addWidgetListeners();
	}	
	
	private void creatInstantControl() 
	{
		Label label = new Label(this, SWT.NONE);
		label.setText("Is instant?");
		this.instantButton = new Button(this, SWT.CHECK);
	}

	private void createProblemDescriptionControl() {
		Label label = new Label(this, SWT.NONE);
		label.setText("Problem description");
		problemDescriptionText = new Text(this, SWT.BORDER);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		problemDescriptionText.setLayoutData(gridData);
	}

	private void createConsequenceControl() {
		Label label = new Label(this, SWT.NONE);
		label.setText("Consequence");
		consequenceText = new Text(this, SWT.BORDER);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		consequenceText.setLayoutData(gridData);
	}

	private void createActionControl() {
		Label label = new Label(this, SWT.NONE);
		label.setText("Action");
		actionText = new Text(this, SWT.BORDER);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		actionText.setLayoutData(gridData);
	}

	private void createCauseControl() 
	{
		Label label = new Label(this, SWT.NONE);
		label.setText("Cause");
		causeText = new Text(this, SWT.BORDER);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		causeText.setLayoutData(gridData);
	}

	private void createPriorityControl() {
		Label label = new Label(this, SWT.NONE);
		label.setText("Priority");
		priorityCombo = new Combo(this, SWT.BORDER);
		
		// TODO: is there an enum for the priority? 
		// the DB has constraints on this field, limiting it to 0,1,2,3!
		String[] items = { "0", "1", "2", "3" };
		priorityCombo.setItems(items);
		priorityCombo.select(0);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.BEGINNING;
		gridData.grabExcessHorizontalSpace = false;
		priorityCombo.setLayoutData(gridData);
	}

	private void createCodeValueControl() 
	{
        Label label = new Label(this, SWT.NONE);
        label.setText("Value");
        codeValueText = new Text(this, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        codeValueText.setLayoutData(gridData);
	}

	private void addWidgetListeners() 
	{
		// At each keystroke computes whether this page is complete
		KeyListener completionKL = new KeyListener() 
		{
			public void keyPressed( KeyEvent e ) {
				// ignore
			}

			public void keyReleased( KeyEvent e ) {
				isComplete();
			}
		};
		codeValueText.addKeyListener(completionKL);
		priorityCombo.addKeyListener(completionKL);
		causeText.addKeyListener(completionKL);
		actionText.addKeyListener(completionKL);
		consequenceText.addKeyListener(completionKL);
		problemDescriptionText.addKeyListener(completionKL);
		
		SelectionListener dirtySelectionListener = new SelectionListener() {
			@Override
			public void widgetDefaultSelected(SelectionEvent evt) {
				widgetSelected(evt);
			}

			@Override
			public void widgetSelected(SelectionEvent evt) {
				setDirty(true);
			}
		};
		
		// now add listener for setting dirty flag
		instantButton.addSelectionListener(dirtySelectionListener);
		problemDescriptionText.addKeyListener(new SetDirtyKeyListener());
		consequenceText.addKeyListener(new SetDirtyKeyListener());
		actionText.addKeyListener(new SetDirtyKeyListener());
		causeText.addKeyListener(new SetDirtyKeyListener());
		codeValueText.addKeyListener(new SetDirtyKeyListener());
		priorityCombo.addSelectionListener(dirtySelectionListener);
        
        // and a listener to verify that code field gets only integer input
        codeValueText.addVerifyListener(new IntegerVerifyListener());
	}
	
	private boolean faultCodeExistsInConfig()
	{
		boolean retVal = false;

		if(null == existingFaultCodes) 
		{
			this.existingFaultCodes = owningFamily.getFaultCodes();
		} 	

		try	{
			retVal = foundCorrespondingFaultCode();
		}
		catch(Exception ex) {
			ex.printStackTrace();
			throw new RuntimeException("Unable to get the fault codes for the fault family", ex);
		}
		
		if(retVal == true) {
			this.setStatus(FAULTCODE_ALREADY_EXISTS);
		} else {
			this.setStatus(null);
		}
		return retVal;
	}
	
	private boolean foundCorrespondingFaultCode() {
		boolean retVal = false;
		for(FaultCode fc: existingFaultCodes) 
		{
			if(fc.getCodeValue().equals(getCodeValue())) 
			{
				retVal = true;
				break;
			}
		}
		return retVal;
	}
	
	
	/** @return <code>true</code> when all required fields are populated */
	public boolean isComplete() 
	{	
		boolean complete = 
			!faultCodeExistsInConfig() &&
			(codeValueText.getText().length() > 0) &&
			(priorityCombo.getText().length() > 0) &&
			(causeText.getText().length() > 0) &&
			(actionText.getText().length() > 0) &&
			(consequenceText.getText().length() > 0) &&
			(problemDescriptionText.getText().length() > 0);

		notifyListenersOfCompletion(complete);
		return complete;
	}

	private class SetDirtyKeyListener implements KeyListener 
	{
		@Override
		public void keyPressed(KeyEvent e) {
			setDirty(true);
		}

		@Override
		public void keyReleased(KeyEvent e) {
		}
	}

	public void setFaultCode(FaultCode fc) 
	{
		if(null == fc) 
		{
			return;
		}
		
		String consequence = fc.getConsequence() == null ? "" : fc.getConsequence();
		String action = fc.getAction() == null ? "" : fc.getAction();
		String problemDescription = fc.getProblemDescription() == null ? "" : fc.getProblemDescription();
		String cause = fc.getCause() == null ? "" : fc.getCause();
		
		this.codeValueText.setText(fc.getCodeValue().toString());
		this.actionText.setText(action);
		this.causeText.setText(cause);
		this.consequenceText.setText(consequence);
		this.priorityCombo.select(fc.getPriority());
		this.problemDescriptionText.setText(problemDescription);
		this.instantButton.setSelection(fc.getIsInstant());
	}

	public String getProblemDescription() {
		return this.problemDescriptionText.getText();
	}
	
	public String getAction() {
		return this.actionText.getText();
	}
	
	public String getConsequence() {
		return this.consequenceText.getText();
	}

	public String getCause() {
		return this.causeText.getText();
	}

	public boolean getInstant() {
		return instantButton.getSelection();
	}

	public Integer getPriority() {
		Integer retVal = null;
		if(priorityCombo.getText() != null && !priorityCombo.getText().trim().equals("")) {
			retVal = Integer.parseInt(this.priorityCombo.getText().trim());
		}
		return retVal;
	}
	
	/**
	 * Getter for the new fault code's value.
	 * @return the new fault code's value.
	 */
	public Integer getCodeValue() 
	{
		Integer retVal = null;
		if(codeValueText.getText() != null && !codeValueText.getText().trim().equals("")) {
			retVal = Integer.valueOf(codeValueText.getText().trim());
		}
		return retVal;
	}
}
