/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import java.util.Collection;

import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import alma.acs.tmcdb.Contact;
import alma.obops.tmcdb.alarms.ui.actions.add.AddContactAction;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewContactListener;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ContactHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;

public class ContactSelectionDialog extends ElementListSelectionDialog implements INewContactListener
{
	private Collection<Contact> allContacts;

	public ContactSelectionDialog(Shell shell, ILabelProvider renderer) 
	{
		super(shell, renderer);
		loadContacts();
		setIgnoreCase(true);
		setMessage("Select a contact");
		setMultipleSelection(false);
	}

	protected Control createDialogArea(Composite parent) 
	{
		final Composite comp = new Composite(parent, SWT.NONE);
		comp.setLayout(new GridLayout(2, false));
		comp.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		GridData gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		gd.horizontalSpan = 2;
		Control control = super.createDialogArea(comp);
		control.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		gd.horizontalIndent = 10;
		Label iconLabel = new Label(comp, SWT.NONE);
		iconLabel.setImage(RcpUtils.getImage("icons/contact-new.png"));
		iconLabel.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Link newContactLink = new Link(comp, SWT.NONE);
		newContactLink.setText("<a href=\"create\">Create a new contact</a>");
		newContactLink.setLayoutData(gd);
		final INewContactListener contactlistener = this;
		newContactLink.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				new AddContactAction(getShell(), contactlistener).run();
			}
		});
		newContactLink.setLayoutData(gd);

		getShell().setText("Contact selection");
		return control;
	}

	private synchronized void loadContacts() 
	{
		allContacts = ContactHelper.getContacts();
		if(allContacts != null) 
		{
			Contact[] contactsArray = allContacts.toArray(new Contact[0]);
			setElements(contactsArray);
		} 
		else 
		{
			setElements(null);
		}
	}

	@Override
	public void update(Contact newContact) {
		addContact(newContact);
		this.setListElements(allContacts.toArray(new Contact[0]));
		this.setFilter(newContact.getContactName());
	}

	private synchronized void addContact(Contact newContact) 
	{
		ContactHelper.findContact(newContact);
		allContacts.add(newContact);
	}
}
