/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.acs.tmcdb.Contact;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ContactHelper;
import alma.obops.tmcdbgui.widgets.StatusPublishingComposite;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;

public class ContactAttributesComposite extends StatusPublishingComposite 
{
	public static final String CONTACT_ALREADY_EXISTS = "Contact already exists: name must be unique";
	private Contact contact;

	private Text contactNameText;
	private Text gsmText;
	private Text emailText;
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 * @param contact the faultcode that is being "dealt" with.
	 */
	public ContactAttributesComposite(Composite parent, int style, Contact contact, StatusListener statusListener, DirtyListener dirtyListener)
	{
		super(parent, style);
		this.addStatusListener(statusListener);
		this.addDirtyListener(dirtyListener);
		createControl(contact);
	}
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public ContactAttributesComposite(Composite parent, int style, DirtyListener dirtyListener) 
	{
		this(parent, style, null, null, dirtyListener);
	}

	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public ContactAttributesComposite(Composite parent, int style, StatusListener statusListener) 
	{
		this(parent, style, null, statusListener, null);
	}

	private void createControl(Contact contact1) 
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;  // label, entry
		setLayout( layout );
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		setLayoutData(gridData);

		createContactNameControl();
		createGsmControl();
		createEmailControl();
        setContact(contact1);
        addKeyListeners();
	}	
	
	private void createContactNameControl() 
	{
		Label label = new Label(this, SWT.NONE);
		label.setText("Name");
		contactNameText = new Text(this, SWT.BORDER);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		contactNameText.setLayoutData(gridData);
		contactNameText.addKeyListener(new SetDirtyKeyListener());
	}

	private void createGsmControl() {
		Label label = new Label(this, SWT.NONE);
		label.setText("Gsm");
		gsmText = new Text(this, SWT.BORDER);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gsmText.setLayoutData(gridData);
		gsmText.addKeyListener(new SetDirtyKeyListener());
	}

	private void createEmailControl() 
	{
        Label label = new Label(this, SWT.NONE);
        label.setText("Email");
        emailText = new Text(this, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        emailText.setLayoutData(gridData);
        emailText.addKeyListener(new SetDirtyKeyListener());
	}

	private void addKeyListeners() 
	{
		// At each keystroke computes whether this page is complete
		KeyListener completionKL = new KeyListener() 
		{
			public void keyPressed( KeyEvent e ) {
				// ignore
			}

			public void keyReleased( KeyEvent e ) {
				isComplete();
			}
		};
		contactNameText.addKeyListener(completionKL);
		gsmText.addKeyListener(completionKL);
		emailText.addKeyListener(completionKL);
	}
	
	private boolean contactExists()
	{
		boolean retVal = false;

		retVal = foundCorrespondingContact();
		
		if(retVal == true) {
			this.setStatus(CONTACT_ALREADY_EXISTS);
		} else {
			this.setStatus(null);
		}
		return retVal;
	}
	
	private boolean foundCorrespondingContact() 
	{
		boolean retVal = false;
		for(Contact con: ContactHelper.getContacts()) 
		{
			if(con.getContactName().equals(getContactName())) 
			{
				retVal = true;
				break;
			}
		}
		return retVal;
	}
	
	
	/** @return <code>true</code> when all required fields are populated */
	public boolean isComplete() 
	{	
		boolean complete = 
			!contactExists() &&
			(contactNameText.getText().length() > 0) &&
			(gsmText.getText().length() > 0) &&
			(emailText.getText().length() > 0); 

		notifyListenersOfCompletion(complete);
		return complete;
	}

	private class SetDirtyKeyListener implements KeyListener 
	{
		@Override
		public void keyPressed(KeyEvent e) {
			setDirty(true);
		}

		@Override
		public void keyReleased(KeyEvent e) {
		}
	}

	public void setContact(Contact cont) 
	{
		if(null == cont || this.contact == cont) 
		{
			return;
		}
		this.contact = cont;
		
		this.contactNameText.setText(cont.getContactName());
	}

	public String getContactName() {
		return this.contactNameText.getText();
	}
	
	public String getGsm() {
		return this.gsmText.getText();
	}

	public String getEmail() {
		return this.emailText.getText();
	}
}
